﻿---
title: Semantic search with the inference API
description: The instructions in this tutorial shows you how to use the inference API workflow with various services to perform semantic search on your data. The following...
url: https://www.elastic.co/docs/solutions/search/semantic-search/semantic-search-inference
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Semantic search with the inference API
The instructions in this tutorial shows you how to use the inference API workflow with various services to perform semantic search on your data.
<important>
  For the easiest way to perform semantic search in the Elastic Stack, refer to the [`semantic_text`](https://www.elastic.co/docs/solutions/search/semantic-search/semantic-search-semantic-text) end-to-end tutorial.
</important>

The following examples use the:
- `embed-english-v3.0` model for [Cohere](https://docs.cohere.com/docs/cohere-embed)
- `all-mpnet-base-v2` model from [HuggingFace](https://huggingface.co/sentence-transformers/all-mpnet-base-v2)
- `text-embedding-ada-002` second generation embedding model for OpenAI
- models available through [Azure AI Studio](https://ai.azure.com/explore/models?selectedTask=embeddings) or [Azure OpenAI](https://learn.microsoft.com/en-us/azure/ai-services/openai/concepts/models)
- `text-embedding-004` model for [Google Vertex AI](https://cloud.google.com/vertex-ai/generative-ai/docs/model-reference/text-embeddings-api)
- `mistral-embed` model for [Mistral](https://docs.mistral.ai/getting-started/models/)
- `amazon.titan-embed-text-v1` model for [Amazon Bedrock](https://docs.aws.amazon.com/bedrock/latest/userguide/model-ids.html)
- `ops-text-embedding-zh-001` model for [AlibabaCloud AI](https://help.aliyun.com/zh/open-search/search-platform/developer-reference/text-embedding-api-details)

You can use any Cohere and OpenAI models, they are all supported by the inference API.
For a list of recommended models available on HuggingFace, refer to the supported model list in the [API documentation](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-inference-put-hugging-face).
Click the name of the service you want to use on any of the widgets below to review the corresponding instructions.

## Requirements

<tab-set>
  <tab-item title="Cohere">
    A [Cohere account](https://cohere.com/) is required to use the inference API with the Cohere service.
  </tab-item>

  <tab-item title="ELSER">
    ELSER is a model trained by Elastic. If you have an Elasticsearch deployment, there is no further requirement for using the inference API with the `elasticsearch` service.
  </tab-item>

  <tab-item title="HuggingFace">
    A [HuggingFace account](https://huggingface.co/) is required to use the inference API with the HuggingFace service.
  </tab-item>

  <tab-item title="OpenAI">
    An [OpenAI account](https://openai.com/) is required to use the inference API with the OpenAI service.
  </tab-item>

  <tab-item title="Azure OpenAI">
    - An [Azure subscription](https://azure.microsoft.com/free/cognitive-services?azure-portal=true)
    - Access granted to Azure OpenAI in the desired Azure subscription. You can apply for access to Azure OpenAI by completing the form at [[https://aka.ms/oai/access](https://aka.ms/oai/access)](https://aka.ms/oai/access).
    - An embedding model deployed in [Azure OpenAI Studio](https://oai.azure.com/).
  </tab-item>

  <tab-item title="Azure AI Studio">
    - An [Azure subscription](https://azure.microsoft.com/free/cognitive-services?azure-portal=true)
    - Access to [Azure AI Studio](https://ai.azure.com/)
    - A deployed [embeddings](https://ai.azure.com/explore/models?selectedTask=embeddings) or [chat completion](https://ai.azure.com/explore/models?selectedTask=chat-completion) model.
  </tab-item>

  <tab-item title="Google Vertex AI">
    - A [Google Cloud account](https://console.cloud.google.com/)
    - A project in Google Cloud
    - The Vertex AI API enabled in your project
    - A valid service account for the Google Vertex AI API
    - The service account must have the Vertex AI User role and the `aiplatform.endpoints.predict` permission.
  </tab-item>

  <tab-item title="Mistral">
    - A Mistral Account on [La Plateforme](https://console.mistral.ai/)
    - An API key generated for your account
  </tab-item>

  <tab-item title="Amazon Bedrock">
    - An AWS Account with [Amazon Bedrock](https://aws.amazon.com/bedrock/) access
    - A pair of access and secret keys used to access Amazon Bedrock
  </tab-item>

  <tab-item title="AlibabaCloud AI Search">
    - An AlibabaCloud Account with [AlibabaCloud](https://console.aliyun.com) access
    - An API key generated for your account from the [API keys section](https://opensearch.console.aliyun.com/cn-shanghai/rag/api-key)
  </tab-item>
</tab-set>


## Create an inference endpoint

Create an inference endpoint by using the [Create inference API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-inference-put):
<tab-set>
  <tab-item title="Cohere">
    ```json

    {
        "service": "cohere",
        "service_settings": {
            "api_key": "<api_key>", <2>
            "model_id": "embed-english-v3.0", <3>
            "embedding_type": "byte"
        }
    }
    ```

    <note>
      When using this model the recommended similarity measure to use in the `dense_vector` field mapping is `dot_product`. In the case of Cohere models, the embeddings are normalized to unit length in which case the `dot_product` and the `cosine` measures are equivalent.
    </note>
  </tab-item>

  <tab-item title="ELSER">
    ```json

    {
      "service": "elasticsearch",
      "service_settings": {
        "num_allocations": 1,
        "num_threads": 1
      }
    }
    ```
    You don’t need to download and deploy the ELSER model upfront, the API request above will download the model if it’s not downloaded yet and then deploy it.
    <note>
      You might see a 502 bad gateway error in the response when using the Kibana Console. This error usually just reflects a timeout, while the model downloads in the background. You can check the download progress in the Machine Learning UI. If using the Python client, you can set the `timeout` parameter to a higher value.
    </note>
  </tab-item>

  <tab-item title="HuggingFace">
    First, you need to create a new inference endpoint on [the Hugging Face endpoint page](https://ui.endpoints.huggingface.co/) to get an endpoint URL. Select the model `all-mpnet-base-v2` on the new endpoint creation page, then select the `Sentence Embeddings` task under the Advanced configuration section. Create the endpoint. Copy the URL after the endpoint initialization has been finished, you need the URL in the following inference API call.
    ```json

    {
      "service": "hugging_face",
      "service_settings": {
        "api_key": "<access_token>", <2>
        "url": "<url_endpoint>" <3>
      }
    }
    ```
  </tab-item>

  <tab-item title="OpenAI">
    ```json

    {
        "service": "openai",
        "service_settings": {
            "api_key": "<api_key>", <2>
            "model_id": "text-embedding-ada-002" <3>
        }
    }
    ```

    <note>
      When using this model the recommended similarity measure to use in the `dense_vector` field mapping is `dot_product`. In the case of OpenAI models, the embeddings are normalized to unit length in which case the `dot_product` and the `cosine` measures are equivalent.
    </note>
  </tab-item>

  <tab-item title="Azure OpenAI">
    ```json

    {
        "service": "azureopenai",
        "service_settings": {
            "api_key": "<api_key>", <2>
            "resource_name": "<resource_name>", <3>
            "deployment_id": "<deployment_id>", <4>
            "api_version": "2024-02-01"
        }
    }
    ```

    <note>
      It may take a few minutes for your model’s deployment to become available after it is created. If you try and create the model as above and receive a `404` error message, wait a few minutes and try again. Also, when using this model the recommended similarity measure to use in the `dense_vector` field mapping is `dot_product`. In the case of Azure OpenAI models, the embeddings are normalized to unit length in which case the `dot_product` and the `cosine` measures are equivalent.
    </note>
  </tab-item>

  <tab-item title="Azure AI Studio">
    ```json

    {
        "service": "azureaistudio",
        "service_settings": {
            "api_key": "<api_key>", <2>
            "target": "<target_uri>", <3>
            "provider": "<provider>", <4>
            "endpoint_type": "<endpoint_type>" <5>
        }
    }
    ```

    <note>
      It may take a few minutes for your model’s deployment to become available after it is created. If you try and create the model as above and receive a `404` error message, wait a few minutes and try again. Also, when using this model the recommended similarity measure to use in the `dense_vector` field mapping is `dot_product`.
    </note>
  </tab-item>

  <tab-item title="Google Vertex AI">
    ```json

    {
        "service": "googlevertexai",
        "service_settings": {
            "service_account_json": "<service_account_json>", <2>
            "model_id": "text-embedding-004", <3>
            "location": "<location>", <4>
            "project_id": "<project_id>" <5>
        }
    }
    ```
  </tab-item>

  <tab-item title="Mistral">
    ```json

    {
        "service": "mistral",
        "service_settings": {
            "api_key": "<api_key>", <2>
            "model": "<model_id>" <3>
        }
    }
    ```
  </tab-item>

  <tab-item title="Amazon Bedrock">
    ```json

    {
        "service": "amazonbedrock",
        "service_settings": {
            "access_key": "<aws_access_key>", <2>
            "secret_key": "<aws_secret_key>", <3>
            "region": "<region>", <4>
            "provider": "<provider>", <5>
            "model": "<model_id>" <6>
        }
    }
    ```
  </tab-item>

  <tab-item title="AlibabaCloud AI Search">
    ```json

    {
        "service": "alibabacloud-ai-search",
        "service_settings": {
            "api_key": "<api_key>", <2>
            "service_id": "<service_id>", <3>
            "host": "<host>", <4>
            "workspace": "<workspace>" <5>
        }
    }
    ```
  </tab-item>
</tab-set>


## Create the index mapping

The mapping of the destination index - the index that contains the embeddings that the model will create based on your input text - must be created. The destination index must have a field with the [`dense_vector`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/dense-vector) field type for most models and the [`sparse_vector`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/sparse-vector) field type for the sparse vector models like in the case of the `elasticsearch` service to index the output of the used model.
<tab-set>
  <tab-item title="Cohere">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 1024, <3>
            "element_type": "byte"
          },
          "content": { <4>
            "type": "text" <5>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="ELSER">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "sparse_vector" <2>
          },
          "content": { <3>
            "type": "text" <4>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="HuggingFace">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 768, <3>
            "element_type": "float"
          },
          "content": { <4>
            "type": "text" <5>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="OpenAI">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 1536, <3>
            "element_type": "float",
            "similarity": "dot_product" <4>
          },
          "content": { <5>
            "type": "text" <6>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="Azure OpenAI">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 1536, <3>
            "element_type": "float",
            "similarity": "dot_product" <4>
          },
          "content": { <5>
            "type": "text" <6>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="Azure AI Studio">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 1536, <3>
            "element_type": "float",
            "similarity": "dot_product" <4>
          },
          "content": { <5>
            "type": "text" <6>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="Google Vertex AI">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 768, <3>
            "element_type": "float",
            "similarity": "dot_product" <4>
          },
          "content": { <5>
            "type": "text" <6>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="Mistral">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 1024, <3>
            "element_type": "float",
            "similarity": "dot_product" <4>
          },
          "content": { <5>
            "type": "text" <6>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="Amazon Bedrock">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 1024, <3>
            "element_type": "float",
            "similarity": "dot_product" <4>
          },
          "content": { <5>
            "type": "text" <6>
          }
        }
      }
    }
    ```
  </tab-item>

  <tab-item title="AlibabaCloud AI Search">
    ```json

    {
      "mappings": {
        "properties": {
          "content_embedding": { <1>
            "type": "dense_vector", <2>
            "dims": 1024, <3>
            "element_type": "float"
          },
          "content": { <4>
            "type": "text" <5>
          }
        }
      }
    }
    ```
  </tab-item>
</tab-set>


## Create an ingest pipeline with an inference processor

Create an [ingest pipeline](https://www.elastic.co/docs/manage-data/ingest/transform-enrich/ingest-pipelines) with an [inference processor](https://www.elastic.co/docs/reference/enrich-processor/inference-processor) and use the model you created above to infer against the data that is being ingested in the pipeline.
<tab-set>
  <tab-item title="Cohere">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "cohere_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="ELSER">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "elser_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="HuggingFace">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "hugging_face_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="OpenAI">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "openai_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="Azure OpenAI">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "azure_openai_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="Azure AI Studio">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "azure_ai_studio_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="Google Vertex AI">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "google_vertex_ai_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="Mistral">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "mistral_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="Amazon Bedrock">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "amazon_bedrock_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>

  <tab-item title="AlibabaCloud AI Search">
    ```json

    {
      "processors": [
        {
          "inference": {
            "model_id": "alibabacloud_ai_search_embeddings", <1>
            "input_output": { <2>
              "input_field": "content",
              "output_field": "content_embedding"
            }
          }
        }
      ]
    }
    ```
  </tab-item>
</tab-set>


## Load data

In this step, you load the data that you later use in the inference ingest pipeline to create embeddings from it.
Use the `msmarco-passagetest2019-top1000` data set, which is a subset of the MS MARCO Passage Ranking data set. It consists of 200 queries, each accompanied by a list of relevant text passages. All unique passages, along with their IDs, have been extracted from that data set and compiled into a [tsv file](https://github.com/elastic/stack-docs/blob/main/docs/en/stack/ml/nlp/data/msmarco-passagetest2019-unique.tsv).
Download the file and upload it to your cluster using the [Data Visualizer](https://www.elastic.co/docs/manage-data/ingest/upload-data-files) in the Machine Learning UI. After your data is analyzed, click **Override settings**. Under **Edit field names**, assign `id` to the first column and `content` to the second. Click **Apply**, then **Import**. Name the index `test-data`, and click **Import**. After the upload is complete, you will see an index named `test-data` with 182,469 documents.

## Ingest the data through the inference ingest pipeline

Create embeddings from the text by reindexing the data through the inference pipeline that uses your chosen model. This step uses the [reindex API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-reindex) to simulate data ingestion through a pipeline.
<tab-set>
  <tab-item title="Cohere">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "cohere-embeddings",
        "pipeline": "cohere_embeddings_pipeline"
      }
    }
    ```

    <note>
      The [rate limit of your Cohere account](https://dashboard.cohere.com/billing) may affect the throughput of the reindexing process.
    </note>
  </tab-item>

  <tab-item title="ELSER">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "elser-embeddings",
        "pipeline": "elser_embeddings_pipeline"
      }
    }
    ```
  </tab-item>

  <tab-item title="HuggingFace">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "hugging-face-embeddings",
        "pipeline": "hugging_face_embeddings_pipeline"
      }
    }
    ```
  </tab-item>

  <tab-item title="OpenAI">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "openai-embeddings",
        "pipeline": "openai_embeddings_pipeline"
      }
    }
    ```

    <note>
      The [rate limit of your OpenAI account](https://platform.openai.com/account/limits) may affect the throughput of the reindexing process. If this happens, change `size` to `3` or a similar value in magnitude.
    </note>
  </tab-item>

  <tab-item title="Azure OpenAI">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "azure-openai-embeddings",
        "pipeline": "azure_openai_embeddings_pipeline"
      }
    }
    ```

    <note>
      The [rate limit of your Azure OpenAI account](https://learn.microsoft.com/en-us/azure/ai-services/openai/quotas-limits#quotas-and-limits-reference) may affect the throughput of the reindexing process. If this happens, change `size` to `3` or a similar value in magnitude.
    </note>
  </tab-item>

  <tab-item title="Azure AI Studio">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "azure-ai-studio-embeddings",
        "pipeline": "azure_ai_studio_embeddings_pipeline"
      }
    }
    ```

    <note>
      Your Azure AI Studio model deployment may have rate limits in place that might affect the throughput of the reindexing process. If this happens, change `size` to `3` or a similar value in magnitude.
    </note>
  </tab-item>

  <tab-item title="Google Vertex AI">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "google-vertex-ai-embeddings",
        "pipeline": "google_vertex_ai_embeddings_pipeline"
      }
    }
    ```
  </tab-item>

  <tab-item title="Mistral">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "mistral-embeddings",
        "pipeline": "mistral_embeddings_pipeline"
      }
    }
    ```
  </tab-item>

  <tab-item title="Amazon Bedrock">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "amazon-bedrock-embeddings",
        "pipeline": "amazon_bedrock_embeddings_pipeline"
      }
    }
    ```
  </tab-item>

  <tab-item title="AlibabaCloud AI Search">
    ```json

    {
      "source": {
        "index": "test-data",
        "size": 50 <1>
      },
      "dest": {
        "index": "alibabacloud-ai-search-embeddings",
        "pipeline": "alibabacloud_ai_search_embeddings_pipeline"
      }
    }
    ```
  </tab-item>
</tab-set>

The call returns a task ID to monitor the progress:
```json
```

Reindexing large datasets can take a long time. You can test this workflow using only a subset of the dataset. Do this by cancelling the reindexing process, and only generating embeddings for the subset that was reindexed. The following API request will cancel the reindexing task:
```json
```


## Semantic search

After the data set has been enriched with the embeddings, you can query the data using [semantic search](/docs/solutions/search/vector/knn#knn-semantic-search). In case of dense vector models, pass a `query_vector_builder` to the k-nearest neighbor (kNN) vector search API, and provide the query text and the model you have used to create the embeddings. In case of a sparse vector model like ELSER, use a `sparse_vector` query, and provide the query text with the model you have used to create the embeddings.
<note>
  If you cancelled the reindexing process, you run the query only a part of the data which affects the quality of your results.
</note>

<tab-set>
  <tab-item title="Cohere">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "cohere_embeddings",
            "model_text": "Muscles in human body"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `cohere-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "cohere-embeddings",
            "_id": "-eFWCY4BECzWLnMZuI78",
            "_score": 0.737484,
            "_source": {
              "id": 1690948,
              "content": "Oxygen is supplied to the muscles via red blood cells. Red blood cells carry hemoglobin which oxygen bonds with as the hemoglobin rich blood cells pass through the blood vessels of the lungs.The now oxygen rich blood cells carry that oxygen to the cells that are demanding it, in this case skeletal muscle cells.ther ways in which muscles are supplied with oxygen include: 1  Blood flow from the heart is increased. 2  Blood flow to your muscles in increased. 3  Blood flow from nonessential organs is transported to working muscles."
            }
          },
          {
            "_index": "cohere-embeddings",
            "_id": "HuFWCY4BECzWLnMZuI_8",
            "_score": 0.7176013,
            "_source": {
              "id": 1692482,
              "content": "The thoracic cavity is separated from the abdominal cavity by the  diaphragm. This is a broad flat muscle.    (muscular) diaphragm The diaphragm is a muscle that separat…e the thoracic from the abdominal cavity. The pelvis is the lowest part of the abdominal cavity and it has no physical separation from it    Diaphragm."
            }
          },
          {
            "_index": "cohere-embeddings",
            "_id": "IOFWCY4BECzWLnMZuI_8",
            "_score": 0.7154432,
            "_source": {
              "id": 1692489,
              "content": "Muscular Wall Separating the Abdominal and Thoracic Cavities; Thoracic Cavity of a Fetal Pig; In Mammals the Diaphragm Separates the Abdominal Cavity from the"
            }
          },
          {
            "_index": "cohere-embeddings",
            "_id": "C-FWCY4BECzWLnMZuI_8",
            "_score": 0.695313,
            "_source": {
              "id": 1691493,
              "content": "Burning, aching, tenderness and stiffness are just some descriptors of the discomfort you may feel in the muscles you exercised one to two days ago.For the most part, these sensations you experience after exercise are collectively known as delayed onset muscle soreness.urning, aching, tenderness and stiffness are just some descriptors of the discomfort you may feel in the muscles you exercised one to two days ago."
            }
          },
          (...)
        ]
    ```
  </tab-item>

  <tab-item title="ELSER">
    ```json

    {
      "query":{
        "sparse_vector":{
          "field": "content_embedding",
          "inference_id": "elser_embeddings",
          "query": "How to avoid muscle soreness after running?"
        }
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `cohere-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "elser-embeddings",
            "_id": "ZLGc_pABZbBmsu5_eCoH",
            "_score": 21.472063,
            "_source": {
              "id": 2258240,
              "content": "You may notice some muscle aches while you are exercising. This is called acute soreness. More often, you may begin to feel sore about 12 hours after exercising, and the discomfort usually peaks at 48 to 72 hours after exercise. This is called delayed-onset muscle soreness.It is thought that, during this time, your body is repairing the muscle, making it stronger and bigger.You may also notice the muscles feel better if you exercise lightly. This is normal.his is called delayed-onset muscle soreness. It is thought that, during this time, your body is repairing the muscle, making it stronger and bigger. You may also notice the muscles feel better if you exercise lightly. This is normal."
            }
          },
          {
            "_index": "elser-embeddings",
            "_id": "ZbGc_pABZbBmsu5_eCoH",
            "_score": 21.421381,
            "_source": {
              "id": 2258242,
              "content": "Photo Credit Jupiterimages/Stockbyte/Getty Images. That stiff, achy feeling you get in the days after exercise is a normal physiological response known as delayed onset muscle soreness. You can take it as a positive sign that your muscles have felt the workout, but the pain may also turn you off to further exercise.ou are more likely to develop delayed onset muscle soreness if you are new to working out, if you’ve gone a long time without exercising and start up again, if you have picked up a new type of physical activity or if you have recently boosted the intensity, length or frequency of your exercise sessions."
            }
          },
          {
            "_index": "elser-embeddings",
            "_id": "ZrGc_pABZbBmsu5_eCoH",
            "_score": 20.542095,
            "_source": {
              "id": 2258248,
              "content": "They found that stretching before and after exercise has no effect on muscle soreness. Exercise might cause inflammation, which leads to an increase in the production of immune cells (comprised mostly of macrophages and neutrophils). Levels of these immune cells reach a peak 24-48 hours after exercise.These cells, in turn, produce bradykinins and prostaglandins, which make the pain receptors in your body more sensitive. Whenever you move, these pain receptors are stimulated.hey found that stretching before and after exercise has no effect on muscle soreness. Exercise might cause inflammation, which leads to an increase in the production of immune cells (comprised mostly of macrophages and neutrophils). Levels of these immune cells reach a peak 24-48 hours after exercise."
            }
          },
        (...)
      ]
    ```
  </tab-item>

  <tab-item title="HuggingFace">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "hugging_face_embeddings",
            "model_text": "What's margin of error?"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `hugging-face-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "hugging-face-embeddings",
            "_id": "ljEfo44BiUQvMpPgT20E",
            "_score": 0.8522128,
            "_source": {
              "id": 7960255,
              "content": "The margin of error can be defined by either of the following equations. Margin of error = Critical value x Standard deviation of the statistic. Margin of error = Critical value x Standard error of the statistic. If you know the standard deviation of the statistic, use the first equation to compute the margin of error. Otherwise, use the second equation. Previously, we described how to compute the standard deviation and standard error."
            }
          },
          {
            "_index": "hugging-face-embeddings",
            "_id": "lzEfo44BiUQvMpPgT20E",
            "_score": 0.7865497,
            "_source": {
              "id": 7960259,
              "content": "1 y ou are told only the size of the sample and are asked to provide the margin of error for percentages which are not (yet) known. 2  This is typically the case when you are computing the margin of error for a survey which is going to be conducted in the future."
            }
          },
          {
            "_index": "hugging-face-embeddings1",
            "_id": "DjEfo44BiUQvMpPgT20E",
            "_score": 0.6229427,
            "_source": {
              "id": 2166183,
              "content": "1. In general, the point at which gains equal losses. 2. In options, the market price that a stock must reach for option buyers to avoid a loss if they exercise. For a call, it is the strike price plus the premium paid. For a put, it is the strike price minus the premium paid."
            }
          },
          {
            "_index": "hugging-face-embeddings1",
            "_id": "VzEfo44BiUQvMpPgT20E",
            "_score": 0.6034223,
            "_source": {
              "id": 2173417,
              "content": "How do you find the area of a circle? Can you measure the area of a circle and use that to find a value for Pi?"
            }
          },
          (...)
        ]
    ```
  </tab-item>

  <tab-item title="OpenAI">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "openai_embeddings",
            "model_text": "Calculate fuel cost"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `openai-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "openai-embeddings",
            "_id": "DDd5OowBHxQKHyc3TDSC",
            "_score": 0.83704096,
            "_source": {
              "id": 862114,
              "body": "How to calculate fuel cost for a road trip. By Tara Baukus Mello • Bankrate.com. Dear Driving for Dollars, My family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost.It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes.y family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost. It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes."
            }
          },
          {
            "_index": "openai-embeddings",
            "_id": "ajd5OowBHxQKHyc3TDSC",
            "_score": 0.8345704,
            "_source": {
              "id": 820622,
              "body": "Home Heating Calculator. Typically, approximately 50% of the energy consumed in a home annually is for space heating. When deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important.This calculator can help you estimate the cost of fuel for different heating appliances.hen deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important. This calculator can help you estimate the cost of fuel for different heating appliances."
            }
          },
          {
            "_index": "openai-embeddings",
            "_id": "Djd5OowBHxQKHyc3TDSC",
            "_score": 0.8327426,
            "_source": {
              "id": 8202683,
              "body": "Fuel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel.If you are paying $4 per gallon, the trip would cost you $200.Most boats have much larger gas tanks than cars.uel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel."
            }
          },
          (...)
        ]
    ```
  </tab-item>

  <tab-item title="Azure OpenAI">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "azure_openai_embeddings",
            "model_text": "Calculate fuel cost"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `azure-openai-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "azure-openai-embeddings",
            "_id": "DDd5OowBHxQKHyc3TDSC",
            "_score": 0.83704096,
            "_source": {
              "id": 862114,
              "body": "How to calculate fuel cost for a road trip. By Tara Baukus Mello • Bankrate.com. Dear Driving for Dollars, My family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost.It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes.y family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost. It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes."
            }
          },
          {
            "_index": "azure-openai-embeddings",
            "_id": "ajd5OowBHxQKHyc3TDSC",
            "_score": 0.8345704,
            "_source": {
              "id": 820622,
              "body": "Home Heating Calculator. Typically, approximately 50% of the energy consumed in a home annually is for space heating. When deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important.This calculator can help you estimate the cost of fuel for different heating appliances.hen deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important. This calculator can help you estimate the cost of fuel for different heating appliances."
            }
          },
          {
            "_index": "azure-openai-embeddings",
            "_id": "Djd5OowBHxQKHyc3TDSC",
            "_score": 0.8327426,
            "_source": {
              "id": 8202683,
              "body": "Fuel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel.If you are paying $4 per gallon, the trip would cost you $200.Most boats have much larger gas tanks than cars.uel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel."
            }
          },
          (...)
        ]
    ```
  </tab-item>

  <tab-item title="Azure AI Studio">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "azure_ai_studio_embeddings",
            "model_text": "Calculate fuel cost"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `azure-ai-studio-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "azure-ai-studio-embeddings",
            "_id": "DDd5OowBHxQKHyc3TDSC",
            "_score": 0.83704096,
            "_source": {
              "id": 862114,
              "body": "How to calculate fuel cost for a road trip. By Tara Baukus Mello • Bankrate.com. Dear Driving for Dollars, My family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost.It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes.y family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost. It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes."
            }
          },
          {
            "_index": "azure-ai-studio-embeddings",
            "_id": "ajd5OowBHxQKHyc3TDSC",
            "_score": 0.8345704,
            "_source": {
              "id": 820622,
              "body": "Home Heating Calculator. Typically, approximately 50% of the energy consumed in a home annually is for space heating. When deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important.This calculator can help you estimate the cost of fuel for different heating appliances.hen deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important. This calculator can help you estimate the cost of fuel for different heating appliances."
            }
          },
          {
            "_index": "azure-ai-studio-embeddings",
            "_id": "Djd5OowBHxQKHyc3TDSC",
            "_score": 0.8327426,
            "_source": {
              "id": 8202683,
              "body": "Fuel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel.If you are paying $4 per gallon, the trip would cost you $200.Most boats have much larger gas tanks than cars.uel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel."
            }
          },
          (...)
        ]
    ```
  </tab-item>

  <tab-item title="Google Vertex AI">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "google_vertex_ai_embeddings",
            "model_text": "Calculate fuel cost"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `mistral-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "google-vertex-ai-embeddings",
            "_id": "Ryv0nZEBBFPLbFsdCbGn",
            "_score": 0.86815524,
            "_source": {
              "id": 3041038,
              "content": "For example, the cost of the fuel could be 96.9, the amount could be 10 pounds, and the distance covered could be 80 miles. To convert between Litres per 100KM and Miles Per Gallon, please provide a value and click on the required button.o calculate how much fuel you'll need for a given journey, please provide the distance in miles you will be covering on your journey, and the estimated MPG of your vehicle. To work out what MPG you are really getting, please provide the cost of the fuel, how much you spent on the fuel, and how far it took you."
            }
          },
          {
            "_index": "google-vertex-ai-embeddings",
            "_id": "w4j0nZEBZ1nFq1oiHQvK",
            "_score": 0.8676357,
            "_source": {
              "id": 1541469,
              "content": "This driving cost calculator takes into consideration the fuel economy of the vehicle that you are travelling in as well as the fuel cost. This road trip gas calculator will give you an idea of how much would it cost to drive before you actually travel.his driving cost calculator takes into consideration the fuel economy of the vehicle that you are travelling in as well as the fuel cost. This road trip gas calculator will give you an idea of how much would it cost to drive before you actually travel."
            }
          },
          {
            "_index": "google-vertex-ai-embeddings",
            "_id": "Hoj0nZEBZ1nFq1oiHQjJ",
            "_score": 0.80510974,
            "_source": {
              "id": 7982559,
              "content": "What's that light cost you? 1  Select your electric rate (or click to enter your own). 2  You can calculate results for up to four types of lights. 3  Select the type of lamp (i.e. 4  Select the lamp wattage (lamp lumens). 5  Enter the number of lights in use. 6  Select how long the lamps are in use (or click to enter your own; enter hours on per year). 7  Finally, ..."
            }
          },
          (...)
        ]
    ```
  </tab-item>

  <tab-item title="Mistral">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "mistral_embeddings",
            "model_text": "Calculate fuel cost"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `mistral-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "mistral-embeddings",
            "_id": "DDd5OowBHxQKHyc3TDSC",
            "_score": 0.83704096,
            "_source": {
              "id": 862114,
              "body": "How to calculate fuel cost for a road trip. By Tara Baukus Mello • Bankrate.com. Dear Driving for Dollars, My family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost.It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes.y family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost. It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes."
            }
          },
          {
            "_index": "mistral-embeddings",
            "_id": "ajd5OowBHxQKHyc3TDSC",
            "_score": 0.8345704,
            "_source": {
              "id": 820622,
              "body": "Home Heating Calculator. Typically, approximately 50% of the energy consumed in a home annually is for space heating. When deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important.This calculator can help you estimate the cost of fuel for different heating appliances.hen deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important. This calculator can help you estimate the cost of fuel for different heating appliances."
            }
          },
          {
            "_index": "mistral-embeddings",
            "_id": "Djd5OowBHxQKHyc3TDSC",
            "_score": 0.8327426,
            "_source": {
              "id": 8202683,
              "body": "Fuel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel.If you are paying $4 per gallon, the trip would cost you $200.Most boats have much larger gas tanks than cars.uel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel."
            }
          },
          (...)
        ]
    ```
  </tab-item>

  <tab-item title="Amazon Bedrock">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "amazon_bedrock_embeddings",
            "model_text": "Calculate fuel cost"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `amazon-bedrock-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "amazon-bedrock-embeddings",
            "_id": "DDd5OowBHxQKHyc3TDSC",
            "_score": 0.83704096,
            "_source": {
              "id": 862114,
              "body": "How to calculate fuel cost for a road trip. By Tara Baukus Mello • Bankrate.com. Dear Driving for Dollars, My family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost.It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes.y family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost. It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes."
            }
          },
          {
            "_index": "amazon-bedrock-embeddings",
            "_id": "ajd5OowBHxQKHyc3TDSC",
            "_score": 0.8345704,
            "_source": {
              "id": 820622,
              "body": "Home Heating Calculator. Typically, approximately 50% of the energy consumed in a home annually is for space heating. When deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important.This calculator can help you estimate the cost of fuel for different heating appliances.hen deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important. This calculator can help you estimate the cost of fuel for different heating appliances."
            }
          },
          {
            "_index": "amazon-bedrock-embeddings",
            "_id": "Djd5OowBHxQKHyc3TDSC",
            "_score": 0.8327426,
            "_source": {
              "id": 8202683,
              "body": "Fuel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel.If you are paying $4 per gallon, the trip would cost you $200.Most boats have much larger gas tanks than cars.uel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel."
            }
          },
          (...)
        ]
    ```
  </tab-item>

  <tab-item title="AlibabaCloud AI Search">
    ```json

    {
      "knn": {
        "field": "content_embedding",
        "query_vector_builder": {
          "text_embedding": {
            "model_id": "alibabacloud_ai_search_embeddings",
            "model_text": "Calculate fuel cost"
          }
        },
        "k": 10,
        "num_candidates": 100
      },
      "_source": [
        "id",
        "content"
      ]
    }
    ```
    As a result, you receive the top 10 documents that are closest in meaning to the query from the `alibabacloud-ai-search-embeddings` index sorted by their proximity to the query:
    ```json
    "hits": [
          {
            "_index": "alibabacloud-ai-search-embeddings",
            "_id": "DDd5OowBHxQKHyc3TDSC",
            "_score": 0.83704096,
            "_source": {
              "id": 862114,
              "body": "How to calculate fuel cost for a road trip. By Tara Baukus Mello • Bankrate.com. Dear Driving for Dollars, My family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost.It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes.y family is considering taking a long road trip to finish off the end of the summer, but I'm a little worried about gas prices and our overall fuel cost. It doesn't seem easy to calculate since we'll be traveling through many states and we are considering several routes."
            }
          },
          {
            "_index": "alibabacloud-ai-search-embeddings",
            "_id": "ajd5OowBHxQKHyc3TDSC",
            "_score": 0.8345704,
            "_source": {
              "id": 820622,
              "body": "Home Heating Calculator. Typically, approximately 50% of the energy consumed in a home annually is for space heating. When deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important.This calculator can help you estimate the cost of fuel for different heating appliances.hen deciding on a heating system, many factors will come into play: cost of fuel, installation cost, convenience and life style are all important. This calculator can help you estimate the cost of fuel for different heating appliances."
            }
          },
          {
            "_index": "alibabacloud-ai-search-embeddings",
            "_id": "Djd5OowBHxQKHyc3TDSC",
            "_score": 0.8327426,
            "_source": {
              "id": 8202683,
              "body": "Fuel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel.If you are paying $4 per gallon, the trip would cost you $200.Most boats have much larger gas tanks than cars.uel is another important cost. This cost will depend on your boat, how far you travel, and how fast you travel. A 33-foot sailboat traveling at 7 knots should be able to travel 300 miles on 50 gallons of diesel fuel."
            }
          },
          (...)
        ]
    ```
  </tab-item>
</tab-set>


## Interactive tutorials

You can also find tutorials in an interactive Colab notebook format using the Elasticsearch Python client:
- [Cohere inference tutorial notebook](https://colab.research.google.com/github/elastic/elasticsearch-labs/blob/main/notebooks/integrations/cohere/inference-cohere.ipynb)
- [OpenAI inference tutorial notebook](https://colab.research.google.com/github/elastic/elasticsearch-labs/blob/main/notebooks/search/07-inference.ipynb)