﻿---
title: Hybrid search with semantic_text
description: This tutorial demonstrates how to perform hybrid search, combining semantic search with traditional full-text search. In hybrid search, semantic search...
url: https://www.elastic.co/docs/solutions/search/hybrid-semantic-text
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Hybrid search with semantic_text
This tutorial demonstrates how to perform hybrid search, combining semantic search with traditional full-text search.
In hybrid search, semantic search retrieves results based on the meaning of the text, while full-text search focuses on exact word matches. By combining both methods, hybrid search delivers more relevant results, particularly in cases where relying on a single approach may not be sufficient.
The recommended way to use hybrid search in the Elastic Stack is following the `semantic_text` workflow. This tutorial uses the [`elasticsearch` service](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-inference-put-elasticsearch) for demonstration, but you can use any service and their supported models offered by the Inference API.

## Create an index mapping

The destination index will contain both the embeddings for semantic search and the original text field for full-text search. This structure enables the combination of semantic search and full-text search.
```json

{
  "mappings": {
    "properties": {
      "semantic_text": { <1>
        "type": "semantic_text"
      },
      "content": { <2>
        "type": "text",
        "copy_to": "semantic_text" <3>
      }
    }
  }
}
```

<note>
  If you want to run a search on indices that were populated by web crawlers or connectors, you have to [update the index mappings](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-indices-put-mapping) for these indices to include the `semantic_text` field. Once the mapping is updated, you’ll need to run a full web crawl or a full connector sync. This ensures that all existing documents are reprocessed and updated with the new semantic embeddings, enabling hybrid search on the updated data.
</note>


## Load data

In this step, you load the data that you later use to create embeddings from.
Use the `msmarco-passagetest2019-top1000` data set, which is a subset of the MS MARCO Passage Ranking data set. It consists of 200 queries, each accompanied by a list of relevant text passages. All unique passages, along with their IDs, have been extracted from that data set and compiled into a [tsv file](https://github.com/elastic/stack-docs/blob/main/docs/en/stack/ml/nlp/data/msmarco-passagetest2019-unique.tsv).
Download the file and upload it to your cluster using the [Data Visualizer](https://www.elastic.co/docs/manage-data/ingest/upload-data-files) in the Machine Learning UI. After your data is analyzed, click **Override settings**. Under **Edit field names**, assign `id` to the first column and `content` to the second. Click **Apply**, then **Import**. Name the index `test-data`, and click **Import**. After the upload is complete, you will see an index named `test-data` with 182,469 documents.

## Reindex the data for hybrid search

Reindex the data from the `test-data` index into the `semantic-embeddings` index. The data in the `content` field of the source index is copied into the `content` field of the destination index. The `copy_to` parameter set in the index mapping creation ensures that the content is copied into the `semantic_text` field. The data is processed by the inference endpoint at ingest time to generate embeddings.
<note>
  This step uses the reindex API to simulate data ingestion. If you are working with data that has already been indexed, rather than using the `test-data` set, reindexing is still required to ensure that the data is processed by the inference endpoint and the necessary embeddings are generated.
</note>

```json

{
  "source": {
    "index": "test-data",
    "size": 10 <1>
  },
  "dest": {
    "index": "semantic-embeddings"
  }
}
```

The call returns a task ID to monitor the progress:
```json
```

Reindexing large datasets can take a long time. You can test this workflow using only a subset of the dataset.
To cancel the reindexing process and generate embeddings for the subset that was reindexed:
```json
```


## Perform hybrid search

After reindexing the data into the `semantic-embeddings` index, you can perform hybrid search to combine semantic and lexical search results. Choose between [retrievers](https://www.elastic.co/docs/solutions/search/retrievers-overview) or [ES|QL](https://www.elastic.co/docs/reference/query-languages/esql) syntax to execute the query.
For an overview of all query types supported by `semantic_text` fields and guidance on when to use them, see [Querying `semantic_text` fields](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/semantic-text-search-retrieval).
<tab-set>
  <tab-item title="Query DSL">
    This example uses [retrievers syntax](https://www.elastic.co/docs/solutions/search/retrievers-overview) with [reciprocal rank fusion (RRF)](https://www.elastic.co/docs/reference/elasticsearch/rest-apis/reciprocal-rank-fusion). RRF is a technique that merges the rankings from both semantic and lexical queries, giving more weight to results that rank high in either search. This ensures that the final results are balanced and relevant.
    ```json

    {
      "retriever": {
        "rrf": {
          "retrievers": [
            {
              "standard": { <1>
                "query": {
                  "match": {
                    "content": "How to avoid muscle soreness while running?" <2>
                  }
                }
              }
            },
            {
              "standard": { <3>
                "query": {
                  "semantic": {
                    "field": "semantic_text", <4>
                    "query": "How to avoid muscle soreness while running?"
                  }
                }
              }
            }
          ]
        }
      }
    }
    ```
    After performing the hybrid search, the query will return the combined top 10 documents for both semantic and lexical search criteria. The results include detailed information about each document.
    ```json
    {
      "took": 107,
      "timed_out": false,
      "_shards": {
        "total": 1,
        "successful": 1,
        "skipped": 0,
        "failed": 0
      },
      "hits": {
        "total": {
          "value": 473,
          "relation": "eq"
        },
        "max_score": null,
        "hits": [
          {
            "_index": "semantic-embeddings",
            "_id": "wv65epIBEMBRnhfTsOFM",
            "_score": 0.032786883,
            "_rank": 1,
            "_source": {
              "semantic_text": {
                "inference": {
                  "inference_id": "my-elser-endpoint",
                  "model_settings": {
                    "task_type": "sparse_embedding"
                  },
                  "chunks": [
                    {
                      "text": "What so many out there do not realize is the importance of what you do after you work out. You may have done the majority of the work, but how you treat your body in the minutes and hours after you exercise has a direct effect on muscle soreness, muscle strength and growth, and staying hydrated. Cool Down. After your last exercise, your workout is not over. The first thing you need to do is cool down. Even if running was all that you did, you still should do light cardio for a few minutes. This brings your heart rate down at a slow and steady pace, which helps you avoid feeling sick after a workout.",
                      "embeddings": {
                        "exercise": 1.571044,
                        "after": 1.3603843,
                        "sick": 1.3281639,
                        "cool": 1.3227621,
                        "muscle": 1.2645415,
                        "sore": 1.2561599,
                        "cooling": 1.2335974,
                        "running": 1.1750668,
                        "hours": 1.1104802,
                        "out": 1.0991782,
                        "##io": 1.0794281,
                        "last": 1.0474665,
                       (...)
                      }
                    }
                  ]
                }
              },
              "id": 8408852,
              "content": "What so many out there do not realize is the importance of (...)"
            }
          }
        ]
      }
    }
    ```
  </tab-item>

  <tab-item title="ES|QL">
    The ESQL approach uses a combination of the match operator `:` and the match function `match()` to perform hybrid search.
    ```json

    {
      "query": """
        FROM semantic-embeddings METADATA _score <1>
        | WHERE content: "muscle soreness running?" OR match(semantic_text, "How to avoid muscle soreness while running?", { "boost": 0.75 }) <2> <3>
        | SORT _score DESC <4>
        | LIMIT 1000
      """
    }
    ```
  </tab-item>
</tab-set>