﻿---
title: Work with params and secrets
description: Params allow you to use dynamically defined values in your synthetic monitors. For example, you may want to test a production website with a particular...
url: https://www.elastic.co/docs/solutions/observability/synthetics/work-with-params-secrets
products:
  - Elastic Cloud Serverless
  - Elastic Observability
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Work with params and secrets
Params allow you to use dynamically defined values in your synthetic monitors. For example, you may want to test a production website with a particular demo account whose password is only known to the team managing the synthetic monitors.
For more information about security-sensitive use cases, refer to [Working with secrets and sensitive values](#synthetics-secrets-sensitive).

## Define params

You can declare param values any of the following ways:
- In the *Global parameters* tab of the [Synthetics Settings page in an Observability project](/docs/solutions/observability/synthetics/configure-settings#synthetics-settings-global-parameters).
- Declaring a default value for the parameter in a [configuration file](#synthetics-dynamic-configs).
- Passing the `--params` [CLI argument](#synthetics-cli-params).

<note>
  If you are creating and managing synthetic monitors using a [Synthetics project](https://www.elastic.co/docs/solutions/observability/synthetics/create-monitors-with-projects), you can also use regular environment variables via the standard node `process.env` global object.
</note>

The values in the configuration file are read in the following order:
1. **Global parameters in an Observability project**: The *Global parameters* set using the UI are read first.
2. **Configuration file**: Then the *Global parameters* are merged with any parameters defined in a configuration file. If a parameter is defined in both the Observability UI **and** a Synthetics project configuration file, the value in the configuration file will be used.
3. **CLI**: Then the parameters defined in the configuration are merged with any parameters passed to the CLI `--params` argument. If a parameter is defined in a Synthetics project configuration file **and** using the CLI argument, the value defined using the CLI will be used. When running a script using the CLI, *Global parameters* defined in Kibana or the Observability Serverless project have no impact on the test because it won’t have access to Kibana or the Observability project.


### Global parameters in your Observability project

From any page in the Synthetics UI:
1. Go to **Settings**.
2. Go to the **Global parameters** tab.
3. Define parameters.

![Global parameters tab on the Synthetics Settings page](https://www.elastic.co/docs/solutions/images/observability-synthetics-params-secrets-kibana-define.png)


### Allow users to view global parameter values

<applies-to>
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available since 9.1
</applies-to>

By default, custom user roles do not have access to global parameter values. Administrators can grant read access to global parameters by adding the Synthetics **Can read global parameter values** Kibana sub-feature privilege to a role. When added, users can view a global parameter value using the [**Get parameters** API](https://www.elastic.co/docs/api/doc/kibana/operation/operation-get-parameters) or by selecting the `eye` icon.
To add the **Can read global parameter values** Kibana privilege to a role:
1. Go to the **Roles** management page in the navigation menu or use the [global search field](https://www.elastic.co/docs/explore-analyze/find-and-organize/find-apps-and-objects).
2. Edit an existing role or select **Create role**.
3. From the **Kibana** section, select **Add Kibana privilege**.
4. Expand `Synthetics and Uptime` in the list of [Kibana](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/kibana-privileges) privileges.
5. Turn on **Customize sub-feature privileges**.
6. Turn on **Can read global parameter values**.

Refer to [Defining roles](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/defining-roles) for more on creating custom roles.

### Synthetics project config file

Use a `synthetics.config.js` or `synthetics.config.ts` file to define variables required by your tests. This file should be placed in the root of your Synthetics project.
```js
export default (env) => {
  let my_url = "http://localhost:8080";
  if (env === "production") {
    my_url = "https://elastic.github.io/synthetics-demo/"
  }
  return {
    params: {
      my_url,
    },
  };
};
```

The example above uses the `env` variable, which corresponds to the value of the `NODE_ENV` environment variable.

### CLI argument

To set parameters when running [`npx @elastic/synthetics` on the command line](https://www.elastic.co/docs/solutions/observability/synthetics/cli), use the `--params` or `-p` flag. The provided map is merged over any existing variables defined in the `synthetics.config.{js,ts}` file.
For example, to override the `my_url` parameter, you would run:
```sh
npx @elastic/synthetics . --params '{"my_url": "http://localhost:8080"}'
```


## Use params

You can use params in both lightweight and browser monitors created in either a Synthetics project or the Synthetics UI.

### In a Synthetics project

For lightweight monitors in a Synthetics project, wrap the name of the param in `"${}"` (for example, `"${my_url}"`).
```yaml
- type: http
  name: Todos Lightweight
  id: todos-lightweight
  urls: ["${my_url}"]
  schedule: '@every 1m'
```

In browser monitors, parameters can be referenced via the `params` property available within the argument to a `journey`, `before`, `beforeAll`, `after`, or `afterAll` callback function.
Add `params.` before the name of the param (for example, `params.my_url`):
```js
beforeAll(({params}) => {
  console.log(`Visiting ${params.my_url}`)
})

journey("My Journey", ({ page, params }) => {
  step('launch app', async () => {
    await page.goto(params.my_url)   
  })
})
```


### In the UI

To use a param in a lightweight monitor that is created in the Synthetics UI, wrap the name of the param in `${}` (for example, `${my_url}`).
![Use a param in a lightweight monitor created in the Synthetics UI](https://www.elastic.co/docs/solutions/images/serverless-synthetics-params-secrets-kibana-use-lightweight.png)

To use a param in a browser monitor that is created in the Synthetics UI, add `params.` before the name of the param (for example, `params.my_url`).
![Use a param in a lightweight monitor created in the Synthetics UI](https://www.elastic.co/docs/solutions/images/observability-synthetics-params-secrets-kibana-use-lightweight.png)


## Working with secrets and sensitive values

Your synthetics scripts might require the use of passwords or other sensitive secrets that are not known until runtime.
<warning>
  <applies-to>Elastic Cloud Serverless: Generally available</applies-to><applies-to>Elastic Stack: Generally available since 9.1</applies-to> Params are viewable by administrators and users with the **Can read global parameter values** Kibana privilege. Also, synthetics scripts have no limitations on accessing these values, and a malicious script author could write a synthetics journey that exfiltrates `params` and other data at runtime. Do **not** use truly sensitive passwords (for example, an admin password or a real credit card) in **any** synthetics tools. Instead, set up limited demo accounts, or fake credit cards with limited functionality. If you want to limit access to parameters, ensure that users who are not supposed to access those values do not have the **Can read global parameter values** Kibana privilege, and that any scripts that use those values do not leak them in network requests or screenshots.
</warning>


### Use environment variables in Synthetics Projects

If you are managing monitors with a Synthetics project, you can use environment variables in your `synthetics.config.ts` or `synthetics.config.js` file.
The example below uses `process.env.MY_URL` to reference a variable named `MY_URL` defined in the environment and assigns its value to a param. That param can then be used in both lightweight and browser monitors that are managed in the Synthetics project:
```js
export default {
  params: {
    my_url: process.env.MY_URL
  }
};
```


### Use environment variables with Private Locations

If you are using Kibana-managed monitors and running them on a Private Location, you can inject environment variables directly into the agent's runtime environment.
This method allows you to keep sensitive values out of the Kibana UI. Instead, it stores those values on the infrastructure hosting the Elastic Agent.
<warning>
  These variables are accessible to all monitors running on this specific Private Location.
</warning>


#### 1. Pass variables to the Private Location

When starting your Private Location (Elastic Agent) using Docker, use the `--env` flag to pass your variables.
```bash
docker run \
  --env FLEET_ENROLL=1 \
  --env FLEET_URL={fleet_server_host_url} \
  --env FLEET_ENROLLMENT_TOKEN={enrollment_token} \
  --env MY_URL={secret_value} \
  --cap-add=NET_RAW \
  --cap-add=SETUID \
  -d --restart=unless-stopped \
  docker.elastic.co/elastic-agent/elastic-agent-complete:X.X.X
```


#### 2. Reference variables in the inline script

In the Synthetics **Monitors** editor in Kibana, you can now access these variables using `process.env`.
```js
step('navigate to secure url', async () => {
  // The script reads the environment variable from the private location elastic agent container
  await page.goto(process.env.MY_URL);
});
```