﻿---
title: PowerShell Obfuscation via Negative Index String Reversal
description: Identifies PowerShell scripts that use negative index ranges to reverse the contents of a string or array at runtime as a form of obfuscation. This technique...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/windows/defense_evasion_posh_obfuscation_index_reversal
products:
  - Elastic Security
---

# PowerShell Obfuscation via Negative Index String Reversal
Identifies PowerShell scripts that use negative index ranges to reverse the contents of a string or array at runtime as
a form of obfuscation. This technique avoids direct use of reversal functions by iterating through array elements in
reverse order. These methods are designed to evade static analysis and bypass security protections such as the
Antimalware Scan Interface (AMSI).
**Rule type**: esql
**Rule indices**:
**Rule Severity**: low
**Risk Score**: 21
**Runs every**: 
**Searches indices from**: `now-9m`
**Maximum alerts per execution**: 100
**References**:
**Tags**:
- Domain: Endpoint
- OS: Windows
- Use Case: Threat Detection
- Tactic: Defense Evasion
- Data Source: PowerShell Logs
- Resources: Investigation Guide

**Version**: 7
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Setup

The 'PowerShell Script Block Logging' logging policy must be enabled.
Steps to implement the logging policy with Advanced Audit Configuration:
```
Computer Configuration >
Administrative Templates >
Windows PowerShell >
Turn on PowerShell Script Block Logging (Enable)
```

Steps to implement the logging policy via registry:
```
reg add "hklm\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ScriptBlockLogging" /v EnableScriptBlockLogging /t REG_DWORD /d 1
```


## Investigation guide


## Triage and analysis> **Disclaimer**:
This investigation guide was created using generative AI technology and has been reviewed to improve its accuracy and relevance. While every effort has been made to ensure its quality, we recommend validating the content and adapting it to suit your specific environment and operational needs.

### Investigating PowerShell Obfuscation via Negative Index String Reversal

PowerShell, a powerful scripting language, can be exploited by adversaries using obfuscation techniques like negative index string reversal to evade detection. This method manipulates strings or arrays by iterating in reverse, bypassing static analysis tools. The detection rule identifies scripts with obfuscation patterns by analyzing script length and specific indexing patterns, flagging potential threats for further investigation.

### Possible investigation steps

- Review the `powershell.file.script_block_text` to understand the script's intent and identify any suspicious or malicious behavior.
- Check the `host.name` and `user.id` fields to determine the affected system and user, assessing if they are high-value targets or have a history of similar alerts.
- Analyze the `file.path` to identify the location of the script and assess if it is in a common or suspicious directory.
- Investigate the `powershell.file.script_block_id` and `powershell.sequence` to trace the execution flow and determine if this script is part of a larger, potentially malicious sequence.
- Correlate the `agent.id` with other logs to see if there are additional related activities or alerts from the same endpoint.
- Examine the `count` of detected patterns to assess the level of obfuscation and potential threat severity.


### False positive analysis

- Scripts containing the keyword "GENESIS-5654" are known false positives and are automatically excluded from triggering alerts. Ensure that any legitimate scripts using this keyword are documented to prevent unnecessary investigations.
- Legitimate administrative scripts that use negative indexing for valid purposes may trigger false positives. Review these scripts and consider adding them to an exception list if they are frequently flagged but verified as non-malicious.
- Automated scripts generated by trusted software that use similar obfuscation patterns for performance or compatibility reasons can be excluded by identifying unique identifiers or patterns within these scripts and updating the exclusion criteria accordingly.
- Regularly update the exclusion list to include new patterns or identifiers from trusted sources as they are identified, ensuring that legitimate activities are not hindered by the detection rule.
- Collaborate with IT and security teams to maintain a list of known safe scripts and their characteristics, which can be referenced when analyzing potential false positives.


### Response and remediation

- Isolate the affected host immediately to prevent further spread of potentially malicious scripts or unauthorized access.
- Terminate any suspicious PowerShell processes identified by the alert to halt ongoing obfuscation activities.
- Conduct a thorough review of the PowerShell script block text and related logs to identify any malicious payloads or commands executed.
- Remove any identified malicious scripts or files from the affected system to prevent re-execution.
- Reset credentials for any user accounts involved in the alert to mitigate potential unauthorized access.
- Escalate the incident to the security operations team for further analysis and to determine if additional systems are compromised.
- Update endpoint protection and monitoring tools to enhance detection capabilities for similar obfuscation techniques in the future.


## Rule Query

```esql
from logs-windows.powershell_operational* metadata _id, _version, _index
| where event.code == "4104"

// Filter out smaller scripts that are unlikely to implement obfuscation using the patterns we are looking for
| eval Esql.script_block_length = length(powershell.file.script_block_text)
| where Esql.script_block_length > 500

// replace the patterns we are looking for with the 🔥 emoji to enable counting them
// The emoji is used because it's unlikely to appear in scripts and has a consistent character length of 1
| eval Esql.script_block_tmp = replace(
    powershell.file.script_block_text,
    """\$\w+\[\-\s?1\.\.""",
    "🔥"
)

// count how many patterns were detected by calculating the number of 🔥 characters inserted
| eval Esql.script_block_pattern_count = length(Esql.script_block_tmp) - length(replace(Esql.script_block_tmp, "🔥", ""))

// keep the fields relevant to the query, although this is not needed as the alert is populated using _id
| keep
    Esql.script_block_pattern_count,
    Esql.script_block_length,
    Esql.script_block_tmp,
    powershell.file.*,
    file.path,
    powershell.sequence,
    powershell.total,
    _id,
    _version,
    _index,
    host.name,
    agent.id,
    user.id

// Filter for scripts that match the pattern at least once
| where Esql.script_block_pattern_count >= 1

// FP Patterns
| where not powershell.file.script_block_text like "*GENESIS-5654*"
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Defense Evasion
- Id: TA0005
- Reference URL: [[https://attack.mitre.org/tactics/TA0005/](https://attack.mitre.org/tactics/TA0005/)](https://attack.mitre.org/tactics/TA0005/)
- Technique:
  - Name: Obfuscated Files or Information
- Id: T1027
- Reference URL: [[https://attack.mitre.org/techniques/T1027/](https://attack.mitre.org/techniques/T1027/)](https://attack.mitre.org/techniques/T1027/)
- Technique:
  - Name: Deobfuscate/Decode Files or Information
- Id: T1140
- Reference URL: [[https://attack.mitre.org/techniques/T1140/](https://attack.mitre.org/techniques/T1140/)](https://attack.mitre.org/techniques/T1140/)

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Execution
- Id: TA0002
- Reference URL: [[https://attack.mitre.org/tactics/TA0002/](https://attack.mitre.org/tactics/TA0002/)](https://attack.mitre.org/tactics/TA0002/)
- Technique:
  - Name: Command and Scripting Interpreter
- Id: T1059
- Reference URL: [[https://attack.mitre.org/techniques/T1059/](https://attack.mitre.org/techniques/T1059/)](https://attack.mitre.org/techniques/T1059/)
- Sub Technique:
  - Name: PowerShell
- Id: T1059.001
- Reference URL: [[https://attack.mitre.org/techniques/T1059/001/](https://attack.mitre.org/techniques/T1059/001/)](https://attack.mitre.org/techniques/T1059/001/)