﻿---
title: Potential PowerShell Obfuscation via Backtick-Escaped Variable Expansion
description: Identifies PowerShell scripts that use backtick-escaped characters inside ${} variable expansion as a form of obfuscation. These methods are designed...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/windows/defense_evasion_posh_obfuscation_backtick_var
products:
  - Elastic Security
---

# Potential PowerShell Obfuscation via Backtick-Escaped Variable Expansion
Identifies PowerShell scripts that use backtick-escaped characters inside ${} variable expansion as a form of
obfuscation. These methods are designed to evade static analysis and bypass security protections such as the Antimalware
Scan Interface (AMSI).
**Rule type**: esql
**Rule indices**:
**Rule Severity**: high
**Risk Score**: 73
**Runs every**: 
**Searches indices from**: `now-9m`
**Maximum alerts per execution**: 100
**References**:
**Tags**:
- Domain: Endpoint
- OS: Windows
- Use Case: Threat Detection
- Tactic: Defense Evasion
- Data Source: PowerShell Logs
- Resources: Investigation Guide

**Version**: 7
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Setup

The 'PowerShell Script Block Logging' logging policy must be enabled.
Steps to implement the logging policy with Advanced Audit Configuration:
```
Computer Configuration >
Administrative Templates >
Windows PowerShell >
Turn on PowerShell Script Block Logging (Enable)
```

Steps to implement the logging policy via registry:
```
reg add "hklm\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ScriptBlockLogging" /v EnableScriptBlockLogging /t REG_DWORD /d 1
```


## Investigation guide


## Triage and analysis> **Disclaimer**:
This investigation guide was created using generative AI technology and has been reviewed to improve its accuracy and relevance. While every effort has been made to ensure its quality, we recommend validating the content and adapting it to suit your specific environment and operational needs.

### Investigating Potential PowerShell Obfuscation via Backtick-Escaped Variable Expansion

PowerShell, a powerful scripting language in Windows environments, can be exploited by adversaries using obfuscation techniques like backtick-escaped variable expansion to evade detection. This method involves disguising malicious scripts to bypass security measures. The detection rule identifies scripts with excessive length and specific obfuscation patterns, flagging potential threats for further analysis.

### Possible investigation steps

- Review the `powershell.file.script_block_text` field to understand the content of the script and identify any suspicious or malicious commands.
- Examine the `file.path` and `file.name` fields to determine the origin and context of the script execution, which may provide insights into whether the script is part of a legitimate process or potentially malicious activity.
- Check the `host.name` and `agent.id` fields to identify the affected system and correlate with other security events or logs from the same host for additional context.
- Analyze the `user.id` field to determine which user account executed the script, and assess whether this activity aligns with the user's typical behavior or role.
- Investigate the `powershell.file.script_block_id` and `powershell.sequence` fields to trace the execution flow of the script and identify any related script blocks that may have been executed in sequence.
- Consider the `count` field to evaluate the extent of obfuscation used in the script, which may indicate the level of sophistication or intent behind the script.


### False positive analysis

- Scripts with legitimate administrative functions may use backtick-escaped variable expansion for complex string manipulations. Review the script's context and purpose to determine if it aligns with expected administrative tasks.
- Automated scripts generated by trusted software might include obfuscation patterns as part of their normal operation. Verify the source and integrity of the software to ensure it is from a reputable vendor.
- Developers and IT professionals may use obfuscation techniques during testing or development phases. Establish a process to whitelist known development environments or user accounts to reduce unnecessary alerts.
- PowerShell scripts that are part of legitimate security tools or monitoring solutions may trigger the rule. Identify and exclude these tools by their file path or script block ID to prevent false positives.
- Regularly update the list of known false positives based on historical data and feedback from users to refine the detection rule and improve its accuracy.


### Response and remediation

- Isolate the affected host immediately to prevent further spread of the potentially malicious script across the network.
- Terminate any suspicious PowerShell processes identified by the alert to halt the execution of obfuscated scripts.
- Conduct a thorough review of the script block text and associated file paths to identify and remove any malicious scripts or files from the system.
- Reset credentials for any user accounts involved in the alert to mitigate the risk of compromised credentials being used for further attacks.
- Escalate the incident to the security operations team for a deeper investigation into potential lateral movement or additional compromised systems.
- Implement enhanced monitoring on the affected host and similar systems to detect any recurrence of obfuscation techniques or related suspicious activities.
- Update endpoint protection and intrusion detection systems with indicators of compromise (IOCs) derived from the analysis to improve detection capabilities for similar threats in the future.


## Rule Query

```esql
from logs-windows.powershell_operational* metadata _id, _version, _index
| where event.code == "4104"

// Filter out smaller scripts that are unlikely to implement obfuscation using the patterns we are looking for
| eval Esql.script_block_length = length(powershell.file.script_block_text)
| where Esql.script_block_length > 500

// replace the patterns we are looking for with the 🔥 emoji to enable counting them
// The emoji is used because it's unlikely to appear in scripts and has a consistent character length of 1
| eval Esql.script_block_tmp = replace(powershell.file.script_block_text, """\$\{(\w++`){2,}\w++\}""", "🔥")

// count how many patterns were detected by calculating the number of 🔥 characters inserted
| eval Esql.script_block_pattern_count = length(Esql.script_block_tmp) - length(replace(Esql.script_block_tmp, "🔥", ""))

// keep the fields relevant to the query, although this is not needed as the alert is populated using _id
| keep
    Esql.script_block_pattern_count,
    Esql.script_block_length,
    Esql.script_block_tmp,
    powershell.file.*,
    file.path,
    file.name,
    powershell.sequence,
    powershell.total,
    _id,
    _version,
    _index,
    host.name,
    agent.id,
    user.id

// Filter for scripts that match the pattern at least once
| where Esql.script_block_pattern_count >= 1
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Defense Evasion
- Id: TA0005
- Reference URL: [[https://attack.mitre.org/tactics/TA0005/](https://attack.mitre.org/tactics/TA0005/)](https://attack.mitre.org/tactics/TA0005/)
- Technique:
  - Name: Obfuscated Files or Information
- Id: T1027
- Reference URL: [[https://attack.mitre.org/techniques/T1027/](https://attack.mitre.org/techniques/T1027/)](https://attack.mitre.org/techniques/T1027/)
- Technique:
  - Name: Deobfuscate/Decode Files or Information
- Id: T1140
- Reference URL: [[https://attack.mitre.org/techniques/T1140/](https://attack.mitre.org/techniques/T1140/)](https://attack.mitre.org/techniques/T1140/)

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Execution
- Id: TA0002
- Reference URL: [[https://attack.mitre.org/tactics/TA0002/](https://attack.mitre.org/tactics/TA0002/)](https://attack.mitre.org/tactics/TA0002/)
- Technique:
  - Name: Command and Scripting Interpreter
- Id: T1059
- Reference URL: [[https://attack.mitre.org/techniques/T1059/](https://attack.mitre.org/techniques/T1059/)](https://attack.mitre.org/techniques/T1059/)
- Sub Technique:
  - Name: PowerShell
- Id: T1059.001
- Reference URL: [[https://attack.mitre.org/techniques/T1059/001/](https://attack.mitre.org/techniques/T1059/001/)](https://attack.mitre.org/techniques/T1059/001/)