﻿---
title: Rare Connection to WebDAV Target
description: Identifies rare connection attempts to a Web Distributed Authoring and Versioning (WebDAV) resource. Attackers may inject WebDAV paths in files or features...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/windows/credential_access_rare_webdav_destination
products:
  - Elastic Security
---

# Rare Connection to WebDAV Target
Identifies rare connection attempts to a Web Distributed Authoring and Versioning (WebDAV) resource. Attackers may
inject WebDAV paths in files or features opened by a victim user to leak their NTLM credentials via forced
authentication.
**Rule type**: esql
**Rule indices**:
**Rule Severity**: medium
**Risk Score**: 47
**Runs every**: 
**Searches indices from**: `now-3660s`
**Maximum alerts per execution**: 100
**References**:
- [[https://attack.mitre.org/techniques/T1187/](https://attack.mitre.org/techniques/T1187/)](https://attack.mitre.org/techniques/T1187/)

**Tags**:
- Domain: Endpoint
- OS: Windows
- Use Case: Threat Detection
- Tactic: Credential Access
- Data Source: Elastic Defend
- Data Source: Windows Security Event Logs
- Data Source: Microsoft Defender for Endpoint
- Data Source: Crowdstrike
- Resources: Investigation Guide

**Version**: 5
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Investigation guide


## Triage and analysis


### Investigating Rare Connection to WebDAV Target


### Possible investigation steps

- Examine the reputation of the destination domain or IP address.
- Verify if the target user opened any attachments or clicked links pointing to the same target within seconds from the alert timestamp.
- Correlate the findings with other security logs and alerts to identify any patterns or additional indicators of compromise related to the potential relay attack.


### False positive analysis

- User accessing legit WebDAV resources.


### Response and remediation

- Conduct a password reset for the target account that may have been compromised or are at risk, ensuring the use of strong, unique passwords.
- Verify whether other users were targeted but did not open the lure..
- Escalate the incident to the security operations center (SOC) or incident response team for further investigation and to determine the full scope of the breach.
- Conduct a post-incident review to identify any gaps in security controls and update policies or procedures to prevent recurrence, ensuring lessons learned are applied to improve overall security posture.


## Rule Query

```esql
from logs-endpoint.events.process-*, logs-windows.sysmon_operational-*, logs-system.security-*, logs-windows.*, winlogbeat-*, logs-crowdstrike.fdr*, logs-m365_defender.event-* METADATA _id, _version, _index
| where
    @timestamp > now() - 8 hours and
    event.category == "process" and
    event.type == "start" and
    process.name == "rundll32.exe" and
    process.command_line like "*DavSetCookie*"
| keep host.id, process.command_line, user.name, user.id
// extract domain or IP address from process cmdline
| grok process.command_line """(?<Esql.server_webdav_server>((http|https)://[a-zA-Z0-9-\.]{1,}\.[a-zA-Z]{2,3}[@\/]+)|(\b(?:(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)\.){3}(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)\b)[@\/]+)"""
// remove sub domains from URL
| eval Esql.server_webdav_server = REPLACE(Esql.server_webdav_server, """((http|https)://[a-zA-Z0-9-]{1,}\.)""", "")
| eval Esql.server_webdav_server = REPLACE(Esql.server_webdav_server, "/", "")
| where
    Esql.server_webdav_server is not null and
    not Esql.server_webdav_server in ("www.google.com", "www.elastic.co", "sharepoint.com", "live.net", "google.com", "SHAREPOINT.COM", "github.com") and
    // excludes private IP ranges
    not Esql.server_webdav_server rlike """(10\.(\d{1,3}\.){2}\d{1,3}|172\.(1[6-9]|2\d|3[0-1])\.(\d{1,3}\.)\d{1,3}|192\.168\.(\d{1,3}\.)\d{1,3})"""
| stats
    Esql.event_count = count(*),
    Esql.host_id_count_distinct = count_distinct(host.id),
    Esql.host_id_values = values(host.id),
    Esql.user_name_values = values(user.name)
  by Esql.server_webdav_server
| where
    Esql.host_id_count_distinct == 1 and Esql.event_count <= 3
| eval host.id = MV_MIN(Esql.host_id_values), user.name = MV_MIN(Esql.user_name_values), destination.domain = MV_MIN(Esql.server_webdav_server)
| KEEP host.id, user.name, destination.domain, Esql.*
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Credential Access
- Id: TA0006
- Reference URL: [[https://attack.mitre.org/tactics/TA0006/](https://attack.mitre.org/tactics/TA0006/)](https://attack.mitre.org/tactics/TA0006/)
- Technique:
  - Name: Forced Authentication
- Id: T1187
- Reference URL: [[https://attack.mitre.org/techniques/T1187/](https://attack.mitre.org/techniques/T1187/)](https://attack.mitre.org/techniques/T1187/)