﻿---
title: Multiple Logon Failure from the same Source Address
description: Identifies multiple consecutive logon failures from the same source address and within a short time interval. Adversaries will often brute force login...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/windows/credential_access_bruteforce_multiple_logon_failure_same_srcip
products:
  - Elastic Security
---

# Multiple Logon Failure from the same Source Address
Identifies multiple consecutive logon failures from the same source address and within a short time interval.
Adversaries will often brute force login attempts across multiple users with a common or known password, in an attempt
to gain access to accounts.
**Rule type**: esql
**Rule indices**:
**Rule Severity**: medium
**Risk Score**: 47
**Runs every**: 
**Searches indices from**: `now-9m`
**Maximum alerts per execution**: 100
**References**:
- [[https://docs.microsoft.com/en-us/windows/security/threat-protection/auditing/event-4625](https://docs.microsoft.com/en-us/windows/security/threat-protection/auditing/event-4625)](https://docs.microsoft.com/en-us/windows/security/threat-protection/auditing/event-4625)
- [[https://www.ultimatewindowssecurity.com/securitylog/encyclopedia/event.aspx?eventid=4624](https://www.ultimatewindowssecurity.com/securitylog/encyclopedia/event.aspx?eventid=4624)](https://www.ultimatewindowssecurity.com/securitylog/encyclopedia/event.aspx?eventid=4624)
- [[https://social.technet.microsoft.com/Forums/ie/en-US/c82ac4f3-a235-472c-9fd3-53aa646cfcfd/network-information-missing-in-event-id-4624?forum=winserversecurity](https://social.technet.microsoft.com/Forums/ie/en-US/c82ac4f3-a235-472c-9fd3-53aa646cfcfd/network-information-missing-in-event-id-4624?forum=winserversecurity)](https://social.technet.microsoft.com/Forums/ie/en-US/c82ac4f3-a235-472c-9fd3-53aa646cfcfd/network-information-missing-in-event-id-4624?forum=winserversecurity)
- [[https://serverfault.com/questions/379092/remote-desktop-failed-logon-event-4625-not-logging-ip-address-on-2008-terminal-s/403638#403638](https://serverfault.com/questions/379092/remote-desktop-failed-logon-event-4625-not-logging-ip-address-on-2008-terminal-s/403638#403638)](https://serverfault.com/questions/379092/remote-desktop-failed-logon-event-4625-not-logging-ip-address-on-2008-terminal-s/403638#403638)

**Tags**:
- Domain: Endpoint
- OS: Windows
- Use Case: Threat Detection
- Tactic: Credential Access
- Resources: Investigation Guide
- Data Source: Windows Security Event Logs

**Version**: 116
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Setup

- In some cases the source network address in Windows events 4625/4624 is not populated due to Microsoft logging limitations (examples in the references links). This edge case will break the rule condition and it won't trigger an alert.


## Investigation guide


## Triage and analysis


### Investigating Multiple Logon Failure from the same Source Address

Adversaries with no prior knowledge of legitimate credentials within the system or environment may guess passwords to attempt access to accounts. Without knowledge of the password for an account, an adversary may opt to guess the password using a repetitive or iterative mechanism systematically. More details can be found [here](https://attack.mitre.org/techniques/T1110/001/).
This rule identifies potential password guessing/brute force activity from a single address.> **Note**:
This investigation guide uses the [Osquery Markdown Plugin](https://www.elastic.co/guide/en/security/current/invest-guide-run-osquery.html) introduced in Elastic Stack version 8.5.0. Older Elastic Stack versions will display unrendered Markdown in this guide.

#### Possible investigation steps

- Investigate the logon failure reason code and the targeted user names.
  - Prioritize the investigation if the account is critical or has administrative privileges over the domain.
- Investigate the source IP address of the failed Network Logon attempts.
  - Identify whether these attempts are coming from the internet or are internal.
- Investigate other alerts associated with the involved users and source host during the past 48 hours.
- Identify the source and the target computer and their roles in the IT environment.
- Check whether the involved credentials are used in automation or scheduled tasks.
- If this activity is suspicious, contact the account owner and confirm whether they are aware of it.
- Examine the source host for derived artifacts that indicate compromise:
  - Observe and collect information about the following activities in the alert source host:
  - Attempts to contact external domains and addresses.
  - Examine the DNS cache for suspicious or anomalous entries.
  - $osquery_0
- Examine the host services for suspicious or anomalous entries.
  - $osquery_1
- $osquery_2
- $osquery_3
- Investigate potentially compromised accounts. Analysts can do this by searching for login events (for example, 4624) to the host which is the source of this activity


### False positive analysis

- Understand the context of the authentications by contacting the asset owners. This activity can be related to a new or existing automation or business process that is in a failing state.
- Authentication misconfiguration or obsolete credentials.
- Service account password expired.
- Domain trust relationship issues.
- Infrastructure or availability issues.


### Related rules

- Multiple Logon Failure Followed by Logon Success - 4e85dc8a-3e41-40d8-bc28-91af7ac6cf60


### Response and remediation

- Initiate the incident response process based on the outcome of the triage.
- Isolate the source host to prevent further post-compromise behavior.
- If the asset is exposed to the internet with RDP or other remote services available, take the necessary measures to restrict access to the asset. If not possible, limit the access via the firewall to only the needed IP addresses. Also, ensure the system uses robust authentication mechanisms and is patched regularly.
- Investigate credential exposure on systems compromised or used by the attacker to ensure all compromised accounts are identified. Reset passwords for these accounts and other potentially compromised credentials, such as email, business systems, and web services.
- Run a full antimalware scan. This may reveal additional artifacts left in the system, persistence mechanisms, and malware components.
- Determine the initial vector abused by the attacker and take action to prevent reinfection through the same vector.
- Using the incident response data, update logging and audit policies to improve the mean time to detect (MTTD) and the mean time to respond (MTTR).


## Rule Query

```esql
from logs-system.security*, logs-windows.forwarded*, winlogbeat-* metadata _id, _version, _index
| where event.category == "authentication" and host.os.type == "windows" and event.action == "logon-failed" and
  winlog.logon.type == "Network" and source.ip is not null and winlog.computer_name is not null and
  not cidr_match(TO_IP(source.ip), "127.0.0.0/8", "::1") and
  not user.name in ("ANONYMOUS LOGON", "-") and not user.name like "*$" and user.domain != "NT AUTHORITY" and
  /*
    noisy failure status codes often associated to authentication misconfiguration
     0xC000015B - The user has not been granted the requested logon type (also called the logon right) at this machine.
     0XC000005E - There are currently no logon servers available to service the logon request.
     0XC0000133 - Clocks between DC and other computer too far out of sync.
     0XC0000192 An attempt was made to logon, but the Netlogon service was not started.
     0xc00000dc - DC is in shutdown phase, it will normally tell current clients to use another DC for authentication.
  */
  not winlog.event_data.Status in ("0xc000015b", "0xc000005e", "0xc0000133", "0xc0000192", "0xc00000dc")
// truncate the timestamp to a 60-second window
| eval Esql.time_window = date_trunc(60 seconds, @timestamp)
| stats Esql.failed_auth_count = COUNT(*), Esql.target_user_name_values = VALUES(winlog.event_data.TargetUserName), Esql.user_domain_values = VALUES(user.domain), Esql.error_codes = VALUES(winlog.event_data.Status), Esql.data_stream_namespace.values = VALUES(data_stream.namespace) by winlog.computer_name, source.ip, Esql.time_window, winlog.logon.type
| where Esql.failed_auth_count >= 100
| KEEP winlog.computer_name, source.ip, Esql.time_window, winlog.logon.type, Esql.*
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Credential Access
- Id: TA0006
- Reference URL: [[https://attack.mitre.org/tactics/TA0006/](https://attack.mitre.org/tactics/TA0006/)](https://attack.mitre.org/tactics/TA0006/)
- Technique:
  - Name: Brute Force
- Id: T1110
- Reference URL: [[https://attack.mitre.org/techniques/T1110/](https://attack.mitre.org/techniques/T1110/)](https://attack.mitre.org/techniques/T1110/)
- Sub Technique:
  - Name: Password Guessing
- Id: T1110.001
- Reference URL: [[https://attack.mitre.org/techniques/T1110/001/](https://attack.mitre.org/techniques/T1110/001/)](https://attack.mitre.org/techniques/T1110/001/)
- Sub Technique:
  - Name: Password Spraying
- Id: T1110.003
- Reference URL: [[https://attack.mitre.org/techniques/T1110/003/](https://attack.mitre.org/techniques/T1110/003/)](https://attack.mitre.org/techniques/T1110/003/)