﻿---
title: First-Time FortiGate Administrator Login
description: This rule detects the first observed successful login of a user with the Administrator role to the FortiGate management interface within the last 5 days...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/network/initial_access_newly_observed_fortigate_admin_logon
products:
  - Elastic Security
---

# First-Time FortiGate Administrator Login
This rule detects the first observed successful login of a user with the Administrator role to the FortiGate management
interface within the last 5 days. First-time administrator logins can indicate newly provisioned accounts,
misconfigurations, or unauthorized access using valid credentials and should be reviewed promptly.
**Rule type**: esql
**Rule indices**:
**Rule Severity**: high
**Risk Score**: 73
**Runs every**: 5m
**Searches indices from**: `now-7205m`
**Maximum alerts per execution**: 100
**References**:
- [[https://www.elastic.co/docs/reference/integrations/fortinet_fortigate](https://www.elastic.co/docs/reference/integrations/fortinet_fortigate)](https://www.elastic.co/docs/reference/integrations/fortinet_fortigate)
- [[https://www.cisa.gov/news-events/alerts/2026/01/28/fortinet-releases-guidance-address-ongoing-exploitation-authentication-bypass-vulnerability-cve-2026](https://www.cisa.gov/news-events/alerts/2026/01/28/fortinet-releases-guidance-address-ongoing-exploitation-authentication-bypass-vulnerability-cve-2026)](https://www.cisa.gov/news-events/alerts/2026/01/28/fortinet-releases-guidance-address-ongoing-exploitation-authentication-bypass-vulnerability-cve-2026)

**Tags**:
- Use Case: Threat Detection
- Tactic: Initial Access
- Resources: Investigation Guide
- Domain: Network
- Domain: Identity
- Data Source: Fortinet
- Data Source: Fortinet FortiGate

**Version**: 1
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Investigation guide


## Triage and Analysis


### Investigating First-Time FortiGate Administrator Login

This alert indicates that a user with the **Administrator** role has successfully logged in to the FortiGate management interface for the first time within the last 5 days of observed data.
Because administrator access provides full control over network security devices, any newly observed admin login should be validated to confirm it is expected and authorized.

### Investigation Steps

- **Identify the account**
  - Review `source.user.name` and confirm whether the account is known and officially provisioned.
- Determine whether this is a newly created administrator or an existing account logging in for the first time.
- **Validate the source**
  - Review `source.ip` and confirm whether it originates from a trusted management network, VPN, or jump host.
- Investigate geolocation or ASN if the source IP is external or unusual.
- **Review login context**
  - Examine associated FortiGate log messages for details such as login method, interface, or authentication source.
- Check for additional administrative actions following the login (policy changes, user creation, configuration exports).
- **Correlate with recent changes**
  - Verify whether there were recent change requests, onboarding activities, or maintenance windows that explain the login.
- Look for other authentication attempts (failed or successful) from the same source or user.


### False Positive Considerations

- Newly onboarded administrators or service accounts.
- First-time logins after log retention changes or data source onboarding.
- Automation, backup, or monitoring tools introduced recently.
- Lab, development, or test FortiGate devices.


### Response and Remediation

- **If authorized**
  - Document the activity and consider adding an exception if the behavior is expected.
- Ensure the account follows least-privilege and MFA best practices.
- **If suspicious or unauthorized**
  - Disable or restrict the administrator account immediately.
- Rotate credentials and review authentication sources.
- Audit recent FortiGate configuration changes.
- Review surrounding network activity for lateral movement or persistence attempts.


## Rule Query

```esql
FROM logs-fortinet_fortigate.*, filebeat-* metadata _id

| WHERE event.dataset == "fortinet_fortigate.log" and
        event.category == "authentication" and event.action == "login" and
        event.outcome == "success" and source.user.roles == "Administrator" and source.user.name is not null
| stats Esql.logon_count = count(*),
        Esql.first_time_seen = MIN(@timestamp),
        Esql.source_ip_values = VALUES(source.ip),
        Esql.message_values = VALUES(message) by source.user.name

// first time seen is within 6m of the rule execution time and for the last 5d of events history
| eval Esql.recent = DATE_DIFF("minute", Esql.first_time_seen, now())
| where Esql.recent <= 6 and Esql.logon_count == 1

// move dynamic fields to ECS equivalent for rule exceptions
| eval source.ip = MV_FIRST(Esql.source_ip_values)

| keep source.ip, source.user.name, Esql.*
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Initial Access
- Id: TA0001
- Reference URL: [[https://attack.mitre.org/tactics/TA0001/](https://attack.mitre.org/tactics/TA0001/)](https://attack.mitre.org/tactics/TA0001/)
- Technique:
  - Name: Valid Accounts
- Id: T1078
- Reference URL: [[https://attack.mitre.org/techniques/T1078/](https://attack.mitre.org/techniques/T1078/)](https://attack.mitre.org/techniques/T1078/)