﻿---
title: Entra ID OAuth User Impersonation to Microsoft Graph
description: Identifies potential session hijacking or token replay in Microsoft Entra ID. This rule detects cases where a user signs in and subsequently accesses...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/integrations/azure/initial_access_entra_id_graph_single_session_from_multiple_addresses
products:
  - Elastic Security
---

# Entra ID OAuth User Impersonation to Microsoft Graph
Identifies potential session hijacking or token replay in Microsoft Entra ID. This rule detects cases where a user signs
in and subsequently accesses Microsoft Graph from a different IP address using the same session ID. This may indicate a
successful OAuth phishing attack, session hijacking, or token replay attack, where an adversary has stolen a session
cookie or refresh/access token and is impersonating the user from an alternate host or location.
**Rule type**: esql
**Rule indices**:
**Rule Severity**: medium
**Risk Score**: 47
**Runs every**: 30m
**Searches indices from**: `now-31m`
**Maximum alerts per execution**: 100
**References**:
- [[https://www.volexity.com/blog/2025/04/22/phishing-for-codes-russian-threat-actors-target-microsoft-365-oauth-workflows/](https://www.volexity.com/blog/2025/04/22/phishing-for-codes-russian-threat-actors-target-microsoft-365-oauth-workflows/)](https://www.volexity.com/blog/2025/04/22/phishing-for-codes-russian-threat-actors-target-microsoft-365-oauth-workflows/)
- [[https://github.com/dirkjanm/ROADtools](https://github.com/dirkjanm/ROADtools)](https://github.com/dirkjanm/ROADtools)
- [[https://attack.mitre.org/techniques/T1078/004/](https://attack.mitre.org/techniques/T1078/004/)](https://attack.mitre.org/techniques/T1078/004/)
- [[https://pushsecurity.com/blog/consentfix](https://pushsecurity.com/blog/consentfix)](https://pushsecurity.com/blog/consentfix)

**Tags**:
- Domain: Cloud
- Domain: Identity
- Domain: API
- Data Source: Azure
- Data Source: Microsoft Entra ID
- Data Source: Microsoft Entra ID Sign-In Logs
- Data Source: Microsoft Graph
- Data Source: Microsoft Graph Activity Logs
- Use Case: Identity and Access Audit
- Use Case: Threat Detection
- Resources: Investigation Guide
- Tactic: Defense Evasion
- Tactic: Initial Access

**Version**: 7
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

#### Required Microsoft Entra ID Sign-In and Graph Activity Logs

This rule requires the Microsoft Entra ID Sign-In Logs and Microsoft Graph Activity Logs integration to be enabled and configured to collect audit and activity logs via Azure Event Hub.

## Investigation guide


## Triage and analysis


### Investigating Entra ID OAuth User Impersonation to Microsoft Graph

Identifies potential phishing, session hijacking or token replay in Microsoft Entra ID. This rule detects cases where a user signs in and subsequently accesses Microsoft Graph from a different IP address using the same session ID and client application. This may indicate a successful OAuth phishing attack, session hijacking, or token replay attack, where an adversary has stolen a session cookie or refresh/access token and is impersonating the user from an alternate host or location.
This rule uses ESQL aggregations and thus has dynamically generated fields. Correlation of the values in the alert document may need to be
performed to the original sign-in and Graph events for further context.

### Possible investigation steps

- This rule relies on an aggregation-based ESQL query, therefore the alert document will contain dynamically generated fields.
  - To pivot into the original events, it is recommended to use the values captured to filter in timeline or discovery for the original sign-in and Graph events.
- Review the session ID and user ID to identify the user account involved in the suspicious activity.
- Check the source addresses involved in the sign-in and Graph access to determine if they are known or expected locations for the user.
  - The sign-in source addresses should be two, one for the initial phishing sign-in and the other when exchanging the auth code for a token by the adversary.
- The Graph API source address should identify the IP address used by the adversary to access Microsoft Graph.
- Review the user agent strings for the sign-in and Graph access events to identify any anomalies or indicators of compromise.
- Analyze the Graph permission scopes to identify what resources were accessed and whether they align with the user's expected behavior.
- Check the timestamp difference between the sign-in and Graph access events to determine if they occurred within a reasonable time frame that would suggest successful phishing to token issuance and then Graph access.
- Identify the original sign-in event to investigation if conditional access policies were applied, such as requiring multi-factor authentication or blocking access from risky locations. In phishing scenarios, these policies likely were applied as the victim user would have been prompted to authenticate.


### False positive analysis

- This pattern may occur during legitimate device switching or roaming between networks (e.g., corporate to mobile).
- Developers or power users leveraging multiple environments may also trigger this detection if session persistence spans IP ranges. Still, this behavior is rare and warrants investigation when rapid IP switching and Graph access are involved.


### Response and remediation

- If confirmed malicious, revoke all refresh/access tokens for the user principal.
- Block the source IP(s) involved in the Graph access.
- Notify the user and reset credentials.
- Review session control policies and conditional access enforcement.
- Monitor for follow-on activity, such as lateral movement or privilege escalation.
- Review conditional access policies to ensure they are enforced correctly.


## Rule Query

```esql
from logs-azure.signinlogs-*, logs-azure.graphactivitylogs-* metadata _id, _version, _index
| where
    (event.dataset == "azure.signinlogs"
     and source.`as`.organization.name != "MICROSOFT-CORP-MSN-AS-BLOCK"
     and azure.signinlogs.properties.session_id is not null)
    or
    (event.dataset == "azure.graphactivitylogs"
     and source.`as`.organization.name != "MICROSOFT-CORP-MSN-AS-BLOCK"
     and azure.graphactivitylogs.properties.c_sid is not null)

| eval
    Esql.azure_signinlogs_properties_session_id_coalesce = coalesce(azure.signinlogs.properties.session_id, azure.graphactivitylogs.properties.c_sid),
    Esql.azure_signinlogs_properties_user_id_coalesce = coalesce(azure.signinlogs.properties.user_id, azure.graphactivitylogs.properties.user_principal_object_id),
    Esql.azure_signinlogs_properties_app_id_coalesce = coalesce(azure.signinlogs.properties.app_id, azure.graphactivitylogs.properties.app_id),
    Esql.source_ip = source.ip,
    Esql.@timestamp = @timestamp,
    Esql.event_type_case = case(
        event.dataset == "azure.signinlogs", "signin",
        event.dataset == "azure.graphactivitylogs", "graph",
        "other"
    )

| where Esql.azure_signinlogs_properties_app_id_coalesce not in (
    "4354e225-50c9-4423-9ece-2d5afd904870", 
    "cc15fd57-2c6c-4117-a88c-83b1d56b4bbe", 
    "ecd6b820-32c2-49b6-98a6-444530e5a77a", 
    "e8be65d6-d430-4289-a665-51bf2a194bda", 
    "ab9b8c07-8f02-4f72-87fa-80105867a763", 
    "394866fc-eedb-4f01-8536-3ff84b16be2a", 
    "66a88757-258c-4c72-893c-3e8bed4d6899", 
    "9ea1ad79-fdb6-4f9a-8bc3-2b70f96e34c7", 
    "d7b530a4-7680-4c23-a8bf-c52c121d2e87", 
    "6f7e0f60-9401-4f5b-98e2-cf15bd5fd5e3", 
    "52c2e0b5-c7b6-4d11-a89c-21e42bcec444", 
    "27922004-5251-4030-b22d-91ecd9a37ea4", 
    "bb893c22-978d-4cd4-a6f7-bb6cc0d6e6ce", 
    "26a7ee05-5602-4d76-a7ba-eae8b7b67941", 
    "66a88757-258c-4c72-893c-3e8bed4d6899", 
    "9ea1ad79-fdb6-4f9a-8bc3-2b70f96e34c7", 
    "d7b530a4-7680-4c23-a8bf-c52c121d2e87", 
    "00000007-0000-0000-c000-000000000000", 
    "6bc3b958-689b-49f5-9006-36d165f30e00", 
    "0ec893e0-5785-4de6-99da-4ed124e5296c", 
    "fc108d3f-543d-4374-bbff-c7c51f651fe5", 
    "01fc33a7-78ba-4d2f-a4b7-768e336e890e"  
    )

| keep
    Esql.azure_signinlogs_properties_session_id_coalesce,
    Esql.source_ip,
    Esql.@timestamp,
    Esql.event_type_case,
    Esql.azure_signinlogs_properties_user_id_coalesce,
    Esql.azure_signinlogs_properties_app_id_coalesce,
    source.`as`.organization.name,
    user_agent.original,
    url.original,
    azure.graphactivitylogs.properties.scopes,
    azure.signinlogs.properties.user_principal_name

| stats
    Esql.azure_signinlogs_properties_user_id_coalesce_values = values(Esql.azure_signinlogs_properties_user_id_coalesce),
    Esql.azure_signinlogs_properties_session_id_coalesce_values = values(Esql.azure_signinlogs_properties_session_id_coalesce),
    Esql_priv.azure_signinlogs_properties_user_principal_name_values = values(azure.signinlogs.properties.user_principal_name),
    Esql.source_ip_values = values(Esql.source_ip),
    Esql.source_ip_count_distinct = count_distinct(Esql.source_ip),
    Esql.source_as_organization_name_values = values(source.`as`.organization.name),
    Esql.user_agent_original_values = values(user_agent.original),
    Esql.azure_signinlogs_properties_app_id_coalesce_values = values(Esql.azure_signinlogs_properties_app_id_coalesce),
    Esql.azure_signinlogs_properties_app_id_coalesce_count_distinct = count_distinct(Esql.azure_signinlogs_properties_app_id_coalesce),
    Esql.event_type_case_values = values(Esql.event_type_case),
    Esql.event_type_case_count_distinct = count_distinct(Esql.event_type_case),
    Esql.signin_time_min = min(case(Esql.event_type_case == "signin", Esql.@timestamp, null)),
    Esql.graph_time_min = min(case(Esql.event_type_case == "graph", Esql.@timestamp, null)),
    Esql.url_original_values = values(url.original),
    Esql.azure_graphactivitylogs_properties_scopes_values = values(azure.graphactivitylogs.properties.scopes),
    Esql.event_count = count()
  by
    Esql.azure_signinlogs_properties_session_id_coalesce,
    Esql.azure_signinlogs_properties_app_id_coalesce,
    Esql.azure_signinlogs_properties_user_id_coalesce

| eval
    Esql.event_signin_to_graph_delay_minutes_date_diff = date_diff("minutes", Esql.signin_time_min, Esql.graph_time_min),
    Esql.event_signin_to_graph_delay_days_date_diff = date_diff("days", Esql.signin_time_min, Esql.graph_time_min)

| where
    Esql.event_type_case_count_distinct > 1 and
    Esql.source_ip_count_distinct > 1 and
    Esql.azure_signinlogs_properties_app_id_coalesce_count_distinct == 1 and
    Esql.signin_time_min is not null and
    Esql.graph_time_min is not null and
    Esql.event_signin_to_graph_delay_minutes_date_diff >= 0 and
    Esql.event_signin_to_graph_delay_days_date_diff == 0
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Initial Access
- Id: TA0001
- Reference URL: [[https://attack.mitre.org/tactics/TA0001/](https://attack.mitre.org/tactics/TA0001/)](https://attack.mitre.org/tactics/TA0001/)
- Technique:
  - Name: Valid Accounts
- Id: T1078
- Reference URL: [[https://attack.mitre.org/techniques/T1078/](https://attack.mitre.org/techniques/T1078/)](https://attack.mitre.org/techniques/T1078/)
- Sub Technique:
  - Name: Cloud Accounts
- Id: T1078.004
- Reference URL: [[https://attack.mitre.org/techniques/T1078/004/](https://attack.mitre.org/techniques/T1078/004/)](https://attack.mitre.org/techniques/T1078/004/)

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Defense Evasion
- Id: TA0005
- Reference URL: [[https://attack.mitre.org/tactics/TA0005/](https://attack.mitre.org/tactics/TA0005/)](https://attack.mitre.org/tactics/TA0005/)
- Technique:
  - Name: Use Alternate Authentication Material
- Id: T1550
- Reference URL: [[https://attack.mitre.org/techniques/T1550/](https://attack.mitre.org/techniques/T1550/)](https://attack.mitre.org/techniques/T1550/)
- Sub Technique:
  - Name: Application Access Token
- Id: T1550.001
- Reference URL: [[https://attack.mitre.org/techniques/T1550/001/](https://attack.mitre.org/techniques/T1550/001/)](https://attack.mitre.org/techniques/T1550/001/)