﻿---
title: AWS RDS DB Instance or Cluster Password Modified
description: Identifies the modification of the master password for an AWS RDS DB instance or cluster. Changing the master password is a legitimate recovery action...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/integrations/aws/persistence_rds_db_instance_password_modified
products:
  - Elastic Security
---

# AWS RDS DB Instance or Cluster Password Modified
Identifies the modification of the master password for an AWS RDS DB instance or cluster. Changing the master password
is a legitimate recovery action when access is lost, but adversaries with sufficient permissions may modify it to regain
access, establish persistence, bypass existing controls, or escalate privileges within a compromised environment.
Because RDS does not expose the password in API responses, this operation can meaningfully alter access pathways to
sensitive data stores.
**Rule type**: eql
**Rule indices**:
- filebeat-*
- logs-aws.cloudtrail-*

**Rule Severity**: medium
**Risk Score**: 47
**Runs every**: 
**Searches indices from**: `now-6m`
**Maximum alerts per execution**: 100
**References**:
- [[https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html)](https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html)
- [[https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Overview.DBInstance.Modifying.html)
- [[https://cloud.hacktricks.xyz/pentesting-cloud/aws-security/aws-privilege-escalation/aws-rds-privesc#rds-modifydbinstance](https://cloud.hacktricks.xyz/pentesting-cloud/aws-security/aws-privilege-escalation/aws-rds-privesc#rds-modifydbinstance)](https://cloud.hacktricks.xyz/pentesting-cloud/aws-security/aws-privilege-escalation/aws-rds-privesc#rds-modifydbinstance)

**Tags**:
- Domain: Cloud
- Data Source: AWS
- Data Source: Amazon Web Services
- Data Source: AWS RDS
- Resources: Investigation Guide
- Use Case: Threat Detection
- Tactic: Persistence
- Tactic: Privilege Escalation
- Tactic: Defense Evasion

**Version**: 7
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Investigation guide


## Triage and analysis


### Investigating AWS RDS DB Instance or Cluster Password Modified

The RDS master user password controls privileged access to a database instance or cluster. Modifying it can immediately shift access from one operator to another, break application functionality, or allow an adversary to regain control over a compromised DB instance. Because RDS never returns the password via API, this operation is a strong signal of intentional access reconfiguration.
This rule detects successful password-modification events via `ModifyDBInstance` or `ModifyDBCluster`. Such changes may indicate credential loss recovery—or malicious actions related to persistence, privilege escalation, or defense evasion.

#### Possible investigation steps

- **Identify the actor and execution context**
  - Review `aws.cloudtrail.user_identity.arn` and `aws.cloudtrail.user_identity.access_key_id`.
- Inspect `user.name`, `source.ip`, and `user_agent.original` to determine whether the modification originated from expected networks, automation roles, or unusual sources.
- **Determine what was modified**
  - Examine `aws.cloudtrail.request_parameters` to identify:
  - The DB instance or cluster identifier.
- Whether other parameters were modified in the same call (e.g., `manageMasterUserPassword`, engine version, instance class, parameter group).
- Review instance metadata in AWS to understand the sensitivity level, environment (prod/stage/dev), and potential business impact.
- **Reconstruct timing and associated actions**
  - Use `@timestamp` to compare the event against:
  - Recent configuration changes such as `ModifyDBInstance`, `ModifyDBCluster`, or networking/security group updates.
- Other access-related operations (e.g., `AddRoleToDBInstance`, changes to Secrets Manager associations, disabling deletion protection).
- Check for signs of credential misuse leading up to the event (e.g., `DescribeDBInstances`, `GetCallerIdentity`, unauthorized console logins).
- **Correlate with broader activity**
  - Pivot in CloudTrail using the same access key, principal ARN, or source IP.
- Look for:
  - Privilege-escalating or persistence-related behavior (IAM policy changes, role modifications, STS session creation).
- Subsequent DB-impacting operations, such as snapshot deletion, backup retention changes, or cluster deletion.
- Evidence of data access anomalies (backup exports, data snapshot copies, cross-region actions).
- **Validate intent with operational owners**
  - Confirm with DBAs, platform engineers, and application owners whether the password change:
  - Was requested or scheduled.
- Aligns with pending migrations, credential rotations, or recovery actions.
- If not recognized, treat this as a high-risk event requiring deeper containment.


### False positive analysis

- **Recovery or maintenance tasks**
  - Password resets occur during lost-credential scenarios or planned rotations. Confirm if this aligns with a documented workflow.
- **Secrets Manager integration changes**
  - When `manageMasterUserPassword` is toggled or Secrets Manager rotates passwords, a modification event may occur. Validate whether an automation pipeline triggered the change.
- **Non-production workloads**
  - Development or staging environments may see frequent password resets. Consider tuning exceptions based on tags, instance identifiers, or IAM roles tied to automation.


### Response and remediation

- **Contain unauthorized access**
  - If activity is suspicious:
  - Immediately rotate the master password again using a secure, validated workflow.
- Verify whether Secrets Manager integration was disabled (`manageMasterUserPassword=false`) and restore it if necessary.
- Restrict inbound DB access by tightening associated security group rules or isolating the instance temporarily.
- **Investigate surrounding activity**
  - Review CloudTrail to identify:
  - Who accessed the instance after the password change.
- Whether any destructive or data-exfiltrating RDS actions occurred.
- Other IAM or STS activity tied to the same user or session.
- **Restore guardrails and enhance monitoring**
  - Ensure deletion protection, backup retention, and networking controls are correctly configured.
- Add real-time alerts for password-related modifications and high-risk RDS API actions.
- **Strengthen IAM and operational controls**
  - Limit permissions for `rds:ModifyDBInstance` and `rds:ModifyDBCluster`, especially when modifying authentication parameters.
- Require MFA and role-based access for DB administrators.
- Tighten SCPs or Config rules to restrict unauthorized DB configuration changes.


### Additional information

- **[AWS IR Playbooks](https://github.com/aws-samples/aws-incident-response-playbooks/blob/c151b0dc091755fffd4d662a8f29e2f6794da52c/playbooks/)**
- **[AWS Customer Playbook Framework](https://github.com/aws-samples/aws-customer-playbook-framework/tree/a8c7b313636b406a375952ac00b2d68e89a991f2/docs)**
- **Security Best Practices:** [AWS Knowledge Center – Security Best Practices](https://aws.amazon.com/premiumsupport/knowledge-center/security-best-practices/).


## Rule Query

```eql
info where event.dataset == "aws.cloudtrail"
    and event.provider == "rds.amazonaws.com"
    and event.action in ("ModifyDBInstance", "ModifyDBCluster")
    and event.outcome == "success"
    and stringContains(aws.cloudtrail.request_parameters, "masterUserPassword=*")
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Persistence
- Id: TA0003
- Reference URL: [[https://attack.mitre.org/tactics/TA0003/](https://attack.mitre.org/tactics/TA0003/)](https://attack.mitre.org/tactics/TA0003/)
- Technique:
  - Name: Account Manipulation
- Id: T1098
- Reference URL: [[https://attack.mitre.org/techniques/T1098/](https://attack.mitre.org/techniques/T1098/)](https://attack.mitre.org/techniques/T1098/)
- Sub Technique:
  - Name: Additional Cloud Credentials
- Id: T1098.001
- Reference URL: [[https://attack.mitre.org/techniques/T1098/001/](https://attack.mitre.org/techniques/T1098/001/)](https://attack.mitre.org/techniques/T1098/001/)

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Privilege Escalation
- Id: TA0004
- Reference URL: [[https://attack.mitre.org/tactics/TA0004/](https://attack.mitre.org/tactics/TA0004/)](https://attack.mitre.org/tactics/TA0004/)

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Defense Evasion
- Id: TA0005
- Reference URL: [[https://attack.mitre.org/tactics/TA0005/](https://attack.mitre.org/tactics/TA0005/)](https://attack.mitre.org/tactics/TA0005/)