﻿---
title: AWS EC2 Instance Console Login via Assumed Role
description: Detects successful AWS Management Console or federation login activity performed using an EC2 instance’s assumed role credentials. EC2 instances typically...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/integrations/aws/lateral_movement_ec2_instance_console_login
products:
  - Elastic Security
---

# AWS EC2 Instance Console Login via Assumed Role
Detects successful AWS Management Console or federation login activity performed using an EC2 instance’s assumed role
credentials. EC2 instances typically use temporary credentials to make API calls, not to authenticate interactively via
the console. A successful "ConsoleLogin" or "GetSigninToken" event using a session pattern that includes "i-" (the EC2
instance ID) is highly anomalous and may indicate that an adversary obtained the instance’s temporary credentials from
the instance metadata service (IMDS) and used them to access the console. Such activity can enable lateral movement,
privilege escalation, or persistence within the AWS account.
**Rule type**: eql
**Rule indices**:
- filebeat-*
- logs-aws.cloudtrail-*

**Rule Severity**: high
**Risk Score**: 73
**Runs every**: 
**Searches indices from**: `now-6m`
**Maximum alerts per execution**: 100
**References**:
- [[https://redcanary.com/blog/aws-sts/](https://redcanary.com/blog/aws-sts/)](https://redcanary.com/blog/aws-sts/)
- [[https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_enable-console-custom-url.html/](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_enable-console-custom-url.html/)](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_enable-console-custom-url.html/)

**Tags**:
- Domain: Cloud
- Data Source: AWS
- Data Source: Amazon Web Services
- Data Source: AWS EC2
- Data Source: AWS STS
- Data Source: AWS Sign-In
- Use Case: Identity and Access Audit
- Tactic: Lateral Movement
- Tactic: Credential Access
- Tactic: Persistence
- Resources: Investigation Guide

**Version**: 6
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Investigation guide


## Triage and analysis


### Investigating AWS EC2 Instance Console Login via Assumed Role

This rule detects successful AWS console or federation logins using temporary credentials tied to EC2 instance profiles. Under normal conditions, EC2 instances use their temporary credentials for programmatic API access — **not** for interactive console sessions. When an attacker gains access to an instance’s IMDS (Instance Metadata Service) or its environment variables, they may retrieve temporary STS credentials and attempt console logins to gain full access to the AWS account. A successful login of this type is rare and high-risk, as it strongly suggests credential theft or unauthorized session hijacking.

#### Possible investigation steps

- **Identify the source and actor**
  - Review `aws.cloudtrail.user_identity.arn`, `user.id`, and `user_agent.original` fields to confirm the session originated from an EC2 instance (`:i-` pattern).
- Correlate the instance ID (`i-xxxxxx`) with the specific EC2 instance in your environment to identify its owner, purpose, and running applications.
- Check `source.ip` and `cloud.region` to determine if the login originated from within AWS infrastructure (expected) or an external location (suspicious).
- **Correlate surrounding activity**
  - Pivot in Timeline to view the sequence of events leading up to the login, including:
  - STS token retrievals (`GetSessionToken`, `AssumeRole`, `GetCallerIdentity`)
- Calls to the IMDS endpoint or local credential exfiltration attempts from the instance.
- Investigate whether the same role or credentials were used for API actions following the login (e.g., `CreateUser`, `AttachRolePolicy`, or `ListBuckets`).
- **Assess IAM role exposure**
  - Determine which IAM role was associated with the instance at the time of the event and review its attached permissions.
- Evaluate whether the role grants console access or permissions beyond what that workload normally requires.
- Check for any recent changes to that role’s trust policy or attached policies.
- **Validate authorization**
  - Contact the EC2 instance owner or service team to confirm if any legitimate process should be logging in to the console.
- If no legitimate activity can explain the login, treat the credentials as compromised.


### False positive analysis

This is very uncommon behavior.

Known legitimate causes include:
- AWS or internal security automation that programmatically initiates console sessions for validation or testing.
- Forensic or incident-response automation that logs in using temporary credentials from a compromised instance.
- Red-team or penetration-testing activity designed to validate IMDS exposure or lateral movement scenarios.

For any other occurrence, treat the alert as potentially malicious.

Validate through:
- The originating instance’s purpose and owner.
- Known automation patterns in `user_agent.original`.
- The timestamp alignment with planned testing or security validation.


### Response and remediation

- **Immediate containment**
  - Revoke the temporary credentials for the affected role (`aws sts revoke-session-token` or rotate the role credentials).
- Isolate the associated EC2 instance (e.g., detach it from the VPC or security groups) to prevent further credential misuse.
- Invalidate active console sessions via AWS CLI or the AWS Console.
- **Investigation and scoping**
  - Review CloudTrail logs for all actions associated with the compromised role in the preceding 24 hours.
- Determine if additional roles or instances show similar `ConsoleLogin` patterns.
- Search for network indicators of IMDS exploitation (e.g., requests to `169.254.169.254` from unauthorized binaries or users).
- **Recovery and hardening**
  - Rotate all credentials for affected roles and users.
- Apply IMDSv2 enforcement to prevent credential harvesting from EC2 metadata.
- Implement restrictive IAM policies: deny console access (`iam:PassRole`, `sts:GetFederationToken`) for non-human roles.


### Additional information

- **[AWS IR Playbooks](https://github.com/aws-samples/aws-incident-response-playbooks/blob/c151b0dc091755fffd4d662a8f29e2f6794da52c/playbooks/)**
- **[AWS Customer Playbook Framework](https://github.com/aws-samples/aws-customer-playbook-framework/tree/a8c7b313636b406a375952ac00b2d68e89a991f2/docs)**
- **Security Best Practices:** [AWS Knowledge Center – Security Best Practices](https://aws.amazon.com/premiumsupport/knowledge-center/security-best-practices/).


## Rule Query

```eql
info where event.dataset == "aws.cloudtrail"
   and event.provider == "signin.amazonaws.com"
   and event.action in ("ConsoleLogin", "GetSigninToken")
   and event.outcome == "success"
   and aws.cloudtrail.user_identity.type == "AssumedRole"
   and stringContains (user.id, ":i-")
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Lateral Movement
- Id: TA0008
- Reference URL: [[https://attack.mitre.org/tactics/TA0008/](https://attack.mitre.org/tactics/TA0008/)](https://attack.mitre.org/tactics/TA0008/)
- Technique:
  - Name: Remote Services
- Id: T1021
- Reference URL: [[https://attack.mitre.org/techniques/T1021/](https://attack.mitre.org/techniques/T1021/)](https://attack.mitre.org/techniques/T1021/)
- Sub Technique:
  - Name: Cloud Services
- Id: T1021.007
- Reference URL: [[https://attack.mitre.org/techniques/T1021/007/](https://attack.mitre.org/techniques/T1021/007/)](https://attack.mitre.org/techniques/T1021/007/)
- Technique:
  - Name: Use Alternate Authentication Material
- Id: T1550
- Reference URL: [[https://attack.mitre.org/techniques/T1550/](https://attack.mitre.org/techniques/T1550/)](https://attack.mitre.org/techniques/T1550/)
- Sub Technique:
  - Name: Application Access Token
- Id: T1550.001
- Reference URL: [[https://attack.mitre.org/techniques/T1550/001/](https://attack.mitre.org/techniques/T1550/001/)](https://attack.mitre.org/techniques/T1550/001/)

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Persistence
- Id: TA0003
- Reference URL: [[https://attack.mitre.org/tactics/TA0003/](https://attack.mitre.org/tactics/TA0003/)](https://attack.mitre.org/tactics/TA0003/)
- Technique:
  - Name: Valid Accounts
- Id: T1078
- Reference URL: [[https://attack.mitre.org/techniques/T1078/](https://attack.mitre.org/techniques/T1078/)](https://attack.mitre.org/techniques/T1078/)
- Sub Technique:
  - Name: Cloud Accounts
- Id: T1078.004
- Reference URL: [[https://attack.mitre.org/techniques/T1078/004/](https://attack.mitre.org/techniques/T1078/004/)](https://attack.mitre.org/techniques/T1078/004/)

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Credential Access
- Id: TA0006
- Reference URL: [[https://attack.mitre.org/tactics/TA0006/](https://attack.mitre.org/tactics/TA0006/)](https://attack.mitre.org/tactics/TA0006/)
- Technique:
  - Name: Unsecured Credentials
- Id: T1552
- Reference URL: [[https://attack.mitre.org/techniques/T1552/](https://attack.mitre.org/techniques/T1552/)](https://attack.mitre.org/techniques/T1552/)
- Sub Technique:
  - Name: Cloud Instance Metadata API
- Id: T1552.005
- Reference URL: [[https://attack.mitre.org/techniques/T1552/005/](https://attack.mitre.org/techniques/T1552/005/)](https://attack.mitre.org/techniques/T1552/005/)