﻿---
title: AWS Access Token Used from Multiple Addresses
description: This rule identifies potentially suspicious activity by detecting instances where a single IAM user's temporary session token is accessed from multiple...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/integrations/aws/initial_access_iam_session_token_used_from_multiple_addresses
products:
  - Elastic Security
---

# AWS Access Token Used from Multiple Addresses
This rule identifies potentially suspicious activity by detecting instances where a single IAM user's temporary session
token is accessed from multiple IP addresses within a short time frame. Such behavior may suggest that an adversary has
compromised temporary credentials and is utilizing them from various locations. To enhance detection accuracy and
minimize false positives, the rule incorporates criteria that evaluate unique IP addresses, user agents, cities, and
networks. These additional checks help distinguish between legitimate distributed access patterns and potential
credential misuse. Detected activities are classified into different types based on the combination of unique
indicators, with each classification assigned a fidelity score reflecting the likelihood of malicious behavior. High
fidelity scores are given to patterns most indicative of threats, such as multiple unique IPs, networks, cities, and
user agents. Medium and low fidelity scores correspond to less severe patterns, enabling security teams to effectively
prioritize alerts.
**Rule type**: esql
**Rule indices**:
**Rule Severity**: medium
**Risk Score**: 47
**Runs every**: 5m
**Searches indices from**: `now-32m`
**Maximum alerts per execution**: 100
**References**:
- [[https://www.sygnia.co/blog/sygnia-investigation-bybit-hack/](https://www.sygnia.co/blog/sygnia-investigation-bybit-hack/)](https://www.sygnia.co/blog/sygnia-investigation-bybit-hack/)

**Tags**:
- Domain: Cloud
- Data Source: AWS
- Data Source: Amazon Web Services
- Data Source: AWS IAM
- Data Source: AWS CloudTrail
- Tactic: Initial Access
- Use Case: Identity and Access Audit
- Resources: Investigation Guide

**Version**: 105
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Investigation guide


## Triage and Analysis


### Investigating AWS Access Token Used from Multiple Addresses

Access tokens are bound to a single user. Usage from multiple IP addresses may indicate the token was stolen and used elsewhere. By correlating this with additional detection criteria like multiple user agents, different cities, and different networks, we can improve the fidelity of the rule and help to eliminate false positives associated with expected behavior, like dual-stack IPV4/IPV6 usage.

#### Possible investigation steps

- **Identify the IAM User**: Examine the `aws.cloudtrail.user_identity.arn` stored in `user_id` and correlate with the `source.ips` stored in `ip_list` and `unique_ips` count to determine how widely the token was used.
- **Correlate Additional Detection Context**: Examine `activity_type` and `fidelity_score` to determine additional cities, networks or user agents associated with the token usage.
- **Determine Access Key Type**: Examine the `access_key_id` to determine whether the token is short-term (beginning with ASIA) or long-term (beginning with AKIA).
- **Check Recent MFA Events**: Determine whether the user recently enabled MFA, registered devices, or assumed a role using this token.
- **Review Workload Context**: Confirm whether the user was expected to be active across multiple cities, networks or user agent environments.
- **Trace Adversary Movement**: Pivot to related actions (e.g., `s3:ListBuckets`, `iam:ListUsers`, `sts:GetCallerIdentity`) to track further enumeration.


### False positive analysis

- Automation frameworks that rotate through multiple IPs or cloud functions with dynamic egress IPs may cause this alert to fire.
- Confirm geolocation and workload context before escalating.


### Response and remediation

- **Revoke the Token**: Disable or rotate the IAM credentials and invalidate the temporary session token.
- **Audit the Environment**: Look for signs of lateral movement or data access during the token's validity.
- **Strengthen Controls**: Require MFA for high-privilege actions, restrict access via policy conditions (e.g., IP range or device).


### Additional information

- [IAM Long-Term Credentials](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_access-keys.html)
- [STS Temporary Credentials](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp.html)
- [Using MFA with Temporary Credentials](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_enable-regions.html)
- [AWS Threat Detection Use Cases](https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards-fsbp-controls.html)


## Rule Query

```esql
from logs-aws.cloudtrail* metadata _id, _version, _index
| where @timestamp > now() - 30 minutes
  and event.dataset == "aws.cloudtrail"
  and aws.cloudtrail.user_identity.arn is not null
  and aws.cloudtrail.user_identity.type == "IAMUser"
  and source.ip is not null
  and aws.cloudtrail.user_identity.access_key_id is not null
  and not (
    user_agent.original like "*Terraform*" or
    user_agent.original like "*Ansible*" or
    user_agent.original like "*Pulumi*"
  )
  and `source.as.organization.name` != "AMAZON-AES"
  and not ((
    `source.as.organization.name` == "AMAZON-02" and aws.cloudtrail.event_category == "Data"))
  and event.provider not in (
    "health.amazonaws.com", "monitoring.amazonaws.com", "notifications.amazonaws.com",
    "ce.amazonaws.com", "cost-optimization-hub.amazonaws.com",
    "servicecatalog-appregistry.amazonaws.com", "securityhub.amazonaws.com", 
    "account.amazonaws.com", "budgets.amazonaws.com", "freetier.amazonaws.com", "support.amazonaws.com",
    "support-console.amazonaws.com"
  )

| eval
  Esql.time_window_date_trunc = date_trunc(30 minutes, @timestamp),
  Esql.aws_cloudtrail_user_identity_arn = aws.cloudtrail.user_identity.arn,
  Esql.aws_cloudtrail_user_identity_access_key_id = aws.cloudtrail.user_identity.access_key_id,
  Esql.source_ip = source.ip,
  Esql.user_agent_original = user_agent.original,
  Esql.source_ip_string = to_string(source.ip),
  Esql.source_ip_user_agent_pair = concat(Esql.source_ip_string, " - ", user_agent.original),
  Esql.source_ip_city_pair = concat(Esql.source_ip_string, " - ", source.geo.city_name),
  Esql.source_geo_city_name = source.geo.city_name,
  Esql.source_network_org_name = `source.as.organization.name`,
  Esql.source_ip_network_pair = concat(Esql.source_ip_string, "-", `source.as.organization.name`),
  Esql.event_timestamp = @timestamp,
  Esql.data_stream_namespace = data_stream.namespace

| stats
  Esql.event_action_values = values(event.action),
  Esql.event_provider_values = values(event.provider),
  Esql.aws_cloudtrail_user_identity_access_key_id_values = values(Esql.aws_cloudtrail_user_identity_access_key_id),
  Esql.aws_cloudtrail_user_identity_arn_values = values(Esql.aws_cloudtrail_user_identity_arn),
  Esql.source_ip_values = values(Esql.source_ip),
  Esql.user_agent_original_values = values(Esql.user_agent_original),
  Esql.source_ip_user_agent_pair_values = values(Esql.source_ip_user_agent_pair),
  Esql.source_geo_city_name_values = values(Esql.source_geo_city_name),
  Esql.source_ip_city_pair_values = values(Esql.source_ip_city_pair),
  Esql.source_network_org_name_values = values(Esql.source_network_org_name),
  Esql.source_ip_network_pair_values = values(Esql.source_ip_network_pair),
  Esql.source_ip_count_distinct = count_distinct(Esql.source_ip),
  Esql.user_agent_original_count_distinct = count_distinct(Esql.user_agent_original),
  Esql.source_geo_city_name_count_distinct = count_distinct(Esql.source_geo_city_name),
  Esql.source_network_org_name_count_distinct = count_distinct(Esql.source_network_org_name),
  Esql.data_stream_namespace_values = values(Esql.data_stream_namespace),
  Esql.timestamp_first_seen = min(Esql.event_timestamp),
  Esql.timestamp_last_seen = max(Esql.event_timestamp),
  Esql.event_count = count()
  by Esql.time_window_date_trunc, Esql.aws_cloudtrail_user_identity_access_key_id

| eval
  Esql.activity_type = case(
    Esql.source_ip_count_distinct >= 2 and Esql.source_network_org_name_count_distinct >= 2 and Esql.source_geo_city_name_count_distinct >= 2 and Esql.user_agent_original_count_distinct >= 2, "multiple_ip_network_city_user_agent",
    Esql.source_ip_count_distinct >= 2 and Esql.source_network_org_name_count_distinct >= 2 and Esql.source_geo_city_name_count_distinct >= 2, "multiple_ip_network_city",
    Esql.source_ip_count_distinct >= 2 and Esql.source_geo_city_name_count_distinct >= 2, "multiple_ip_and_city",
    Esql.source_ip_count_distinct >= 2 and Esql.source_network_org_name_count_distinct >= 2, "multiple_ip_and_network",
    Esql.source_ip_count_distinct >= 2 and Esql.user_agent_original_count_distinct >= 2, "multiple_ip_and_user_agent",
    "normal_activity"
  ),
  Esql.activity_fidelity_score = case(
    Esql.activity_type == "multiple_ip_network_city_user_agent", "high",
    Esql.activity_type == "multiple_ip_network_city", "high",
    Esql.activity_type == "multiple_ip_and_city", "medium",
    Esql.activity_type == "multiple_ip_and_network", "medium",
    Esql.activity_type == "multiple_ip_and_user_agent", "low"
  )

| keep
  Esql.time_window_date_trunc,
  Esql.activity_type,
  Esql.activity_fidelity_score,
  Esql.event_count,
  Esql.timestamp_first_seen,
  Esql.timestamp_last_seen,
  Esql.aws_cloudtrail_user_identity_arn_values,
  Esql.aws_cloudtrail_user_identity_access_key_id_values,
  Esql.event_action_values,
  Esql.event_provider_values,
  Esql.source_ip_values,
  Esql.user_agent_original_values,
  Esql.source_ip_user_agent_pair_values,
  Esql.source_geo_city_name_values,
  Esql.source_ip_city_pair_values,
  Esql.source_network_org_name_values,
  Esql.source_ip_network_pair_values,
  Esql.source_ip_count_distinct,
  Esql.user_agent_original_count_distinct,
  Esql.source_geo_city_name_count_distinct,
  Esql.source_network_org_name_count_distinct,
  Esql.data_stream_namespace_values

| where Esql.activity_fidelity_score == "high"

// this rule only alerts for "high" fidelity cases, to broaden the rule scope to include all activity
// change the final condition to 
// | where Esql.activity_type != "normal_activity"
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Initial Access
- Id: TA0001
- Reference URL: [[https://attack.mitre.org/tactics/TA0001/](https://attack.mitre.org/tactics/TA0001/)](https://attack.mitre.org/tactics/TA0001/)
- Technique:
  - Name: Valid Accounts
- Id: T1078
- Reference URL: [[https://attack.mitre.org/techniques/T1078/](https://attack.mitre.org/techniques/T1078/)](https://attack.mitre.org/techniques/T1078/)
- Sub Technique:
  - Name: Cloud Accounts
- Id: T1078.004
- Reference URL: [[https://attack.mitre.org/techniques/T1078/004/](https://attack.mitre.org/techniques/T1078/004/)](https://attack.mitre.org/techniques/T1078/004/)