﻿---
title: AWS S3 Object Encryption Using External KMS Key
description: Identifies use of the S3 CopyObject API where the destination object is encrypted using an AWS KMS key from an external AWS account. This behavior may...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/integrations/aws/impact_s3_object_encryption_with_external_key
products:
  - Elastic Security
---

# AWS S3 Object Encryption Using External KMS Key
Identifies use of the S3 CopyObject API where the destination object is encrypted using an AWS KMS key from an external
AWS account. This behavior may indicate ransomware-style impact activity where an adversary with access to a
misconfigured S3 bucket encrypts objects using a KMS key they control, preventing the bucket owner from decrypting their
own data. This technique is a critical early signal of destructive intent or cross-account misuse.
**Rule type**: esql
**Rule indices**:
**Rule Severity**: medium
**Risk Score**: 47
**Runs every**: 
**Searches indices from**: `now-6m`
**Maximum alerts per execution**: 100
**References**:
- [[https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingKMSEncryption.html/](https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingKMSEncryption.html/)](https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingKMSEncryption.html/)
- [[https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html/](https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html/)](https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html/)
- [[https://www.gem.security/post/cloud-ransomware-a-new-take-on-an-old-attack-pattern/](https://www.gem.security/post/cloud-ransomware-a-new-take-on-an-old-attack-pattern/)](https://www.gem.security/post/cloud-ransomware-a-new-take-on-an-old-attack-pattern/)
- [[https://rhinosecuritylabs.com/aws/s3-ransomware-part-1-attack-vector/](https://rhinosecuritylabs.com/aws/s3-ransomware-part-1-attack-vector/)](https://rhinosecuritylabs.com/aws/s3-ransomware-part-1-attack-vector/)

**Tags**:
- Domain: Cloud
- Data Source: AWS
- Data Source: Amazon Web Services
- Data Source: AWS S3
- Data Source: AWS KMS
- Use Case: Threat Detection
- Tactic: Impact
- Resources: Investigation Guide

**Version**: 10
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2
AWS S3 data event types need to be enabled in the CloudTrail trail configuration for CopyObject events.

## Investigation guide


## Triage and analysis


### Investigating AWS S3 Object Encryption Using External KMS Key

This rule detects when an S3 `CopyObject` operation encrypts an object using a KMS key belonging to a different AWS account than the bucket owner. This behavior is unusual and a strong indicator of:
- Cloud ransomware techniques, where adversaries encrypt data using a key only they control.
- Cross-account privilege misuse, especially when an unauthorized principal has write access to S3.
- Misconfigured bucket permissions, enabling principals from another account to perform privileged copy operations.
- Early impact-stage activity in incidents where attackers prepare to destroy availability or deny the owner access.

The rule uses ESQL to identify cases where the `cloud.account.id` (bucket owner) differs from the dissected `kms_key_account_id` used for encrypting the new object version.

#### Possible investigation steps

**Identify the actor and access pathway**
- Review `aws.cloudtrail.user_identity.arn` and `aws.cloudtrail.user_identity.access_key_id`.
- Check whether the caller is:
  - A legitimate cross-account automation role,
- A compromised IAM user or workload identity, or
- A federated identity behaving outside of normal patterns.
- Inspect `user_agent.original` to determine whether the action came from the AWS Console, CLI, SDK, or unusual tooling.

**Analyze the encryption behavior**
- Inspect the dissected KMS key fields:
  - `Esql.aws_cloudtrail_request_parameters_kms_key_account_id`
- `Esql.aws_cloudtrail_request_parameters_kms_key_id`
- Confirm whether the external key:
  - Belongs to an attacker-controlled account,
- Is unknown to your organization, or
- Lives in a shared or security tooling account.

**Assess the objects affected**
- Review:
  - `Esql.aws_cloudtrail_request_parameters_target_bucket_name`
- `Esql.aws_cloudtrail_request_parameters_target_object_key`
- Identify:
  - Whether objects were overwritten or new encrypted copies were created.
- The sensitivity or criticality of the affected data.
- Whether object versioning is enabled (important for recovery).

**Correlate surrounding access patterns**
Pivot in CloudTrail on:
- The same access key ID
- The same IAM principal
- Affected bucket ARN

Look for:
- `DeleteObject` or `DeleteObjects` calls (common in ransomware behavior)
- Mass enumeration prior to the event (`ListObjectsV2`, `GetObject`)
- Other impact-stage actions (`PutBucketPolicy`, `PutBucketAcl`, disabling logging)
- Attempts to encrypt additional objects in rapid succession

**Evaluate bucket permissions and exposure**
Review:
- S3 bucket policy changes
- IAM roles with `s3:PutObject` or `s3:PutObjectAcl` permissions
- Whether unintended cross-account `Principal` entries exist
- Whether the KMS key policy explicitly trusts your account or a foreign one

**Validate business justification**
- Confirm with storage, data engineering, or application teams whether:
  - Any migration, transformation, or backup workflows should be encrypting objects cross-account.
- Scheduled jobs or CI/CD pipelines were operating at the time of the event.


### False positive analysis

- **Expected cross-account encryption**

  Many organizations use centralized encryption accounts or shared security accounts. Validate:
  - Whether the KMS key account is part of your AWS Organization
- Whether the workflow, role, or application is documented
- Whether the principal routinely performs CopyObject operations


### Response and remediation

**Contain and prevent further impact**
- Immediately restrict S3 write access for the principal involved.
- If the KMS key is attacker-controlled, the impacted objects may be unrecoverable without versioning.
- If object versioning is disabled, enable it on the affected bucket to strengthen future resilience.

**Investigate scope and severity**
- Identify:
  - Additional objects encrypted using external keys
- Related suspicious actions (delete, modify, exfiltration events)
- Whether any ransom markers or unauthorized files were uploaded
- Validate whether the external KMS key grants *decrypt* permission back to the bucket owner (rare in attacker use).

**Recover and secure the bucket**
- Restore accessible previous versions if versioning is enabled.
- Revoke unauthorized access key pairs or session credentials.
- Audit bucket policies, ACLs, and IAM conditions (`aws:PrincipalArn`, `aws:SourceAccount`, `aws:SourceArn`).
- Tighten cross-account access controls:
  - Remove unintended `Principal` clauses
- Restrict KMS usage to known accounts
- Enforce SCPs that block cross-account KMS use unless explicitly approved

**Long-term hardening**
- Integrate object-level access logging and S3 server access logging into security monitoring.
- Add AWS Config rules (or Security Hub controls) detecting:
  - Public buckets
- Cross-account access to S3
- KMS policies permitting foreign principals
- Document required cross-account workflows and add explicit allowlists.


### Additional information

- **[AWS IR Playbooks](https://github.com/aws-samples/aws-incident-response-playbooks/blob/c151b0dc091755fffd4d662a8f29e2f6794da52c/playbooks/)**
- **[AWS Customer Playbook Framework](https://github.com/aws-samples/aws-customer-playbook-framework/tree/a8c7b313636b406a375952ac00b2d68e89a991f2/docs)**
- **Security Best Practices:** [AWS Knowledge Center – Security Best Practices](https://aws.amazon.com/premiumsupport/knowledge-center/security-best-practices/).


## Rule Query

```esql
from logs-aws.cloudtrail-* metadata _id, _version, _index

// any successful S3 copy event
| where
  event.dataset == "aws.cloudtrail"
  and event.provider == "s3.amazonaws.com"
  and event.action == "CopyObject"
  and event.outcome == "success"

// dissect request parameters to extract KMS key info and target object info
| dissect aws.cloudtrail.request_parameters
    "{%{?bucketName}=%{Esql.aws_cloudtrail_request_parameters_target_bucket_name},%{?x-amz-server-side-encryption-aws-kms-key-id}=%{?arn}:%{?aws}:%{?kms}:%{?region}:%{Esql.aws_cloudtrail_request_parameters_kms_key_account_id}:%{?key}/%{Esql.aws_cloudtrail_request_parameters_kms_key_id},%{?Host}=%{?tls.client.server.name},%{?x-amz-server-side-encryption}=%{?server_side_encryption},%{?x-amz-copy-source}=%{?bucket.object.name},%{?key}=%{Esql.aws_cloudtrail_request_parameters_target_object_key}}"

// detect cross-account key usage
| where cloud.account.id != Esql.aws_cloudtrail_request_parameters_kms_key_account_id

// keep ECS and dissected fields
| keep
  @timestamp,
  data_stream.namespace,
  user.name,
  user_agent.original,
  source.ip,
  aws.cloudtrail.user_identity.arn,
  aws.cloudtrail.user_identity.type,
  aws.cloudtrail.user_identity.access_key_id,
  aws.cloudtrail.resources.arn,
  aws.cloudtrail.resources.type,
  event.action,
  event.outcome,
  cloud.account.id,
  cloud.region,
  aws.cloudtrail.request_parameters,
  aws.cloudtrail.response_elements,
  Esql.aws_cloudtrail_request_parameters_target_bucket_name,
  Esql.aws_cloudtrail_request_parameters_target_object_key,
  Esql.aws_cloudtrail_request_parameters_kms_key_account_id,
  Esql.aws_cloudtrail_request_parameters_kms_key_id,
  _id,
  _version,
  _index
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Impact
- Id: TA0040
- Reference URL: [[https://attack.mitre.org/tactics/TA0040/](https://attack.mitre.org/tactics/TA0040/)](https://attack.mitre.org/tactics/TA0040/)
- Technique:
  - Name: Data Encrypted for Impact
- Id: T1486
- Reference URL: [[https://attack.mitre.org/techniques/T1486/](https://attack.mitre.org/techniques/T1486/)](https://attack.mitre.org/techniques/T1486/)