﻿---
title: AWS EC2 EBS Snapshot Shared or Made Public
description: Detects when an Amazon Elastic Block Store (EBS) snapshot is shared with another AWS account or made public. EBS snapshots contain copies of data volumes...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/integrations/aws/exfiltration_ec2_ebs_snapshot_shared_with_another_account
products:
  - Elastic Security
---

# AWS EC2 EBS Snapshot Shared or Made Public
Detects when an Amazon Elastic Block Store (EBS) snapshot is shared with another AWS account or made public. EBS
snapshots contain copies of data volumes that may include sensitive or regulated information. Adversaries may exploit
ModifySnapshotAttribute to share snapshots with external accounts or the public, allowing them to copy and access data
in an environment they control. This activity often precedes data exfiltration or persistence operations, where the
attacker transfers stolen data out of the victim account or prepares a staging area for further exploitation.
**Rule type**: eql
**Rule indices**:
- filebeat-*
- logs-aws.cloudtrail-*

**Rule Severity**: medium
**Risk Score**: 47
**Runs every**: 
**Searches indices from**: `now-6m`
**Maximum alerts per execution**: 100
**References**:
- [[https://docs.aws.amazon.com/ebs/latest/userguide/ebs-modifying-snapshot-permissions.html](https://docs.aws.amazon.com/ebs/latest/userguide/ebs-modifying-snapshot-permissions.html)](https://docs.aws.amazon.com/ebs/latest/userguide/ebs-modifying-snapshot-permissions.html)
- [[https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html)](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html)
- [[https://cloud.hacktricks.xyz/pentesting-cloud/aws-security/aws-post-exploitation/aws-ec2-ebs-ssm-and-vpc-post-exploitation/aws-ebs-snapshot-dump](https://cloud.hacktricks.xyz/pentesting-cloud/aws-security/aws-post-exploitation/aws-ec2-ebs-ssm-and-vpc-post-exploitation/aws-ebs-snapshot-dump)](https://cloud.hacktricks.xyz/pentesting-cloud/aws-security/aws-post-exploitation/aws-ec2-ebs-ssm-and-vpc-post-exploitation/aws-ebs-snapshot-dump)
- [[https://hackingthe.cloud/aws/exploitation/Misconfigured_Resource-Based_Policies/exploting_public_resources_attack_playbook/](https://hackingthe.cloud/aws/exploitation/Misconfigured_Resource-Based_Policies/exploting_public_resources_attack_playbook/)](https://hackingthe.cloud/aws/exploitation/Misconfigured_Resource-Based_Policies/exploting_public_resources_attack_playbook/)

**Tags**:
- Domain: Cloud
- Data Source: AWS
- Data Source: Amazon Web Services
- Data Source: AWS EC2
- Use Case: Threat Detection
- Tactic: Exfiltration
- Resources: Investigation Guide

**Version**: 9
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Investigation guide


## Triage and analysis


### Investigating AWS EC2 EBS Snapshot Shared or Made Public

This rule detects when an Amazon Elastic Block Store (EBS) snapshot is shared with another AWS account or made public. EBS snapshots store copies of data volumes that may contain sensitive or regulated information. Adversaries may exploit the `ModifySnapshotAttribute` API to share these snapshots externally, allowing them to copy and access the data in an environment they control. This activity is commonly associated with data exfiltration or persistence techniques, where attackers transfer data outside the victim account or prepare backups they can later retrieve. Public sharing (`group=all`) represents a severe data exposure risk, as it makes the snapshot globally readable.

#### Possible investigation steps:

**Identify who performed the action**: Review `aws.cloudtrail.user_identity.arn` and `aws.cloudtrail.user_identity.access_key_id` to identify who modified the snapshot’s permissions. Evaluate whether this identity is authorized to share EBS snapshots (check IAM policies for `ec2:ModifySnapshotAttribute`).
**Analyze the source of the request**: Examine `source.ip` and `source.geo` fields to determine the geographical origin of the request. An unfamiliar or external location may indicate compromised credentials or unauthorized access. Review `user_agent.original` to confirm whether the request originated from an expected administrative tool or host.
**Examine the scope of the change**:
- Review `aws.cloudtrail.request_parameters` to determine which AWS account(s) were added to the `createVolumePermission` list.
  - If the account ID matches the snapshot owner’s account, this is redundant and typically non-malicious.
- If another account ID or `group=all` appears, verify whether the target is an approved AWS Organization account or an external party.
- Cross-check the affected `snapshotId` in the AWS console or via CLI (`describe-snapshot-attribute`) to confirm current sharing status.
- Identify whether other snapshots or AMIs were shared in the same timeframe.

**Correlate with other activities**:
- Search CloudTrail for related events involving the same actor or `source.ip`.
  - Look for `CreateSnapshot`, `CopySnapshot`, `ExportImage`, or `PutBucketAcl` events that could indicate broader exfiltration or replication behavior.
- Correlate with detections such as `EBS Snapshot Access Removed` or `EBS Encryption Disabled`, which may signal a coordinated campaign involving both exfiltration and impact.
- Check GuardDuty and Security Hub for findings related to data exposure, cross-account sharing, or unauthorized data transfer.

**Evaluate timing and intent**: Compare `@timestamp` against scheduled maintenance or approved change windows. Actions performed outside business hours or without documented change tickets should be prioritized for review.

### False positive analysis:

- **Authorized internal sharing**: Confirm if the snapshot sharing was part of an approved workflow, such as internal replication or migration between AWS Organization accounts.
- **Automated replication or tooling**: Infrastructure-as-code or backup automation may temporarily share snapshots for cross-region or cross-account transfers. Verify automation identifiers, source IPs, and tags.
- **Self-account addition**: Adding the owner’s own account ID to `createVolumePermission` has no operational impact and can be safely ignored.

If verified as legitimate, document the event under change management and reconcile it against organizational policies for snapshot sharing.

### Response and remediation:

**Containment and validation**
- If unauthorized, immediately remove added permissions using the AWS CLI:

  `aws ec2 modify-snapshot-attribute --snapshot-id <id> --create-volume-permission "Remove=[{UserId=<unauthorized_id>}]"`
- Revoke public sharing (`group=all`) to prevent external access.
- Restrict `ec2:ModifySnapshotAttribute` permissions to trusted administrative roles only.

**Investigate for data exfiltration or persistence**
- Determine whether the shared snapshot was copied to another account (`CopySnapshot`).
- Engage AWS Support if evidence suggests external copying or data theft.
- Review subsequent API calls or IAM changes for further persistence or data movement.

**Strengthen detection and monitoring**
- Enable AWS Config rules such as `ebs-snapshot-public-restorable-check`.
- Implement continuous monitoring for `ModifySnapshotAttribute` and `CopySnapshot` operations.
- Correlate future detections by actor, access key, and source IP to identify repeated or automated exfiltration attempts.

**Recovery and hardening**
- Enable default encryption and validate that all snapshots remain private.
- Apply Service Control Policies (SCPs) to prevent public snapshot sharing organization-wide.
- Audit existing snapshots to ensure no others have unauthorized permissions.
- Implement least-privilege IAM principles and enforce multi-factor authentication (MFA) for administrative accounts.


### Additional information

- **[AWS Incident Response Playbooks](https://github.com/aws-samples/aws-incident-response-playbooks/blob/c151b0dc091755fffd4d662a8f29e2f6794da52c/playbooks/)**: reference playbooks for investigating data exfiltration and unauthorized access.
- **[AWS Customer Playbook Framework](https://github.com/aws-samples/aws-customer-playbook-framework/)**: example framework for developing custom playbooks for snapshot configuration and data protection.
- **AWS Documentation**
  - [EBS Snapshot Permissions](https://docs.aws.amazon.com/ebs/latest/userguide/ebs-modifying-snapshot-permissions.html)
- [ModifySnapshotAttribute API Reference](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html)


## Rule Query

```eql
info where event.dataset == "aws.cloudtrail"  
  and event.action == "ModifySnapshotAttribute"
  and event.outcome == "success"
  and stringContains (aws.cloudtrail.request_parameters, "attributeType=CREATE_VOLUME_PERMISSION")
  and stringContains (aws.cloudtrail.request_parameters, "add=")
```

**Framework:** MITRE ATT&CK
- Tactic:
  - Name: Exfiltration
- Id: TA0010
- Reference URL: [[https://attack.mitre.org/tactics/TA0010/](https://attack.mitre.org/tactics/TA0010/)](https://attack.mitre.org/tactics/TA0010/)
- Technique:
  - Name: Transfer Data to Cloud Account
- Id: T1537
- Reference URL: [[https://attack.mitre.org/techniques/T1537/](https://attack.mitre.org/techniques/T1537/)](https://attack.mitre.org/techniques/T1537/)