﻿---
title: Elastic Defend and Network Security Alerts Correlation
description: This rule correlate any Elastic Defend alert with a set of suspicious events from Network security devices like Palo Alto Networks (PANW) and Fortinet...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/cross-platform/multiple_alerts_elastic_defend_netsecurity_by_host
products:
  - Elastic Security
---

# Elastic Defend and Network Security Alerts Correlation
This rule correlate any Elastic Defend alert with a set of suspicious events from Network security devices like Palo Alto
Networks (PANW) and Fortinet Fortigate by host.ip and source.ip. This may indicate that this host is compromised and
triggering multi-datasource alerts.
**Rule type**: esql
**Rule indices**:
**Rule Severity**: high
**Risk Score**: 73
**Runs every**: 10m
**Searches indices from**: `now-60m`
**Maximum alerts per execution**: 100
**References**:
**Tags**:
- Use Case: Threat Detection
- Rule Type: Higher-Order Rule
- Resources: Investigation Guide
- Data Source: Elastic Defend
- Data Source: Fortinet
- Data Source: PAN-OS

**Version**: 4
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Investigation guide


## Triage and analysis


### Investigating Elastic Defend and Network Security Alerts Correlation

This rule correlate any Elastic Defend alert with suspicious events from Network Security datasources like Palo Alto Networks (PANW), Fortinet Fortigate and Suricata by host.ip and source.ip.

### Possible investigation steps

- Review the alert details to identify the specific host and users involved.
- Investiguate the network alerts by destination.ip and message.
- Examine the timeline of the alerts to understand the sequence of events and determine if there is a pattern or progression in the tactics used.
- Correlate the alert data with other logs and telemetry from the host, such as process creation, network connections, and file modifications, to gather additional context.
- Check for any indicators of compromise (IOCs) associated with the alerts, such as suspicious IP addresses, domains, or file hashes, and search for these across the network.
- Assess the impact and scope of the potential compromise by determining if other hosts or systems have similar alerts or related activity.


### False positive analysis

- IP address ranges overlap where the host.ip value from the Elastic Defend alert is unrelated to the source.ip value from the Network Security alert.
- Alerts from routine administrative tasks may trigger multiple alerts. Review and exclude known benign activities such as scheduled software updates or system maintenance.
- Security tools running on the host might generate alerts across different tactics. Identify and exclude alerts from trusted security applications to reduce noise.
- Automated scripts or batch processes can mimic adversarial behavior. Analyze and whitelist these processes if they are verified as non-threatening.
- Frequent alerts from development or testing environments can be misleading. Consider excluding these environments from the rule or applying a different risk score.
- User behavior anomalies, such as accessing multiple systems or applications, might trigger alerts. Implement user behavior baselines to differentiate between normal and suspicious activities.


### Response and remediation

- Isolate the affected host from the network immediately to prevent further lateral movement by the adversary.
- Conduct a thorough forensic analysis of the host to identify the specific vulnerabilities exploited and gather evidence of the attack phases involved.
- Remove any identified malicious software or unauthorized access tools from the host, ensuring all persistence mechanisms are eradicated.
- Apply security patches and updates to the host to address any exploited vulnerabilities and prevent similar attacks.
- Restore the host from a known good backup if necessary, ensuring that the backup is free from compromise.
- Monitor the host and network for any signs of re-infection or further suspicious activity, using enhanced logging and alerting based on the identified attack patterns.
- Escalate the incident to the appropriate internal or external cybersecurity teams for further investigation and potential legal action if the attack is part of a larger campaign.


## Rule Query

```esql
FROM logs-* metadata _id
| WHERE
        // Elastic Defend Alerts
        (event.module == "endpoint" and event.dataset == "endpoint.alerts") or

        // PANW suspicious events
        (event.dataset == "panw.panos" and
         event.action in ("virus_detected", "wildfire_virus_detected", "c2_communication", "spyware_detected", "large_upload", "denied", "exploit_detected")) or

        // Fortigate suspicious events
        (event.dataset == "fortinet_fortigate.log" and
         (event.action in ("outbreak-prevention", "infected", "blocked") or message like "backdoor*" or message like "Proxy*" or message like "anomaly*" or message like "P2P*" or message like "misc*" or message like "DNS.Over.HTTPS" or message like "Remote.Access")) or

        // Suricata
        (event.dataset == "suricata.eve" and message in ("Command and Control Traffic", "Potentially Bad Traffic", "A Network Trojan was detected", "Detection of a Network Scan", "Domain Observed Used for C2 Detected", "Malware Command and Control Activity Detected"))

// extract source.ip from PANW or Fortigate events and host.ip from Elastic Defend alert
|eval fw_alert_source_ip = CASE(event.dataset in ("panw.panos", "fortinet_fortigate.log"), source.ip, null),
      elastic_defend_alert_host_ip = CASE(event.module == "endpoint" and event.dataset == "endpoint.alerts", host.ip, null)
| eval Esql.source_ip = COALESCE(fw_alert_source_ip, elastic_defend_alert_host_ip)
| where Esql.source_ip is not null

// group by host_source_ip shared between FG/PANW and Elastic Defend
| stats Esql.alerts_count = COUNT(*),
        Esql.event_module_distinct_count = COUNT_DISTINCT(event.module),
        Esql.message_values_distinct_count = COUNT_DISTINCT(message),
        Esql.event_module_values = VALUES(event.module),
        Esql.message_values = VALUES(message),
        Esql.event_action_values = VALUES(event.action),
        Esql.process_executable_values = VALUES(process.executable),
        Esql.process_hash_sha256_values = VALUES(process.hash.sha256),
        Esql.process_cmdline_values = VALUES(process.command_line),
        Esql.file_path_values = VALUES(file.path),
        Esql.file_hash_sha256_values = VALUES(file.hash.sha256),
        Esql.host_id_values = VALUES(host.id),
        Esql.user_name_values = VALUES(user.name),
        Esql.destination_ip_values = VALUES(destination.ip)
        by Esql.source_ip
| where Esql.event_module_distinct_count >= 2 AND Esql.message_values_distinct_count >= 2
| eval concat_module_values = MV_CONCAT(Esql.event_module_values, ",")
// Make sure an endpoint alert is present along one of the network ones
| where concat_module_values like "*endpoint*"

// Move single values to their corresponding ECS fields for alerts exclusion
| eval source.ip = mv_min(Esql.source_ip),
       host.id = mv_min(Esql.host_id_values),
       user.name = mv_min(Esql.user_name_values)

| keep source.ip, host.id, user.name, Esql.*
```