﻿---
title: Suspected Lateral Movement from Compromised Host
description: Detects potential lateral movement or post-compromise activity by correlating alerts where the host.ip of one alert matches the source.ip of a subsequent...
url: https://www.elastic.co/docs/reference/security/prebuilt-rules/rules/cross-platform/multiple_alerts_by_host_ip_and_source_ip
products:
  - Elastic Security
---

# Suspected Lateral Movement from Compromised Host
Detects potential lateral movement or post-compromise activity by correlating alerts where the host.ip of one alert matches
the source.ip of a subsequent alert. This behavior may indicate a compromised host being used to authenticate to another system
or resource, including cloud services.
**Rule type**: esql
**Rule indices**:
**Rule Severity**: high
**Risk Score**: 73
**Runs every**: 29m
**Searches indices from**: `now-30m`
**Maximum alerts per execution**: 100
**References**:
**Tags**:
- Use Case: Threat Detection
- Rule Type: Higher-Order Rule
- Resources: Investigation Guide

**Version**: 2
**Rule authors**:
- Elastic

**Rule license**: Elastic License v2

## Investigation guide


## Triage and analysis> **Disclaimer**:
This investigation guide was created using generative AI technology and has been reviewed to improve its accuracy and relevance. While every effort has been made to ensure its quality, we recommend validating the content and adapting it to suit your specific environment and operational needs.

### Investigating Suspected Lateral Movement from Compromised Host

The detection rule uses alert data to determine when multiple alerts from different integrations involving the same user.name are triggered.

### Possible investigation steps

- Review the alert details to identify the specific host involved and the different modules and rules that triggered the alert.
- Examine the timeline of the alerts to understand the sequence of events and determine if there is a pattern or progression in the tactics used.
- Correlate the alert data with other logs and telemetry from the host, such as process creation, network connections, and file modifications, to gather additional context.
- Investigate any known vulnerabilities or misconfigurations on the host that could have been exploited by the adversary.
- Check for any indicators of compromise (IOCs) associated with the alerts, such as suspicious IP addresses, domains, or file hashes, and search for these across the network.
- Assess the impact and scope of the potential compromise by determining if other hosts or systems have similar alerts or related activity.


### False positive analysis

- Vulnerability scanners.
- Jump hosts, NAT gateways and proxies.


### Response and remediation

- Isolate the affected host from the network immediately to prevent further lateral movement by the adversary.
- Conduct a thorough forensic analysis of the host to identify the specific vulnerabilities exploited and gather evidence of the attack phases involved.
- Remove any identified malicious software or unauthorized access tools from the host, ensuring all persistence mechanisms are eradicated.
- Apply security patches and updates to the host to address any exploited vulnerabilities and prevent similar attacks.
- Restore the host from a known good backup if necessary, ensuring that the backup is free from compromise.
- Monitor the host and network for any signs of re-infection or further suspicious activity, using enhanced logging and alerting based on the identified attack patterns.
- Escalate the incident to the appropriate internal or external cybersecurity teams for further investigation and potential legal action if the attack is part of a larger campaign.


## Rule Query

```esql
from .alerts-security.*

// any alerts excluding deprecated, low severity and threat_match rules
| where kibana.alert.rule.name is not null and kibana.alert.risk_score > 21 and
        kibana.alert.rule.type != "threat_match" and
        not kibana.alert.rule.name like "Deprecated - *"

// alerts with existing source.ip or host.ip
| eval alert_source_ip = CASE(source.ip is not null, source.ip, null),
       alert_host_ip = CASE(host.ip is not null and source.ip is null, host.ip, null)

| eval Esql.source_ip = COALESCE(alert_source_ip, alert_host_ip)
| where Esql.source_ip is not null and Esql.source_ip != "127.0.0.1" and Esql.source_ip != "::1"

| stats Esql.alerts_count = COUNT(*),
        Esql.event_module_distinct_count = COUNT_DISTINCT(event.module),
        Esql.host_id_distinct_count = COUNT_DISTINCT(host.id),
        Esql.rule_name_distinct_count = COUNT_DISTINCT(kibana.alert.rule.name),
        Esql.event_module_values = VALUES(event.module),
        Esql.message_values = VALUES(message),
        Esql.rule_name = VALUES(kibana.alert.rule.name),
        Esql.event_action_values = VALUES(event.action),
        Esql.event_category_values = VALUES(event.category),
        Esql.process_executable_values = VALUES(process.executable),
        Esql.process_cmdline_values = VALUES(process.command_line),
        Esql.file_path_values = VALUES(file.path),
        Esql.host_id_values = VALUES(host.id),
        Esql.host_ip_values = VALUES(host.ip),
        Esql.destination_ip_values = VALUES(destination.ip),
        Esql.user_name_values = VALUES(user.name),
        SRC_IP = VALUES(source.ip)
        by Esql.source_ip

// filter for different alerts from multiple hosts and where the host.ip of one alert matches the source.ip of the other alert
| eval concat_ip_values = MV_CONCAT(TO_STRING(Esql.host_ip_values), ",")
| eval host_ip_equal_to_source_ip =LOCATE(concat_ip_values, TO_STRING(Esql.source_ip))
| where Esql.rule_name_distinct_count >= 2 and Esql.host_id_distinct_count >= 2 and host_ip_equal_to_source_ip > 0 and SRC_IP is not null and Esql.alerts_count <= 100

// Move single values to their corresponding ECS fields for alerts exclusion
| eval source.ip = mv_min(Esql.source_ip),
       host.id = mv_min(Esql.host_id_values)
| KEEP Esql.*, source.ip, host.id
```