﻿---
title: Using ingest processors in Painless
description: Painless scripts in ingest pipelines can access certain ingest processor functionality through the Processors namespace, enabling custom logic while leveraging...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/using-ingest-processors-in-painless
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Using ingest processors in Painless
Painless scripts in [ingest pipelines](https://www.elastic.co/docs/manage-data/ingest/transform-enrich/ingest-pipelines) can access certain [ingest processor](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-ingest-processor-context) functionality through the `Processors` namespace, enabling custom logic while leveraging Elasticsearch built-in transformations. Scripts execute within the `ctx` context to modify documents during ingestion.
Only a subset of ingest processors expose methods in the `Processors` namespace for use in Painless scripts. The following ingest processors expose methods in Painless:
- Bytes
- Lowercase
- Uppercase
- Json


## When to choose each approach


| Method                                                                                                 | Use for                                                           | Pros                                                                 | Cons                                                                                |
|--------------------------------------------------------------------------------------------------------|-------------------------------------------------------------------|----------------------------------------------------------------------|-------------------------------------------------------------------------------------|
| [script processor](https://www.elastic.co/docs/reference/enrich-processor/script-processor) (Painless) | * complex logic * conditional operations * multi-field validation | * full control * custom business logic * cross-field operations      | * performance overhead * complexity                                                 |
| [ingest processor](https://www.elastic.co/docs/manage-data/ingest/transform-enrich/ingest-pipelines)   | * common transformations * standard operations                    | * optimized performance * built-in validation * simple configuration | * limited logic * single-field focus                                                |
| [runtime fields](https://www.elastic.co/docs/manage-data/data-store/mapping/runtime-fields)            | * query-time calculations * schema flexibility                    | * no reindexing required * dynamic computation during queries        | * query-time performance cost * read-only operations * not used in ingest pipeline. |

**Performance considerations:** Script processors can impact pipeline performance. Prefer ingest processors for simple transformations.

## Method usage

All ingest methods available in Painless are scoped to the `Processors` namespace. For example:
```json

{
  "pipeline": {
    "processors": [
      {
        "script": {
          "lang": "painless",
          "source": """
            long bytes = Processors.bytes(ctx.size);
            ctx.size_in_bytes = bytes;
          """
        }
      }
    ]
  },
  "docs": [
    {
      "_source": {
        "size": "1kb"
      }
    }
  ]
}
```


## Ingest methods reference


### Byte conversion

Use the [bytes processor](https://www.elastic.co/docs/reference/enrich-processor/bytes-processor) to return the number of bytes in the human-readable byte value supplied in the `value` parameter.
```java
long bytes(String value);
```


### Lowercase conversion

Use the [lowercase processor](https://www.elastic.co/docs/reference/enrich-processor/lowercase-processor) to convert the supplied string in the `value` parameter to its lowercase equivalent.
```java
String lowercase(String value);
```


### Uppercase conversion

Use the [uppercase processor](https://www.elastic.co/docs/reference/enrich-processor/uppercase-processor) to convert the supplied string in the `value` parameter to its uppercase equivalent.
```java
String uppercase(String value);
```


### JSON parsing

Use the [JSON processor](https://www.elastic.co/docs/reference/enrich-processor/json-processor) to parse a string containing JSON data into a structured object, string, or other value. There are two `json` methods:
```java
void json(Map<String, Object> map, String key);
Object json(Object value);
```

The first `json` method accepts a map and a key. The processor parses the JSON string in the given map at the given key to a structured object. The entries in that object are added directly to the given map.
For example, if the input document looks like this:
```js
{
  "foo": {
    "inputJsonString": "{\"bar\": 999}"
  }
}
```

then executing this script:
```java
Processors.json(ctx.foo, 'inputJsonString');
```

will result in this document:
```js
{
  "foo": {
    "inputJsonString": "{\"bar\": 999}",
    "bar" : 999
  }
}
```

The second `json` method accepts a JSON string in the `value` parameter and returns a structured object or other value.
You can then add this object to the document through the context object:
```java
ctx.parsedJson = Processors.json(ctx.inputJsonString);
```


### URL decoding

Use the [URL decode processor](https://www.elastic.co/docs/reference/enrich-processor/urldecode-processor) to URL-decode the string supplied in the `value` parameter.
```java
String urlDecode(String value);
```


### URI decomposition

Use the [URI parts processor](https://www.elastic.co/docs/reference/enrich-processor/uri-parts-processor) to decompose the URI string supplied in the `value` parameter. Returns a map of key-value pairs in which the key is the name of the URI component such as `domain` or `path` and the value is the corresponding value for that component.
```java
String uriParts(String value);
```


### Network community ID

Use the [community ID processor](https://www.elastic.co/docs/reference/enrich-processor/community-id-processor) to compute the network community ID for network flow data.
```java
String communityId(String sourceIpAddrString, String destIpAddrString, Object ianaNumber, Object transport, Object sourcePort, Object destinationPort, Object icmpType, Object icmpCode, int seed)
String communityId(String sourceIpAddrString, String destIpAddrString, Object ianaNumber, Object transport, Object sourcePort, Object destinationPort, Object icmpType, Object icmpCode)
```