﻿---
title: Weight context
description: Use a Painless script to create a weight for use in a similarity script. The weight makes up the part of the similarity calculation that is independent...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-weight-context
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Weight context
Use a Painless script to create a [weight](https://www.elastic.co/docs/reference/elasticsearch/index-settings/similarity) for use in a [similarity script](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-similarity-context). The weight makes up the part of the similarity calculation that is independent of the document being scored, and so can be built up front and cached.
Queries that contain multiple terms calculate a separate weight for each term.

## Variables

<definitions>
  <definition term="query.boost (float, read-only)">
    The boost value if provided by the query. If this is not provided the value is `1.0f`.
  </definition>
  <definition term="field.docCount (long, read-only)">
    The number of documents that have a value for the current field.
  </definition>
  <definition term="field.sumDocFreq (long, read-only)">
    The sum of all terms that exist for the current field. If this is not available the value is `-1`.
  </definition>
  <definition term="field.sumTotalTermFreq (long, read-only)">
    The sum of occurrences in the index for all the terms that exist in the current field. If this is not available the value is `-1`.
  </definition>
  <definition term="term.docFreq (long, read-only)">
    The number of documents that contain the current term in the index.
  </definition>
  <definition term="term.totalTermFreq (long, read-only)">
    The total occurrences of the current term in the index.
  </definition>
</definitions>


## Return

<definitions>
  <definition term="double">
    A scoring factor used across all documents.
  </definition>
</definitions>


## API

The standard [Painless API](https://www.elastic.co/guide/en/elasticsearch/painless/current/painless-api-reference-shared.html) is available.

## Example

To run the example, first [install the eCommerce sample data](/docs/reference/scripting-languages/painless/painless-context-examples#painless-sample-data-install).
This request creates an index with a smart search system that automatically boosts products with rare, distinctive features like "vintage", "handcrafted", or unique materials. It helps discover special items by making uncommon products rank higher in search results.
```json
PUT kibana_sample_data_ecommerce-weight
{
  "settings": {
    "similarity": {
      "rare_term_weight": {
        "type": "scripted",
        "weight_script": {
          "source": """
            double idf = Math.log((field.docCount + 1.0) / (term.docFreq + 1.0));
            
            double rarityFactor = 1.0;
            if (term.docFreq < field.docCount * 0.02) {
              rarityFactor = 3.0; 
            } else if (term.docFreq < field.docCount * 0.10) {
              rarityFactor = 1.5; 
            }
            
            return query.boost * idf * rarityFactor;
          """
        },
        "script": {
          "source": """
            double tf = Math.sqrt(doc.freq);

            return tf * weight;
          """
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "product_name": {
        "type": "text",
        "similarity": "rare_term_weight"
      },
      "description": {
        "type": "text", 
        "similarity": "rare_term_weight"
      }
    }
  }
}
```