﻿---
title: Update by query context
description: Use a Painless script in an update by query operation to add, modify, or delete fields within each of a set of documents collected as the result of query...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-update-by-query-context
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Update by query context
Use a Painless script in an [update by query](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-update-by-query) operation to add, modify, or delete fields within each of a set of documents collected as the result of query.
Check out the [document update tutorial](https://www.elastic.co/docs/explore-analyze/scripting/modules-scripting-document-update-tutorial) for related examples.

## Variables

<definitions>
  <definition term="params (Map, read-only)">
    User-defined parameters passed in as part of the query.
  </definition>
  <definition term="ctx['op'] (String)">
    The name of the operation.
  </definition>
  <definition term="ctx['_routing'] (String, read-only)">
    The value used to select a shard for document storage.
  </definition>
  <definition term="ctx['_index'] (String, read-only)">
    The name of the index.
  </definition>
  <definition term="ctx['_id'] (String, read-only)">
    The unique document id.
  </definition>
  <definition term="ctx['_version'] (int, read-only)">
    The current version of the document.
  </definition>
  <definition term="ctx['_source'] (Map)">
    Contains extracted JSON in a `Map` and `List` structure for the fields existing in a stored document.
  </definition>
</definitions>


## Side Effects

<definitions>
  <definition term="ctx['op']">
    Use the default of `index` to update a document. Set to `none` to specify no operation or `delete` to delete the current document from the index.
  </definition>
  <definition term="ctx['_source']">
    Modify the values in the `Map/List` structure to add, modify, or delete the fields of a document.
  </definition>
</definitions>


## Return

<definitions>
  <definition term="void">
    No expected return value.
  </definition>
</definitions>


## API

The standard [Painless API](https://www.elastic.co/guide/en/elasticsearch/painless/current/painless-api-reference-shared.html) is available.

## Example

To run the example, first [install the eCommerce sample data](/docs/reference/scripting-languages/painless/painless-context-examples#painless-sample-data-install).
The following request uses the `_update_by_query` API to apply a discount to products that meet two conditions:
1. The product’s `base_price` is greater than or equal to 20.
2. The product currently has a `discount_percentage` of 0 (no discount applied).

```json
POST /kibana_sample_data_ecommerce/_update_by_query
{
  "query": {
    "bool": {
      "filter": [
        {
          "range": {
            "products.base_price": {
              "gte": 20
            }
          }
        },
        {
          "match": {
            "products.discount_percentage": 0
          }
        }
      ]
    }
  },
  "script": {
    "lang": "painless",
    "source": """
        for (product in ctx._source.products) { 
            // If product has no discount applied 
            if (product.discount_percentage == 0) { 
                product.discount_percentage = params.discount_rate; 
                product.discount_amount = product.base_price * (params.discount_rate / 100) ;

                product.price = product.base_price - product.discount_amount;
            } 
        }
    """,
    "params": {
      "discount_rate": 15
    }
  }
}
```