﻿---
title: Reindex context
description: Use a Painless script in a reindex operation to add, modify, or delete fields within each document in an original index as its reindexed into a target...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-reindex-context
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Reindex context
Use a Painless script in a [reindex](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-reindex) operation to add, modify, or delete fields within each document in an original index as its reindexed into a target index.

## Variables

<definitions>
  <definition term="params (Map, read-only)">
    User-defined parameters passed in as part of the query.
  </definition>
  <definition term="ctx['op'] (String)">
    The name of the operation.
  </definition>
  <definition term="ctx['_routing'] (String)">
    The value used to select a shard for document storage.
  </definition>
  <definition term="ctx['_index'] (String)">
    The name of the index.
  </definition>
  <definition term="ctx['_id'] (String)">
    The unique document id.
  </definition>
  <definition term="ctx['_version'] (int)">
    The current version of the document.
  </definition>
  <definition term="ctx['_source'] (Map)">
    Contains extracted JSON in a `Map` and `List` structure for the fields existing in a stored document.
  </definition>
</definitions>


## Side Effects

<definitions>
  <definition term="ctx['op']">
    Use the default of `index` to update a document. Set to `noop` to specify no operation or `delete` to delete the current document from the index.
  </definition>
  <definition term="ctx['_routing']">
    Modify this to change the routing value for the current document.
  </definition>
  <definition term="ctx['_index']">
    Modify this to change the destination index for the current document.
  </definition>
  <definition term="ctx['_id']">
    Modify this to change the id for the current document.
  </definition>
  <definition term="ctx['_version'] (int)">
    Modify this to modify the version for the current document.
  </definition>
  <definition term="ctx['_source']">
    Modify the values in the `Map/List` structure to add, modify, or delete the fields of a document.
  </definition>
</definitions>


## Return

<definitions>
  <definition term="void">
    No expected return value.
  </definition>
</definitions>


## API

The standard [Painless API](https://www.elastic.co/guide/en/elasticsearch/painless/current/painless-api-reference-shared.html) is available.

## Example

To run the example, first [install the eCommerce sample data](/docs/reference/scripting-languages/painless/painless-context-examples#painless-sample-data-install).
The following request copies all documents from the `kibana_sample_data_ecommerce` index to a new index called `data_ecommerce_usd`. During this process, a script is used to convert all monetary values to a new currency using a specified exchange rate in the `params`.
```json
POST /_reindex
{
  "source": {
    "index": "kibana_sample_data_ecommerce"
  },
  "dest": {
    "index": "data_ecommerce_usd"
  },
  "script": {
    "source": """
      float er = (float) params.exchange_rate;
      ctx._source.currency = params.target_currency;

      // Convert all prices to dollar
      ctx._source.taxful_total_price = Math.round(ctx._source.taxful_total_price * er * 100) / 100;
      ctx._source.taxless_total_price = Math.round(ctx._source.taxless_total_price * er * 100) / 100;
      
      for (product in ctx._source.products) {
        product.price = Math.round(product.price * er * 100) / 100;
        product.base_price = Math.round(product.base_price * er * 100) / 100;
        product.taxful_price = Math.round(product.taxful_price * er * 100) / 100;
        product.taxless_price = Math.round(product.taxless_price * er * 100) / 100;
      }
    """,
    "lang": "painless",
    "params": {
      "exchange_rate": 1.10,
      "target_currency": "USD"
    }
  }
}
```