﻿---
title: Operators: Reference
description: Use the method call operator '()' to call a member method on a reference type value. Implicit boxing/unboxing is evaluated as necessary per argument during...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-operators-reference
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Operators: Reference
## Method call

Use the `method call operator '()'` to call a member method on a [reference type](/docs/reference/scripting-languages/painless/painless-types#reference-types) value. Implicit [boxing/unboxing](/docs/reference/scripting-languages/painless/painless-casting#boxing-unboxing) is evaluated as necessary per argument during the method call. When a method call is made on a target `def` type value, the parameters and return type value are considered to also be of the `def` type and are evaluated at run-time.
An overloaded method is one that shares the same name with two or more methods. A method is overloaded based on arity where the same name is re-used for multiple methods as long as the number of parameters differs.
**Errors**
- If the reference type value is `null`.
- If the member method name doesn’t exist for a given reference type value.
- If the number of arguments passed in is different from the number of specified parameters.
- If the arguments cannot be implicitly cast or implicitly boxed/unboxed to the correct type values for the parameters.

**Grammar**
```text
method_call: '.' ID arguments;
arguments: '(' (expression (',' expression)*)? ')';
```

**Examples**
- Method calls on different reference types
  ```java
  Map m = new HashMap();                         
  m.put(1, 2);                                   
  int z = m.get(1);                              
  def d = new ArrayList();                       
  d.add(1);                                      
  int i = Integer.parseInt(d.get(0).toString()); 
  ```


## Field access

Use the `field access operator '.'` to store a value to or load a value from a [reference type](/docs/reference/scripting-languages/painless/painless-types#reference-types) member field.
**Errors**
- If the reference type value is `null`.
- If the member field name doesn’t exist for a given reference type value.

**Grammar**
```text
field_access: '.' ID;
```

**Examples**
The examples use the following reference type definition:
```java
name:
  Example

non-static member fields:
  * int x
  * def y
  * List z
```

- Field access with the `Example` type
  ```java
  Example example = new Example(); 
  example.x = 1;                   
  example.y = example.x;           
  example.z = new ArrayList();     
  example.z.add(1);                
  example.x = example.z.get(0);    
  ```


## Null safe

Use the `null safe operator '?.'` instead of the method call operator or field access operator to ensure a reference type value is `non-null` before a method call or field access. A `null` value will be returned if the reference type value is `null`, otherwise the method call or field access is evaluated.
**Errors**
- If the method call return type value or the field access type value is not a reference type value and is not implicitly castable to a reference type value.

**Grammar**
```text
null_safe: null_safe_method_call
         | null_safe_field_access
         ;

null_safe_method_call: '?.' ID arguments;
arguments: '(' (expression (',' expression)*)? ')';

null_safe_field_access: '?.' ID;
```

**Examples**
The examples use the following reference type definition:
```java
name:
  Example

non-static member methods:
  * List factory()

non-static member fields:
  * List x
```

- Null safe without a `null` value
  ```java
  Example example = new Example(); 
  List x = example?.factory();     
  ```
- Null safe with a `null` value
  ```java
  Example example = null; 
  List x = example?.x;    
  ```


## List initialization

Use the `list initialization operator '[]'` to allocate an `List` type instance to the heap with a set of pre-defined values. Each value used to initialize the `List` type instance is cast to a `def` type value upon insertion into the `List` type instance using the `add` method. The order of the specified values is maintained.
**Grammar**
```text
list_initialization: '[' expression (',' expression)* ']'
                   | '[' ']';
```

**Examples**
- List initialization of an empty `List` type value
  ```java
  List empty = []; 
  ```
- List initialization with static values
  ```java
  List list = [1, 2, 3]; 
  ```
- List initialization with non-static values
  ```java
  int i = 1;                  
  long l = 2L;                
  float f = 3.0F;             
  double d = 4.0;             
  String s = "5";             
  List list = [i, l, f*d, s]; 
  ```


## List access

Use the `list access operator '[]'` as a shortcut for a `set` method call or `get` method call made on a `List` type value.
**Errors**
- If a value other than a `List` type value is accessed.
- If a non-integer type value is used as an index for a `set` method call or `get` method call.

**Grammar**
```text
list_access: '[' expression ']'
```

**Examples**
- List access with the `List` type
  ```java
  List list = new ArrayList(); 
  list.add(1);                 
  list.add(2);                 
  list.add(3);                 
  list[0] = 2;                 
  list[1] = 5;                 
  int x = list[0] + list[1];   
  int y = 1;                   
  int z = list[y];             
  ```
- List access with the `def` type
  ```java
  def d = new ArrayList(); 
  d.add(1);                
  d.add(2);                
  d.add(3);                
  d[0] = 2;                
  d[1] = 5;                
  def x = d[0] + d[1];     
  def y = 1;               
  def z = d[y];            
  ```


## Map initialization

Use the `map initialization operator '[:]'` to allocate a `Map` type instance to the heap with a set of pre-defined values. Each pair of values used to initialize the `Map` type instance are cast to `def` type values upon insertion into the `Map` type instance using the `put` method.
**Grammar**
```text
map_initialization: '[' key_pair (',' key_pair)* ']'
                  | '[' ':' ']';
key_pair: expression ':' expression
```

**Examples**
- Map initialization of an empty `Map` type value
  ```java
  Map empty = [:]; 
  ```
- Map initialization with static values
  ```java
  Map map = [1:2, 3:4, 5:6]; 
  ```
- Map initialization with non-static values
  ```java
  byte b = 0;                  
  int i = 1;                   
  long l = 2L;                 
  float f = 3.0F;              
  double d = 4.0;              
  String s = "5";              
  Map map = [b:i, l:f*d, d:s]; 
  ```


## Map access

Use the `map access operator '[]'` as a shortcut for a `put` method call or `get` method call made on a `Map` type value.
**Errors**
- If a value other than a `Map` type value is accessed.

**Grammar**
```text
map_access: '[' expression ']'
```

**Examples**
- Map access with the `Map` type
  ```java
  Map map = new HashMap();               
  map['value2'] = 2;                     
  map['value5'] = 5;                     
  int x = map['value2'] + map['value5']; 
  String y = 'value5';                   
  int z = x[z];                          
  ```
- Map access with the `def` type
  ```java
  def d = new HashMap();             
  d['value2'] = 2;                   
  d['value5'] = 5;                   
  int x = d['value2'] + d['value5']; 
  String y = 'value5';               
  def z = d[y];                      
  ```


## New instance

Use the `new instance operator 'new ()'` to allocate a [reference type](/docs/reference/scripting-languages/painless/painless-types#reference-types) instance to the heap and call a specified constructor. Implicit [boxing/unboxing](/docs/reference/scripting-languages/painless/painless-casting#boxing-unboxing) is evaluated as necessary per argument during the constructor call.
An overloaded constructor is one that shares the same name with two or more constructors. A constructor is overloaded based on arity where the same reference type name is re-used for multiple constructors as long as the number of parameters differs.
**Errors**
- If the reference type name doesn’t exist for instance allocation.
- If the number of arguments passed in is different from the number of specified parameters.
- If the arguments cannot be implicitly cast or implicitly boxed/unboxed to the correct type values for the parameters.

**Grammar**
```text
new_instance: 'new' TYPE '(' (expression (',' expression)*)? ')';
```

**Examples**
- Allocation of new instances with different types

```java
Map m = new HashMap();   
def d = new ArrayList(); 
def e = new HashMap(m);  
```


## String concatenation

Use the `string concatenation operator '+'` to concatenate two values together where at least one of the values is a [`String` type](/docs/reference/scripting-languages/painless/painless-types#string-type).
**Grammar**
```text
concatenate: expression '+' expression;
```

**Examples**
- String concatenation with different primitive types
  ```java
  String x = "con";     
  String y = x + "cat"; 
  String z = 4 + 5 + x; 
  ```
- String concatenation with the `def` type.
  ```java
  def d = 2;             
  d = "con" + d + "cat"; 
  ```


## Elvis

An elvis consists of two expressions. The first expression is evaluated with to check for a `null` value. If the first expression evaluates to `null` then the second expression is evaluated and its value used. If the first expression evaluates to `non-null` then the resultant value of the first expression is used. Use the `elvis operator '?:'` as a shortcut for the conditional operator.
**Errors**
- If the first expression or second expression cannot produce a `null` value.

**Grammar**
```text
elvis: expression '?:' expression;
```

**Examples**
- Elvis with different reference types
  ```java
  List x = new ArrayList();      
  List y = x ?: new ArrayList(); 
  y = null;                      
  List z = y ?: new ArrayList(); 
  ```