﻿---
title: Operators: Numeric
description: Use the post increment operator '++' to INCREASE the value of a numeric type variable/field by 1. An extra implicit cast is necessary to return the promoted...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-operators-numeric
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Operators: Numeric
## Post increment

Use the `post increment operator '++'` to INCREASE the value of a numeric type variable/field by `1`. An extra implicit cast is necessary to return the promoted numeric type value to the original numeric type value of the variable/field for the following types: `byte`, `short`, and `char`. If a variable/field is read as part of an expression the value is loaded prior to the increment.
**Errors**
- If the variable/field is a non-numeric type.

**Grammar**
```text
post_increment: ( variable | field ) '++';
```

**Promotion**

| original | promoted | implicit |
|----------|----------|----------|
| byte     | int      | byte     |
| short    | int      | short    |
| char     | int      | char     |
| int      | int      |          |
| long     | long     |          |
| float    | float    |          |
| double   | double   |          |
| def      | def      |          |

**Examples**
- Post increment with different numeric types
  ```java
  short i = 0; 
  i++;         
  long j = 1;  
  long k;      
  k = j++;     
  ```
- Post increment with the `def` type
  ```java
  def x = 1; 
  x++;       
  ```


## Post decrement

Use the `post decrement operator '--'` to DECREASE the value of a numeric type variable/field by `1`. An extra implicit cast is necessary to return the promoted numeric type value to the original numeric type value of the variable/field for the following types: `byte`, `short`, and `char`. If a variable/field is read as part of an expression the value is loaded prior to the decrement.
**Errors**
- If the variable/field is a non-numeric type.

**Grammar**
```text
post_decrement: ( variable | field ) '--';
```

**Promotion**

| original | promoted | implicit |
|----------|----------|----------|
| byte     | int      | byte     |
| short    | int      | short    |
| char     | int      | char     |
| int      | int      |          |
| long     | long     |          |
| float    | float    |          |
| double   | double   |          |
| def      | def      |          |

**Examples**
- Post decrement with different numeric types
  ```java
  short i = 0; 
  i--;         
  long j = 1;  
  long k;      
  k = j--;     
  ```
- Post decrement with the `def` type
  ```java
  def x = 1; 
  x--;       
  ```


## Pre increment

Use the `pre increment operator '++'` to INCREASE the value of a numeric type variable/field by `1`. An extra implicit cast is necessary to return the promoted numeric type value to the original numeric type value of the variable/field for the following types: `byte`, `short`, and `char`. If a variable/field is read as part of an expression the value is loaded after the increment.
**Errors**
- If the variable/field is a non-numeric type.

**Grammar**
```text
pre_increment: '++' ( variable | field );
```

**Promotion**

| original | promoted | implicit |
|----------|----------|----------|
| byte     | int      | byte     |
| short    | int      | short    |
| char     | int      | char     |
| int      | int      |          |
| long     | long     |          |
| float    | float    |          |
| double   | double   |          |
| def      | def      |          |

**Examples**
- Pre increment with different numeric types
  ```java
  short i = 0; 
  ++i;         
  long j = 1;  
  long k;      
  k = ++j;     
  ```
- Pre increment with the `def` type
  ```java
  def x = 1; 
  ++x;       
  ```


## Pre decrement

Use the `pre decrement operator '--'` to DECREASE the value of a numeric type variable/field by `1`. An extra implicit cast is necessary to return the promoted numeric type value to the original numeric type value of the variable/field for the following types: `byte`, `short`, and `char`. If a variable/field is read as part of an expression the value is loaded after the decrement.
**Errors**
- If the variable/field is a non-numeric type.

**Grammar**
```text
pre_decrement: '--' ( variable | field );
```

**Promotion**

| original | promoted | implicit |
|----------|----------|----------|
| byte     | int      | byte     |
| short    | int      | short    |
| char     | int      | char     |
| int      | int      |          |
| long     | long     |          |
| float    | float    |          |
| double   | double   |          |
| def      | def      |          |

**Examples**
- Pre decrement with different numeric types
  ```java
  short i = 0; 
  --i;         
  long j = 1;  
  long k;      
  k = --j;     
  ```
- Pre decrement operator with the `def` type
  ```java
  def x = 1; 
  --x;       
  ```


## Unary positive

Use the `unary positive operator '+'` to the preserve the IDENTITY of a numeric type value.
**Errors**
- If the value is a non-numeric type.

**Grammar**
```text
unary_positive: '+' expression;
```

**Examples**
- Unary positive with different numeric types
  ```java
  int x = +1;  
  long y = +x; 
  ```
- Unary positive with the `def` type
  ```java
  def z = +1; 
  int i = +z; 
  ```


## Unary negative

Use the `unary negative operator '-'` to NEGATE a numeric type value.
**Errors**
- If the value is a non-numeric type.

**Grammar**
```text
unary_negative: '-' expression;
```

**Examples**
- Unary negative with different numeric types
  ```java
  int x = -1;  
  long y = -x; 
  ```
- Unary negative with the `def` type
  ```java
  def z = -1; 
  int i = -z; 
  ```


## Bitwise not

Use the `bitwise not operator '~'` to NOT each bit in an integer type value where a `1-bit` is flipped to a resultant `0-bit` and a `0-bit` is flipped to a resultant `1-bit`.
**Errors**
- If the value is a non-integer type.

**Bits**

| original | result |
|----------|--------|
| 1        | 0      |
| 0        | 1      |

**Grammar**
```text
bitwise_not: '~' expression;
```

**Promotion**

| original | promoted |
|----------|----------|
| byte     | int      |
| short    | int      |
| char     | int      |
| int      | int      |
| long     | long     |
| def      | def      |

**Examples**
- Bitwise not with different numeric types
  ```java
  byte b = 1;  
  int i = ~b;  
  long l = ~i; 
  ```
- Bitwise not with the `def` type
  ```java
  def d = 1;  
  def e = ~d; 
  ```


## Multiplication

Use the `multiplication operator '*'` to MULTIPLY together two numeric type values. Rules for resultant overflow and NaN values follow the JVM specification.
**Errors**
- If either of the values is a non-numeric type.

**Grammar**
```text
multiplication: expression '*' expression;
```

**Promotion**

|        |        |        |        |        |        |        |        |     |
|--------|--------|--------|--------|--------|--------|--------|--------|-----|
|        | byte   | short  | char   | int    | long   | float  | double | def |
| byte   | int    | int    | int    | int    | long   | float  | double | def |
| short  | int    | int    | int    | int    | long   | float  | double | def |
| char   | int    | int    | int    | int    | long   | float  | double | def |
| int    | int    | int    | int    | int    | long   | float  | double | def |
| long   | long   | long   | long   | long   | long   | float  | double | def |
| float  | float  | float  | float  | float  | float  | float  | double | def |
| double | double | double | double | double | double | double | double | def |
| def    | def    | def    | def    | def    | def    | def    | def    | def |

**Examples**
- Multiplication with different numeric types
  ```java
  int i = 5*4;      
  double d = i*7.0; 
  ```
- Multiplication with the `def` type
  ```java
  def x = 5*4; 
  def y = x*2; 
  ```


## Division

Use the `division operator '/'` to DIVIDE one numeric type value by another. Rules for NaN values and division by zero follow the JVM specification. Division with integer values drops the remainder of the resultant value.
**Errors**
- If either of the values is a non-numeric type.
- If a left-hand side integer type value is divided by a right-hand side integer type value of `0`.

**Grammar**
```text
division: expression '/' expression;
```

**Promotion**

|        |        |        |        |        |        |        |        |     |
|--------|--------|--------|--------|--------|--------|--------|--------|-----|
|        | byte   | short  | char   | int    | long   | float  | double | def |
| byte   | int    | int    | int    | int    | long   | float  | double | def |
| short  | int    | int    | int    | int    | long   | float  | double | def |
| char   | int    | int    | int    | int    | long   | float  | double | def |
| int    | int    | int    | int    | int    | long   | float  | double | def |
| long   | long   | long   | long   | long   | long   | float  | double | def |
| float  | float  | float  | float  | float  | float  | float  | double | def |
| double | double | double | double | double | double | double | double | def |
| def    | def    | def    | def    | def    | def    | def    | def    | def |

**Examples**
- Division with different numeric types
  ```java
  int i = 29/4;     
  double d = i/7.0; 
  ```
- Division with the `def` type
  ```java
  def x = 5/4; 
  def y = x/2; 
  ```


## Remainder

Use the `remainder operator '%'` to calculate the REMAINDER for division between two numeric type values. Rules for NaN values and division by zero follow the JVM specification.
**Errors**
- If either of the values is a non-numeric type.

**Grammar**
```text
remainder: expression '%' expression;
```

**Promotion**

|        |        |        |        |        |        |        |        |     |
|--------|--------|--------|--------|--------|--------|--------|--------|-----|
|        | byte   | short  | char   | int    | long   | float  | double | def |
| byte   | int    | int    | int    | int    | long   | float  | double | def |
| short  | int    | int    | int    | int    | long   | float  | double | def |
| char   | int    | int    | int    | int    | long   | float  | double | def |
| int    | int    | int    | int    | int    | long   | float  | double | def |
| long   | long   | long   | long   | long   | long   | float  | double | def |
| float  | float  | float  | float  | float  | float  | float  | double | def |
| double | double | double | double | double | double | double | double | def |
| def    | def    | def    | def    | def    | def    | def    | def    | def |

**Examples**
- Remainder with different numeric types
  ```java
  int i = 29%4;     
  double d = i%7.0; 
  ```
- Remainder with the `def` type
  ```java
  def x = 5%4; 
  def y = x%2; 
  ```


## Addition

Use the `addition operator '+'` to ADD together two numeric type values. Rules for resultant overflow and NaN values follow the JVM specification.
**Errors**
- If either of the values is a non-numeric type.

**Grammar**
```text
addition: expression '+' expression;
```

**Promotion**

|        |        |        |        |        |        |        |        |     |
|--------|--------|--------|--------|--------|--------|--------|--------|-----|
|        | byte   | short  | char   | int    | long   | float  | double | def |
| byte   | int    | int    | int    | int    | long   | float  | double | def |
| short  | int    | int    | int    | int    | long   | float  | double | def |
| char   | int    | int    | int    | int    | long   | float  | double | def |
| int    | int    | int    | int    | int    | long   | float  | double | def |
| long   | long   | long   | long   | long   | long   | float  | double | def |
| float  | float  | float  | float  | float  | float  | float  | double | def |
| double | double | double | double | double | double | double | double | def |
| def    | def    | def    | def    | def    | def    | def    | def    | def |

**Examples**
- Addition operator with different numeric types
  ```java
  int i = 29+4;     
  double d = i+7.0; 
  ```
- Addition with the `def` type
  ```java
  def x = 5+4; 
  def y = x+2; 
  ```


## Subtraction

Use the `subtraction operator '-'` to SUBTRACT a right-hand side numeric type value from a left-hand side numeric type value. Rules for resultant overflow and NaN values follow the JVM specification.
**Errors**
- If either of the values is a non-numeric type.

**Grammar**
```text
subtraction: expression '-' expression;
```

**Promotion**

|        |        |        |        |        |        |        |        |     |
|--------|--------|--------|--------|--------|--------|--------|--------|-----|
|        | byte   | short  | char   | int    | long   | float  | double | def |
| byte   | int    | int    | int    | int    | long   | float  | double | def |
| short  | int    | int    | int    | int    | long   | float  | double | def |
| char   | int    | int    | int    | int    | long   | float  | double | def |
| int    | int    | int    | int    | int    | long   | float  | double | def |
| long   | long   | long   | long   | long   | long   | float  | double | def |
| float  | float  | float  | float  | float  | float  | float  | double | def |
| double | double | double | double | double | double | double | double | def |
| def    | def    | def    | def    | def    | def    | def    | def    | def |

**Examples**
- Subtraction with different numeric types
  ```java
  int i = 29-4;     
  double d = i-7.5; 
  ```
- Subtraction with the `def` type
  ```java
  def x = 5-4; 
  def y = x-2; 
  ```


## Left shift

Use the `left shift operator '<<'` to SHIFT lower order bits to higher order bits in a left-hand side integer type value by the distance specified in a right-hand side integer type value.
**Errors**
- If either of the values is a non-integer type.
- If the right-hand side value cannot be cast to an int type.

**Grammar**
```text
left_shift: expression '<<' expression;
```

**Promotion**
The left-hand side integer type value is promoted as specified in the table below. The right-hand side integer type value is always implicitly cast to an `int` type value and truncated to the number of bits of the promoted type value.

| original | promoted |
|----------|----------|
| byte     | int      |
| short    | int      |
| char     | int      |
| int      | int      |
| long     | long     |
| def      | def      |

**Examples**
- Left shift with different integer types
  ```java
  int i = 4 << 1;   
  long l = i << 2L; 
  ```
- Left shift with the `def` type
  ```java
  def x = 4 << 2; 
  def y = x << 1; 
  ```


## Right shift

Use the `right shift operator '>>'` to SHIFT higher order bits to lower order bits in a left-hand side integer type value by the distance specified in a right-hand side integer type value. The highest order bit of the left-hand side integer type value is preserved.
**Errors**
- If either of the values is a non-integer type.
- If the right-hand side value cannot be cast to an int type.

**Grammar**
```text
right_shift: expression '>>' expression;
```

**Promotion**
The left-hand side integer type value is promoted as specified in the table below. The right-hand side integer type value is always implicitly cast to an `int` type value and truncated to the number of bits of the promoted type value.

| original | promoted |
|----------|----------|
| byte     | int      |
| short    | int      |
| char     | int      |
| int      | int      |
| long     | long     |
| def      | def      |

**Examples**
- Right shift with different integer types
  ```java
  int i = 32 >> 1;  
  long l = i >> 2L; 
  ```
- Right shift with the `def` type
  ```java
  def x = 16 >> 2; 
  def y = x >> 1;  
  ```


## Unsigned right shift

Use the `unsigned right shift operator '>>>'` to SHIFT higher order bits to lower order bits in a left-hand side integer type value by the distance specified in a right-hand side type integer value. The highest order bit of the left-hand side integer type value is **not** preserved.
**Errors**
- If either of the values is a non-integer type.
- If the right-hand side value cannot be cast to an int type.

**Grammar**
```text
unsigned_right_shift: expression '>>>' expression;
```

**Promotion**
The left-hand side integer type value is promoted as specified in the table below. The right-hand side integer type value is always implicitly cast to an `int` type value and truncated to the number of bits of the promoted type value.

| original | promoted |
|----------|----------|
| byte     | int      |
| short    | int      |
| char     | int      |
| int      | int      |
| long     | long     |
| def      | def      |

**Examples**
- Unsigned right shift with different integer types
  ```java
  int i = -1 >>> 29; 
  long l = i >>> 2L; 
  ```
- Unsigned right shift with the `def` type
  ```java
  def x = 16 >>> 2; 
  def y = x >>> 1;  
  ```


## Bitwise and

Use the `bitwise and operator '&'` to AND together each bit within two integer type values where if both bits at the same index are `1` the resultant bit is `1` and `0` otherwise.
**Errors**
- If either of the values is a non-integer type.

**Bits**

|     |     |     |
|-----|-----|-----|
|     | 1   | 0   |
| 1   | 1   | 0   |
| 0   | 0   | 0   |

**Grammar**
```text
bitwise_and: expression '&' expression;
```

**Promotion**

|       |      |       |      |      |      |     |
|-------|------|-------|------|------|------|-----|
|       | byte | short | char | int  | long | def |
| byte  | int  | int   | int  | int  | long | def |
| short | int  | int   | int  | int  | long | def |
| char  | int  | int   | int  | int  | long | def |
| int   | int  | int   | int  | int  | long | def |
| long  | long | long  | long | long | long | def |
| def   | def  | def   | def  | def  | def  | def |

**Examples**
- Bitwise and with different integer types
  ```java
  int i = 5 & 6;   
  long l = i & 5L; 
  ```
- Bitwise and with the `def` type
  ```java
  def x = 15 & 6; 
  def y = x & 5;  
  ```


## Bitwise xor

Use the `bitwise xor operator '^'` to XOR together each bit within two integer type values where if one bit is a `1` and the other bit is a `0` at the same index the resultant bit is `1` otherwise the resultant bit is `0`.
**Errors**
- If either of the values is a non-integer type

**Bits**
The following table illustrates the resultant bit from the xoring of two bits.

|     |     |     |
|-----|-----|-----|
|     | 1   | 0   |
| 1   | 0   | 1   |
| 0   | 1   | 0   |

**Grammar**
```text
bitwise_xor: expression '^' expression;
```

**Promotion**

|       |      |       |      |      |      |     |
|-------|------|-------|------|------|------|-----|
|       | byte | short | char | int  | long | def |
| byte  | int  | int   | int  | int  | long | def |
| short | int  | int   | int  | int  | long | def |
| char  | int  | int   | int  | int  | long | def |
| int   | int  | int   | int  | int  | long | def |
| long  | long | long  | long | long | long | def |
| def   | def  | def   | def  | def  | def  | def |

**Examples**
- Bitwise xor with different integer types
  ```java
  int i = 5 ^ 6;   
  long l = i ^ 5L; 
  ```
- Bitwise xor with the `def` type
  ```java
  def x = 15 ^ 6; 
  def y = x ^ 5;  
  ```


## Bitwise or

Use the `bitwise or operator '|'` to OR together each bit within two integer type values where if at least one bit is a `1` at the same index the resultant bit is `1` otherwise the resultant bit is `0`.
**Errors**
- If either of the values is a non-integer type.

**Bits**
The following table illustrates the resultant bit from the oring of two bits.

|     |     |     |
|-----|-----|-----|
|     | 1   | 0   |
| 1   | 1   | 1   |
| 0   | 1   | 0   |

**Grammar**
```text
bitwise_or: expression '|' expression;
```

**Promotion**

|       |      |       |      |      |      |     |
|-------|------|-------|------|------|------|-----|
|       | byte | short | char | int  | long | def |
| byte  | int  | int   | int  | int  | long | def |
| short | int  | int   | int  | int  | long | def |
| char  | int  | int   | int  | int  | long | def |
| int   | int  | int   | int  | int  | long | def |
| long  | long | long  | long | long | long | def |
| def   | def  | def   | def  | def  | def  | def |

**Examples**
- Bitwise or with different integer types
  ```java
  int i = 5 | 6;   
  long l = i | 8L; 
  ```
- Bitwise or with the `def` type
  ```java
  def x = 5 ^ 6; 
  def y = x ^ 8; 
  ```