﻿---
title: Operators: General
description: Use the precedence operator '()' to guarantee the order of evaluation for an expression. An expression encapsulated by the precedence operator (enclosed...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-operators-general
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Operators: General
## Precedence

Use the `precedence operator '()'` to guarantee the order of evaluation for an expression. An expression encapsulated by the precedence operator (enclosed in parentheses) overrides existing precedence relationships between operators and is evaluated prior to other expressions in inward-to-outward order.
**Grammar**
```text
precedence: '(' expression ')';
```

**Examples**
- Precedence with numeric operators
  ```java
  int x = (5+4)*6;   
  int y = 12/(x-50); 
  ```


## Function call

Use the `function call operator ()` to call an existing function. A [function call](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-functions) is defined within a script.
**Grammar**
```text
function_call: ID '(' ( expression (',' expression)* )? ')'';
```

**Examples**
- A function call
  ```java
  int add(int x, int y) { 
        return x + y;
    }

  int z = add(1, 2); 
  ```


## Cast

An explicit cast converts the value of an original type to the equivalent value of a target type forcefully as an operation. Use the `cast operator '()'` to specify an explicit cast. Refer to [casting](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-casting) for more information.

## Conditional

A conditional consists of three expressions. The first expression is evaluated with an expected boolean result type. If the first expression evaluates to true then the second expression will be evaluated. If the first expression evaluates to false then the third expression will be evaluated. The second and third expressions will be [promoted](/docs/reference/scripting-languages/painless/painless-casting#promotion) if the evaluated values are not the same type. Use the `conditional operator '? :'` as a shortcut to avoid the need for a full if/else branch in certain expressions.
**Errors**
- If the first expression does not evaluate to a boolean type value.
- If the values for the second and third expressions cannot be promoted.

**Grammar**
```text
conditional: expression '?' expression ':' expression;
```

**Promotion**

|           |        |        |        |        |        |        |        |           |     |
|-----------|--------|--------|--------|--------|--------|--------|--------|-----------|-----|
|           | byte   | short  | char   | int    | long   | float  | double | Reference | def |
| byte      | int    | int    | int    | int    | long   | float  | double | -         | def |
| short     | int    | int    | int    | int    | long   | float  | double | -         | def |
| char      | int    | int    | int    | int    | long   | float  | double | -         | def |
| int       | int    | int    | int    | int    | long   | float  | double | -         | def |
| long      | long   | long   | long   | long   | long   | float  | double | -         | def |
| float     | float  | float  | float  | float  | float  | float  | double | -         | def |
| double    | double | double | double | double | double | double | double | -         | def |
| Reference | -      | -      | -      | -      | -      | -      | -      | Object @  | def |
| def       | def    | def    | def    | def    | def    | def    | def    | def       | def |

@ If the two reference type values are the same then this promotion will not occur.
**Examples**
- Evaluation of conditionals
  ```java
  boolean b = true;
  int x = b ? 1 : 2;	
  ```
  ```java
  int x = 1;
  List y = x > 1 ? new ArrayList() : null; 
  ```
  ```java
  int x = 1;
  def z = x < 2 ? x : 2.0;	
  ```


## Assignment

Use the `assignment operator '='` to store a value in a variable or reference type member field for use in subsequent operations. Any operation that produces a value can be assigned to any variable/field as long as the [types](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-types) are the same or the resultant type can be [implicitly cast](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-casting) to the variable/field type.
See [variable assignment](/docs/reference/scripting-languages/painless/painless-variables#variable-assignment) for examples using variables.
**Errors**
- If the type of value is unable to match the type of variable or field.

**Grammar**
```text
assignment: field '=' expression
```

**Examples**
The examples use the following reference type definition:
```java
name:
  Example

non-static member fields:
  * int x
  * def y
  * List z
```

- Field assignments of different type values
  ```java
  Example example = new Example(); 
  example.x = 1;                   
  example.y = 2.0;                 
  example.z = new ArrayList();     
  ```
- A field assignment from a field access
  ```java
  Example example = new Example(); 
  example.x = 1;                   
  example.y = example.x;           
  ```


## Compound assignment

Use the `compound assignment operator '$='` as a shortcut for an assignment where a binary operation would occur between the variable/field as the left-hand side expression and a separate right-hand side expression.
A compound assignment is equivalent to the expression below where V is the variable/field and T is the type of variable/member.
```java
V = (T)(V op expression);
```

**Operators**
The table below shows the available operators for use in a compound assignment. Each operator follows the casting/promotion rules according to their regular definition. For numeric operations there is an extra implicit cast when necessary to return the promoted numeric type value to the original numeric type value of the variable/field and can result in data loss.

|                      |                 |
|----------------------|-----------------|
| Operator             | Compound Symbol |
| Multiplication       | *=              |
| Division             | /=              |
| Remainder            | %=              |
| Addition             | +=              |
| Subtraction          | -=              |
| Left Shift           | <<=             |
| Right Shift          | >>=             |
| Unsigned Right Shift | >>>=            |
| Bitwise And          | &=              |
| Boolean And          | &=              |
| Bitwise Xor          | ^=              |
| Boolean Xor          | ^=              |
| Bitwise Or           | =               |
| Boolean Or           | =               |
| String Concatenation | +=              |

**Errors**
- If the type of value is unable to match the type of variable or field.

**Grammar**
```text
compound_assignment: ( ID | field ) '$=' expression;
```

The use of the `$=` represents the use of any of the possible binary operators.
**Examples**
- Compound assignment for each numeric operator
  ```java
  int i = 10; 
  i *= 2;     
  i /= 5;     
  i %= 3;     
  i += 5;     
  i -= 5;     
  i <<= 2;    
  i >>= 1;    
  i >>>= 1;   
  i &= 15;    
  i ^= 12;    
  i |= 2;     
  ```
- Compound assignment for each boolean operator
  ```java
  boolean b = true; 
  b &= false;       
  b ^= false;       
  b |= true;        
  ```
- A compound assignment with the string concatenation operator
  ```java
  String s = 'compound'; 
  s += ' assignment';    
  ```
- A compound assignment with the `def` type
  ```java
  def x = 1; 
  x += 2;    
  ```
- A compound assignment with an extra implicit cast
  ```java
  byte b = 1; 
  b += 2;     
  ```