﻿---
title: Datetime now
description: Under most Painless contexts the current datetime, now, is not supported. There are two primary reasons for this. The first is that scripts are often...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-datetime-now
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Datetime now
Under most Painless contexts the current datetime, `now`, is not supported. There are two primary reasons for this. The first is that scripts are often run once per document, so each time the script is run a different `now` is returned. The second is that scripts are often run in a distributed fashion without a way to appropriately synchronize `now`. Instead, pass in a user-defined parameter with either a string datetime or numeric datetime for `now`. A numeric datetime is preferred as there is no need to parse it for comparison.

## Datetime now examples

- Use a numeric datetime as `now`
  - Assumptions:
  - The field `input_datetime` exists in all indexes as part of the query
- All indexed documents contain the field `input_datetime`
- Mappings:
  ```JSON
  {
    "mappings": {
        ...
        "properties": {
          ...
          "input_datetime": {
          "type": "date"
        }
        ...
      }
      ...
    }
  }
  ```
  
- Input:
  ```JSON
  ...
  "script": {
      ...
      "params": {
          "now": <generated numeric datetime in milliseconds since epoch>
      }
  }
  ...
  ```
  
- Script:
  ```java
  long now = params['now'];
  ZonedDateTime inputDateTime = doc['input_datetime'];
  long millisDateTime = inputDateTime.toInstant().toEpochMilli();
  long elapsedTime = now - millisDateTime;
  ```
  
- Use a string datetime as `now`
  - Assumptions:
  - The field `input_datetime` exists in all indexes as part of the query
- All indexed documents contain the field `input_datetime`
- Mappings:
  ```JSON
  {
    "mappings": {
      ...
      "properties": {
        ...
        "input_datetime": {
          "type": "date"
        }
        ...
      }
      ...
    }
  }
  ```
  
- Input:
  ```JSON
  ...
  "script": {
      ...
      "params": {
          "now": "<generated string datetime in ISO-8601>"
      }
  }
  ...
  ```
  
- Script:
  ```java
  String nowString = params['now'];
  ZonedDateTime nowZdt = ZonedDateTime.parse(nowString); 
  long now = ZonedDateTime.toInstant().toEpochMilli();
  ZonedDateTime inputDateTime = doc['input_datetime'];
  long millisDateTime = zdt.toInstant().toEpochMilli();
  long elapsedTime = now - millisDateTime;
  ```