﻿---
title: Datetime Input
description: There are several common ways datetimes are used as input for a script determined by the Painless context. Typically, datetime input will be accessed...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-datetime-input
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Datetime Input
There are several common ways datetimes are used as input for a script determined by the [Painless context](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-contexts). Typically, datetime input will be accessed from parameters specified by the user, from an original source document, or from an indexed document.

## Datetime input from user parameters

Use the [params section](https://www.elastic.co/docs/explore-analyze/scripting/modules-scripting-using) during script specification to pass in a numeric datetime or string datetime as a script input. Access to user-defined parameters within a script is dependent on the Painless context, though, the parameters are most commonly accessible through an input called `params`.
**Examples**
- Parse a numeric datetime from user parameters to a complex datetime
  - Input:
  ```JSON
  ...
  "script": {
      ...
      "params": {
          "input_datetime": 434931327000
      }
  }
  ...
  ```
- Script:
  ```java
  long inputDateTime = params['input_datetime'];
  Instant instant = Instant.ofEpochMilli(inputDateTime);
  ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, ZoneId.of('Z'));
  ```
- Parse a string datetime from user parameters to a complex datetime
  - Input:
  ```JSON
  ...
  "script": {
      ...
      "params": {
          "input_datetime": "custom y 1983 m 10 d 13 22:15:30 Z"
      }
  }
  ...
  ```
- Script:
  ```java
  String datetime = params['input_datetime'];
  DateTimeFormatter dtf = DateTimeFormatter.ofPattern(
          "'custom' 'y' yyyy 'm' MM 'd' dd HH:mm:ss VV");
  ZonedDateTime zdt = ZonedDateTime.parse(datetime, dtf); 
  ```


## Datetime input from a source document

Use an original [source](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-source-field) document as a script input to access a numeric datetime or string datetime for a specific field within that document. Access to an original source document within a script is dependent on the Painless context and is not always available. An original source document is most commonly accessible through an input called `ctx['_source']` or `params['_source']`.
**Examples**
- Parse a numeric datetime from a sourced document to a complex datetime
  - Input:
  ```JSON
  {
    ...
    "input_datetime": 434931327000
    ...
  }
  ```
- Script:
  ```java
  long inputDateTime = ctx['_source']['input_datetime']; 
  Instant instant = Instant.ofEpochMilli(inputDateTime);
  ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, ZoneId.of('Z'));
  ```
- Parse a string datetime from a sourced document to a complex datetime
  - Input:
  ```JSON
  {
    ...
    "input_datetime": "1983-10-13T22:15:30Z"
    ...
  }
  ```
- Script:
  ```java
  String datetime = params['_source']['input_datetime']; 
  ZonedDateTime zdt = ZonedDateTime.parse(datetime); 
  ```


## Datetime input from an indexed document

Use an indexed document as a script input to access a complex datetime for a specific field within that document where the field is mapped as a [standard date](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date) or a [nanosecond date](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date_nanos). Numeric datetime fields mapped as [numeric](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/number) and string datetime fields mapped as [keyword](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/keyword) are accessible through an indexed document as well. Access to an indexed document within a script is dependent on the Painless context and is not always available. An indexed document is most commonly accessible through an input called `doc`.
**Examples**
- Format a complex datetime from an indexed document to a string datetime
  - Assumptions:
  - The field `input_datetime` exists in all indexes as part of the query
- All indexed documents contain the field `input_datetime`
- Mappings:
  ```JSON
  {
    "mappings": {
      ...
      "properties": {
        ...
        "input_datetime": {
          "type": "date"
        }
        ...
      }
      ...
    }
  }
  ```
- Script:
  ```java
  ZonedDateTime input = doc['input_datetime'].value;
  String output = input.format(DateTimeFormatter.ISO_INSTANT); 
  ```
- Find the difference between two complex datetimes from an indexed document
  - Assumptions:
  - The fields `start` and `end` may **not** exist in all indexes as part of the query
- The fields `start` and `end` may **not** have values in all indexed documents
- Mappings:
  ```JSON
  {
    "mappings": {
      ...
      "properties": {
        ...
        "start": {
          "type": "date"
        },
        "end": {
          "type": "date"
        }
        ...
      }
      ...
    }
  }
  ```
- Script:
  ```java
  if (doc.containsKey('start') && doc.containsKey('end')) { 

      if (doc['start'].size() > 0 && doc['end'].size() > 0) { 

          ZonedDateTime start = doc['start'].value;
          ZonedDateTime end = doc['end'].value;
          long differenceInMillis = ChronoUnit.MILLIS.between(start, end);

          // handle difference in times
      } else {
          // handle fields without values
      }
  } else {
      // handle index with missing fields
  }
  ```