﻿---
title: Datetime examples in contexts
description: Try out these Painless datetime examples that include real world contexts. Run the following curl commands to load the data necessary for the context...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-datetime-examples-in-contexts
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Datetime examples in contexts
Try out these Painless datetime examples that include real world contexts.

## Load the example data

Run the following curl commands to load the data necessary for the context examples into an Elasticsearch cluster:
1. Create [mappings](https://www.elastic.co/docs/manage-data/data-store/mapping) for the sample data.
   ```json

   {
     "mappings": {
       "properties": {
         "priority": {
           "type": "integer"
         },
         "datetime": {
           "type": "date"
         },
         "message": {
           "type": "text"
         }
       }
     }
   }
   ```
2. Load the sample data.
   ```json

   { "index" : { "_index" : "messages", "_id" : "1" } }
   { "priority": 1, "datetime": "2019-07-17T12:13:14Z", "message": "m1" }
   { "index" : { "_index" : "messages", "_id" : "2" } }
   { "priority": 1, "datetime": "2019-07-24T01:14:59Z", "message": "m2" }
   { "index" : { "_index" : "messages", "_id" : "3" } }
   { "priority": 2, "datetime": "1983-10-14T00:36:42Z", "message": "m3" }
   { "index" : { "_index" : "messages", "_id" : "4" } }
   { "priority": 3, "datetime": "1983-10-10T02:15:15Z", "message": "m4" }
   { "index" : { "_index" : "messages", "_id" : "5" } }
   { "priority": 3, "datetime": "1983-10-10T17:18:19Z", "message": "m5" }
   { "index" : { "_index" : "messages", "_id" : "6" } }
   { "priority": 1, "datetime": "2019-08-03T17:19:31Z", "message": "m6" }
   { "index" : { "_index" : "messages", "_id" : "7" } }
   { "priority": 3, "datetime": "2019-08-04T17:20:00Z", "message": "m7" }
   { "index" : { "_index" : "messages", "_id" : "8" } }
   { "priority": 2, "datetime": "2019-08-04T18:01:01Z", "message": "m8" }
   { "index" : { "_index" : "messages", "_id" : "9" } }
   { "priority": 3, "datetime": "1983-10-10T19:00:45Z", "message": "m9" }
   { "index" : { "_index" : "messages", "_id" : "10" } }
   { "priority": 2, "datetime": "2019-07-23T23:39:54Z", "message": "m10" }
   ```
   


## Day-of-the-week bucket aggregation example

The following example uses a [terms aggregation](/docs/reference/aggregations/search-aggregations-bucket-terms-aggregation#search-aggregations-bucket-terms-aggregation-script) as part of the [bucket script aggregation context](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-bucket-script-agg-context) to display the number of messages from each day-of-the-week.
```json

{
  "aggs": {
    "day-of-week-count": {
      "terms": {
        "script": "return doc[\"datetime\"].value.getDayOfWeekEnum();"
      }
    }
  }
}
```


## Morning/evening bucket aggregation example

The following example uses a [terms aggregation](/docs/reference/aggregations/search-aggregations-bucket-terms-aggregation#search-aggregations-bucket-terms-aggregation-script) as part of the [bucket script aggregation context](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-bucket-script-agg-context) to display the number of messages received in the morning versus the evening.
```json

{
  "aggs": {
    "am-pm-count": {
      "terms": {
        "script": "return doc[\"datetime\"].value.getHour() < 12 ? \"AM\" : \"PM\";"
      }
    }
  }
}
```


## Age of a message script field example

The following example uses a [script field](/docs/reference/elasticsearch/rest-apis/retrieve-selected-fields#script-fields) as part of the [field context](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-field-context) to display the elapsed time between "now" and when a message was received.
```json

{
  "query": {
    "match_all": {}
  },
  "script_fields": {
    "message_age": {
      "script": {
        "source": "ZonedDateTime now = ZonedDateTime.ofInstant(Instant.ofEpochMilli(params[\"now\"]), ZoneId.of(\"Z\")); ZonedDateTime mdt = doc[\"datetime\"].value; String age; long years = mdt.until(now, ChronoUnit.YEARS); age = years + \"Y \"; mdt = mdt.plusYears(years); long months = mdt.until(now, ChronoUnit.MONTHS); age += months + \"M \"; mdt = mdt.plusMonths(months); long days = mdt.until(now, ChronoUnit.DAYS); age += days + \"D \"; mdt = mdt.plusDays(days); long hours = mdt.until(now, ChronoUnit.HOURS); age += hours + \"h \"; mdt = mdt.plusHours(hours); long minutes = mdt.until(now, ChronoUnit.MINUTES); age += minutes + \"m \"; mdt = mdt.plusMinutes(minutes); long seconds = mdt.until(now, ChronoUnit.SECONDS); age += hours + \"s\"; return age;",
        "params": {
          "now": 1574005645830
        }
      }
    }
  }
}
```

The following shows the script broken into multiple lines:
```java
ZonedDateTime now = ZonedDateTime.ofInstant(
        Instant.ofEpochMilli(params['now']), ZoneId.of('Z')); 
ZonedDateTime mdt = doc['datetime'].value; 

String age;

long years = mdt.until(now, ChronoUnit.YEARS); 
age = years + 'Y '; 
mdt = mdt.plusYears(years); 

long months = mdt.until(now, ChronoUnit.MONTHS);
age += months + 'M ';
mdt = mdt.plusMonths(months);

long days = mdt.until(now, ChronoUnit.DAYS);
age += days + 'D ';
mdt = mdt.plusDays(days);

long hours = mdt.until(now, ChronoUnit.HOURS);
age += hours + 'h ';
mdt = mdt.plusHours(hours);

long minutes = mdt.until(now, ChronoUnit.MINUTES);
age += minutes + 'm ';
mdt = mdt.plusMinutes(minutes);

long seconds = mdt.until(now, ChronoUnit.SECONDS);
age += hours + 's';

return age; 
```