﻿---
title: Casting
description: A cast converts the value of an original type to the equivalent value of a target type. An implicit cast infers the target type and automatically occurs...
url: https://www.elastic.co/docs/reference/scripting-languages/painless/painless-casting
products:
  - Elasticsearch
  - Painless
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Casting
A cast converts the value of an original type to the equivalent value of a target type. An implicit cast infers the target type and automatically occurs during certain [operations](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-operators). An explicit cast specifies the target type and forcefully occurs as its own operation. Use the `cast operator '()'` to specify an explicit cast. You can also check out the [type casting tutorial](https://www.elastic.co/docs/explore-analyze/scripting/modules-scripting-type-casting-tutorial) for related examples, and for help with troubleshooting refer to "type casting issues" in the Troubleshooting section.
Refer to the [cast table](#allowed-casts) for a quick reference on all allowed casts. To help resolve any issues, check [Debug type casting errors in Painless](https://www.elastic.co/docs/explore-analyze/scripting/painless-type-casting-issues).
**Errors**
- If during a cast there exists no equivalent value for the target type.
- If an implicit cast is given, but an explicit cast is required.

**Grammar**
```text
cast: '(' TYPE ')' expression
```

**Examples**
- Valid casts
  ```java
  int i = (int)5L;         
  Map m = new HashMap();   
  HashMap hm = (HashMap)m; 
  ```


## Numeric Type Casting

A [numeric type](/docs/reference/scripting-languages/painless/painless-types#primitive-types) cast converts the value of an original numeric type to the equivalent value of a target numeric type. A cast between two numeric type values results in data loss when the value of the original numeric type is larger than the target numeric type can accommodate. A cast between an integer type value and a floating point type value can result in precision loss.
The allowed casts for values of each numeric type are shown as a row in the following table:

|        |          |          |          |          |          |          |          |
|--------|----------|----------|----------|----------|----------|----------|----------|
|        | byte     | short    | char     | int      | long     | float    | double   |
| byte   |          | implicit | implicit | implicit | implicit | implicit | implicit |
| short  | explicit |          | explicit | implicit | implicit | implicit | implicit |
| char   | explicit | explicit |          | implicit | implicit | implicit | implicit |
| int    | explicit | explicit | explicit |          | implicit | implicit | implicit |
| long   | explicit | explicit | explicit | explicit |          | implicit | implicit |
| float  | explicit | explicit | explicit | explicit | explicit |          | implicit |
| double | explicit | explicit | explicit | explicit | explicit | explicit |          |

**Examples**
- Valid numeric type casts
  ```java
  int a = 1;            
  long b = a;           
  short c = (short)b;   
  double e = (double)a; 
  ```
- Invalid numeric type casts resulting in errors
  ```java
  int a = 1.0;
  int b = 2;            
  byte c = b; 
  ```


## Reference Type Casting

A [reference type](/docs/reference/scripting-languages/painless/painless-types#reference-types) cast converts the value of an original reference type to the equivalent value of a target reference type. An implicit cast between two reference type values is allowed when the original reference type is a descendant of the target type. An explicit cast between two reference type values is allowed when the original type is a descendant of the target type or the target type is a descendant of the original type.
**Examples**
- Valid reference type casts
  ```java
  List x;                        
  ArrayList y = new ArrayList(); 
  x = y;                         
  y = (ArrayList)x;              
  x = (List)y;                   
  ```
- Invalid reference type casts resulting in errors
  ```java
  List x = new ArrayList();          
  ArrayList y = x;         
  Map m = (Map)x;          
  ```


## Dynamic Type Casting

A [dynamic (`def`) type](/docs/reference/scripting-languages/painless/painless-types#dynamic-types) cast converts the value of an original `def` type to the equivalent value of any target type or converts the value of any original type to the equivalent value of a target `def` type.
An implicit cast from any original type value to a `def` type value is always allowed. An explicit cast from any original type value to a `def` type value is always allowed but never necessary.
An implicit or explicit cast from an original `def` type value to any target type value is allowed if and only if the cast is normally allowed based on the current type value the `def` type value represents.
**Examples**
- Valid dynamic type casts with any original type to a target `def` type
  ```java
  def d0 = 3;               
  d0 = new ArrayList();     
  Object o = new HashMap(); 
  def d1 = o;               
  int i = d1.size();        
  ```
- Valid dynamic type casts with an original `def` type to any target type
  ```java
  def d = 1.0;         
  int i = (int)d;      
  d = 1;               
  float f = d;         
  d = new ArrayList(); 
  List l = d;          
  ```
- Invalid dynamic type casts resulting in errors
  ```java
  def d = 1;                  
  short s = d;      
  d = new HashMap();          
  List l = d;       
  ```


## String to Character Casting

Use the cast operator to convert a [`String` type](/docs/reference/scripting-languages/painless/painless-types#string-type) value into a [`char` type](/docs/reference/scripting-languages/painless/painless-types#primitive-types) value.
**Errors**
- If the `String` type value isn’t one character in length.
- If the `String` type value is `null`.

**Examples**
- Casting string literals into `char` type values
  ```java
  char c = (char)"C"; 
  c = (char)'c';      
  ```
- Casting a `String` reference into a `char` type value
  ```java
  String s = "s";   
  char c = (char)s; 
  ```


## Character to String Casting

Use the cast operator to convert a [`char` type](/docs/reference/scripting-languages/painless/painless-types#primitive-types) value into a [`String` type](/docs/reference/scripting-languages/painless/painless-types#string-type) value.
**Examples**
- Casting a `String` reference into a `char` type value
  ```java
  char c = 65;          
  String s = (String)c; 
  ```


## Boxing and Unboxing

Boxing is a special type of cast used to convert a primitive type to its corresponding reference type. Unboxing is the reverse used to convert a reference type to its corresponding primitive type.
Implicit boxing/unboxing occurs during the following operations:
- Conversions between a `def` type and a primitive type are implicitly boxed/unboxed as necessary, though this is referred to as an implicit cast throughout the documentation.
- Method/function call arguments are implicitly boxed/unboxed as necessary.
- A primitive type value is implicitly boxed when a reference type method is called on it.

Explicit boxing/unboxing is not allowed. Use the reference type API to explicitly convert a primitive type value to its respective reference type value and vice versa.
**Errors**
- If an explicit cast is made to box/unbox a primitive type.

**Examples**
- Uses of implicit boxing/unboxing
  ```java
  List l = new ArrayList();       
  l.add(1);                       
  Integer I = Integer.valueOf(0); 
  int i = l.get(i);               
  ```
- Uses of invalid boxing/unboxing resulting in errors.
  ```java
  Integer x = 1;                  
  Integer y = (Integer)1;         
  int a = Integer.valueOf(1);     
  int b = (int)Integer.valueOf(1);
  ```


## Promotion

Promotion is when a single value is implicitly cast to a certain type or multiple values are implicitly cast to the same type as required for evaluation by certain operations. Each operation that requires promotion has a promotion table that shows all required implicit casts based on the type(s) of value(s). A value promoted to a `def` type at compile-time is promoted again at run-time based on the type the `def` value represents.
**Errors**
- If a specific operation cannot find an allowed promotion type for the type(s) of value(s) given.

**Examples**
- Uses of promotion
  ```java
  double d = 2 + 2.0; 
  def x = 1;          
  float f = x + 2.0F; 
  ```


## Allowed Casts

The following tables show all allowed casts. Read the tables row by row, where the original type is shown in the first column, and each subsequent column indicates whether a cast to the specified target type is implicit (I), explicit (E), boxed/unboxed for methods only (A), a reference type cast (@), or is not allowed (-). See [reference type casting](#reference-type-casting) for allowed reference type casts.
**Primitive/Reference Types**

|                 |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |
|-----------------|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|
|                 | O   | N   | T   | b   | y   | s   | c   | i   | j   | f   | d   | B   | Y   | S   | C   | I   | J   | F   | D   | R   | def |
| Object    ( O ) |     | @   | @   | -   | -   | -   | -   | -   | -   | -   | -   | @   | @   | @   | @   | @   | @   | @   | @   | @   | I   |
| Number    ( N ) | I   |     | -   | -   | -   | -   | -   | -   | -   | -   | -   | -   | @   | @   | -   | @   | @   | @   | @   | @   | I   |
| String    ( T ) | I   | -   |     | -   | -   | -   | -   | -   | -   | -   | -   | -   | -   | -   | E   | -   | -   | -   | -   | -   | I   |
| boolean   ( b ) | A   | -   | -   |     | -   | -   | -   | -   | -   | -   | -   | A   | -   | -   | -   | -   | -   | -   | -   | -   | I   |
| byte      ( y ) | A   | A   | -   | -   |     | I   | E   | I   | I   | I   | I   | -   | A   | A   | -   | A   | A   | A   | A   | -   | I   |
| short     ( s ) | A   | A   | -   | -   | E   |     | E   | I   | I   | I   | I   | -   | -   | A   | -   | A   | A   | A   | A   | -   | I   |
| char      ( c ) | A   | -   | E   | -   | E   | E   |     | I   | I   | I   | I   | -   | -   | -   | A   | A   | A   | A   | A   | -   | I   |
| int       ( i ) | A   | A   | -   | -   | E   | E   | E   |     | I   | I   | I   | -   | -   | -   | -   | A   | A   | A   | A   | -   | I   |
| long      ( j ) | A   | A   | -   | -   | E   | E   | E   | E   |     | I   | I   | -   | -   | -   | -   | -   | A   | A   | A   | -   | I   |
| float     ( f ) | A   | A   | -   | -   | E   | E   | E   | E   | E   |     | I   | -   | -   | -   | -   | -   | -   | A   | A   | -   | I   |
| double    ( d ) | A   | A   | -   | -   | E   | E   | E   | E   | E   | E   |     | -   | -   | -   | -   | -   | -   | -   | A   | -   | I   |
| Boolean   ( B ) | A   | -   | -   | A   | -   | -   | -   | -   | -   | -   | -   |     | -   | -   | -   | -   | -   | -   | -   | @   | I   |
| Byte      ( Y ) | A   | I   | -   | -   | A   | A   | -   | A   | A   | A   | A   | -   |     | A   | -   | A   | A   | A   | A   | @   | I   |
| Short     ( S ) | A   | I   | -   | -   | -   | A   | -   | A   | A   | A   | A   | -   | -   |     | -   | A   | A   | A   | A   | @   | I   |
| Character ( C ) | A   | -   | -   | -   | -   | -   | A   | A   | A   | A   | A   | -   | -   | -   |     | A   | A   | A   | A   | @   | I   |
| Integer   ( I ) | A   | -   | -   | -   | -   | -   | -   | A   | A   | A   | A   | -   | -   | -   | -   |     | A   | A   | A   | @   | I   |
| Long      ( J ) | A   | -   | -   | -   | -   | -   | -   | -   | A   | A   | A   | -   | -   | -   | -   | -   |     | A   | A   | @   | I   |
| Float     ( F ) | A   | -   | -   | -   | -   | -   | -   | -   | -   | A   | A   | -   | -   | -   | -   | -   | -   |     | A   | @   | I   |
| Double    ( D ) | A   | -   | -   | -   | -   | -   | -   | -   | -   | -   | A   | -   | -   | -   | -   | -   | -   | -   |     | @   | I   |
| Reference ( R ) | I   | @   | @   | -   | -   | -   | -   | -   | -   | -   | -   | @   | @   | @   | @   | @   | @   | @   | @   | @   | I   |

**`def` Type**

|                        |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |     |
|------------------------|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|-----|
|                        | O   | N   | T   | b   | y   | s   | c   | i   | j   | f   | d   | B   | Y   | S   | C   | I   | J   | F   | D   | R   |
| def as String          | I   | -   | I   | -   | -   | -   | E   | -   | -   | -   | -   | -   | -   | -   | E   | -   | -   | -   | -   | @   |
| def as boolean/Boolean | I   | -   | -   | I   | -   | -   | -   | -   | -   | -   | -   | I   | -   | -   | -   | -   | -   | -   | -   | @   |
| def as byte/Byte       | I   | -   | -   | -   | I   | I   | E   | I   | I   | I   | I   | -   | I   | I   | E   | I   | I   | I   | I   | @   |
| def as short/Short     | I   | -   | -   | -   | E   | I   | E   | I   | I   | I   | I   | -   | E   | I   | E   | I   | I   | I   | I   | @   |
| def as char/Character  | I   | -   | -   | -   | E   | E   | I   | I   | I   | I   | I   | -   | E   | E   | I   | I   | I   | I   | I   | @   |
| def as int/Integer     | I   | -   | -   | -   | E   | E   | E   | I   | I   | I   | I   | -   | E   | E   | E   | I   | I   | I   | I   | @   |
| def as long/Long       | I   | -   | -   | -   | E   | E   | E   | E   | I   | I   | I   | -   | E   | E   | E   | E   | I   | I   | I   | @   |
| def as float/Float     | I   | -   | -   | -   | E   | E   | E   | E   | E   | I   | I   | -   | E   | E   | E   | E   | E   | I   | I   | @   |
| def as double/Double   | I   | -   | -   | -   | E   | E   | E   | E   | E   | E   | I   | -   | E   | E   | E   | E   | E   | E   | I   | @   |
| def as Reference       | @   | @   | @   | -   | -   | -   | -   | -   | -   | -   | -   | @   | @   | @   | @   | @   | @   | @   | @   | @   |