﻿---
title: API usage
description: One can use JDBC through the official java.sql and javax.sql packages: The former through java.sql.Driver and DriverManager: Accessible through the javax.sql.DataSource...
url: https://www.elastic.co/docs/reference/query-languages/sql/sql-jdbc-api-usage
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# API usage
One can use JDBC through the official `java.sql` and `javax.sql` packages:

## `java.sql`

The former through `java.sql.Driver` and `DriverManager`:
```java
String address = "jdbc:es://" + elasticsearchAddress;     
Properties connectionProperties = connectionProperties(); 
Connection connection =
    DriverManager.getConnection(address, connectionProperties);
```


## `javax.sql`

Accessible through the `javax.sql.DataSource` API:
```java
EsDataSource dataSource = new EsDataSource();
String address = "jdbc:es://" + elasticsearchAddress;     
dataSource.setUrl(address);
Properties connectionProperties = connectionProperties(); 
dataSource.setProperties(connectionProperties);
Connection connection = dataSource.getConnection();
```

Which one to use? Typically client applications that provide most configuration properties in the URL rely on the `DriverManager`-style while `DataSource` is preferred when being *passed* around since it can be configured in one place and the consumer only has to call `getConnection` without having to worry about any other properties.
To connect to a secured Elasticsearch server the `Properties` should look like:
```java
Properties properties = new Properties();
properties.put("user", "test_admin");
properties.put("password", "x-pack-test-password");
```

Once you have the connection you can use it like any other JDBC connection. For example:
```java
try (Statement statement = connection.createStatement();
        ResultSet results = statement.executeQuery(
              " SELECT name, page_count"
            + "    FROM library"
            + " ORDER BY page_count DESC"
            + " LIMIT 1")) {
    assertTrue(results.next());
    assertEquals("Don Quixote", results.getString(1));
    assertEquals(1072, results.getInt(2));
    SQLException e = expectThrows(SQLException.class, () ->
        results.getInt(1));
    assertThat(e.getMessage(), containsString("Unable to convert "
            + "value [Don Quixote] of type [TEXT] to [Integer]"));
    assertFalse(results.next());
}
```

<note>
  Elasticsearch SQL doesn’t provide a connection pooling mechanism, thus the connections the JDBC driver creates are not pooled. In order to achieve pooled connections, a third-party connection pooling mechanism is required. Configuring and setting up the third-party provider is outside the scope of this documentation.
</note>