﻿---
title: Simple query string query
description: Returns documents based on a provided query string, using a parser with a limited but fault-tolerant syntax. This query uses a simple syntax to parse...
url: https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-simple-query-string-query
products:
  - Elasticsearch
---

# Simple query string query
Returns documents based on a provided query string, using a parser with a limited but fault-tolerant syntax.
This query uses a [simple syntax](#simple-query-string-syntax) to parse and split the provided query string into terms based on special operators. The query then [analyzes](https://www.elastic.co/docs/manage-data/data-store/text-analysis) each term independently before returning matching documents.
While its syntax is more limited than the [`query_string` query](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-query-string-query), the `simple_query_string` query does not return errors for invalid syntax. Instead, it ignores any invalid parts of the query string.

## Example request

```json

{
  "query": {
    "simple_query_string" : {
        "query": "\"fried eggs\" +(eggplant | potato) -frittata",
        "fields": ["title^5", "body"],
        "default_operator": "and"
    }
  }
}
```


## Top-level parameters for `simple_query_string`

<definitions>
  <definition term="query">
    (Required, string) Query string you wish to parse and use for search. See [Simple query string syntax](#simple-query-string-syntax).
  </definition>
  <definition term="fields">
    (Optional, array of strings) Array of fields you wish to search.
  </definition>
</definitions>

This field accepts wildcard expressions. You also can boost relevance scores for matches to particular fields using a caret (`^`) notation. See [Wildcards and per-field boosts in the `fields` parameter](#simple-query-string-boost) for examples.
Defaults to the `index.query.default_field` index setting, which has a default value of `*`. The `*` value extracts all fields that are eligible to term queries and filters the metadata fields. All extracted fields are then combined to build a query if no `prefix` is specified.
<warning>
  There is a limit on the number of fields that can be queried at once. It is defined by the `indices.query.bool.max_clause_count` [search setting](https://www.elastic.co/docs/reference/elasticsearch/configuration-reference/search-settings), which defaults to `1024`.
</warning>

<definitions>
  <definition term="default_operator">
    (Optional, string) Default boolean logic used to interpret text in the query string if no operators are specified. Valid values are:
    - `OR` (Default)
      For example, a query string of `capital of Hungary` is interpreted as `capital OR of OR Hungary`.
    - `AND`
      For example, a query string of `capital of Hungary` is interpreted as `capital AND of AND Hungary`.
  </definition>
  <definition term="analyze_wildcard">
    (Optional, Boolean) If `true`, the query attempts to analyze wildcard terms in the query string. Defaults to `false`. Note that, in case of  `true`, only queries that end with a `*`
    are fully analyzed. Queries that start with `*` or have it in the middle
    are only [normalized](https://www.elastic.co/docs/reference/text-analysis/normalizers).
  </definition>
  <definition term="analyzer">
    (Optional, string) [Analyzer](https://www.elastic.co/docs/manage-data/data-store/text-analysis) used to convert text in the query string into tokens. Defaults to the [index-time analyzer](https://www.elastic.co/docs/manage-data/data-store/text-analysis/specify-an-analyzer#specify-index-time-analyzer) mapped for the `default_field`. If no analyzer is mapped, the index’s default analyzer is used.
  </definition>
  <definition term="auto_generate_synonyms_phrase_query">
    (Optional, Boolean) If `true`, the parser creates a [`match_phrase`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-match-query-phrase) query for each [multi-position token](https://www.elastic.co/docs/manage-data/data-store/text-analysis/token-graphs#token-graphs-multi-position-tokens). Defaults to `true`. For examples, see [Multi-position tokens](#simple-query-string-synonyms).
  </definition>
  <definition term="flags">
    (Optional, string) List of enabled operators for the [simple query string syntax](#simple-query-string-syntax). Defaults to `ALL` (all operators). See [Limit operators](#supported-flags) for valid values.
  </definition>
  <definition term="fuzzy_max_expansions">
    (Optional, integer) Maximum number of terms to which the query expands for fuzzy matching. Defaults to `50`.
  </definition>
  <definition term="fuzzy_prefix_length">
    (Optional, integer) Number of beginning characters left unchanged for fuzzy matching. Defaults to `0`.
  </definition>
  <definition term="fuzzy_transpositions">
    (Optional, Boolean) If `true`, edits for fuzzy matching include transpositions of two adjacent characters (ab → ba). Defaults to `true`.
  </definition>
  <definition term="lenient">
    (Optional, Boolean) If `true`, format-based errors, such as providing a text value for a [numeric](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/number) field, are ignored. Defaults to `false`.
  </definition>
  <definition term="minimum_should_match">
    (Optional, string) Minimum number of clauses that must match for a document to be returned. See the [`minimum_should_match` parameter](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-minimum-should-match) for valid values and more information.
  </definition>
  <definition term="quote_field_suffix">
    (Optional, string) Suffix appended to quoted text in the query string.
  </definition>
</definitions>

You can use this suffix to use a different analysis method for exact matches. See [Mixing exact search with stemming](https://www.elastic.co/docs/solutions/search/full-text/search-relevance/mixing-exact-search-with-stemming).

## Notes


### Simple query string syntax

The `simple_query_string` query supports the following operators:
- `+` signifies AND operation
- `|` signifies OR operation
- `-` negates a single token
- `"` wraps a number of tokens to signify a phrase for searching
- `*` at the end of a term signifies a prefix query
- `(` and `)` signify precedence
- `~N` after a word signifies edit distance (fuzziness)
- `~N` after a phrase signifies slop amount

To use one of these characters literally, escape it with a preceding backslash (`\`).
The behavior of these operators may differ depending on the `default_operator` value. For example:
```json

{
  "query": {
    "simple_query_string": {
      "fields": [ "content" ],
      "query": "foo bar -baz"
    }
  }
}
```

This search is intended to only return documents containing `foo` or `bar` that also do **not** contain `baz`. However because of a `default_operator` of `OR`, this search actually returns documents that contain `foo` or `bar` and any documents that don’t contain `baz`. To return documents as intended, change the query string to `foo bar +-baz`.

### Limit operators

You can use the `flags` parameter to limit the supported operators for the simple query string syntax.
To explicitly enable only specific operators, use a `|` separator. For example, a `flags` value of `OR|AND|PREFIX` disables all operators except `OR`, `AND`, and `PREFIX`.
```json

{
  "query": {
    "simple_query_string": {
      "query": "foo | bar + baz*",
      "flags": "OR|AND|PREFIX"
    }
  }
}
```


#### Valid values

The available flags are:
<definitions>
  <definition term="ALL (Default)">
    Enables all optional operators.
  </definition>
  <definition term="AND">
    Enables the `+` AND operator.
  </definition>
  <definition term="ESCAPE">
    Enables `\` as an escape character.
  </definition>
  <definition term="FUZZY">
    Enables the `~N` operator after a word, where `N` is an integer denoting the allowed edit distance for matching. See [Fuzziness](/docs/reference/elasticsearch/rest-apis/common-options#fuzziness).
  </definition>
  <definition term="NEAR">
    Enables the `~N` operator, after a phrase where `N` is the maximum number of positions allowed between matching tokens. Synonymous to `SLOP`.
  </definition>
  <definition term="NONE">
    Disables all operators.
  </definition>
  <definition term="NOT">
    Enables the `-` NOT operator.
  </definition>
  <definition term="OR">
    Enables the `\|` OR operator.
  </definition>
  <definition term="PHRASE">
    Enables the `"` quotes operator used to search for phrases.
  </definition>
  <definition term="PRECEDENCE">
    Enables the `(` and `)` operators to control operator precedence.
  </definition>
  <definition term="PREFIX">
    Enables the `*` prefix operator.
  </definition>
  <definition term="SLOP">
    Enables the `~N` operator, after a phrase where `N` is maximum number of positions allowed between matching tokens. Synonymous to `NEAR`.
  </definition>
  <definition term="WHITESPACE">
    Enables whitespace as split characters.
  </definition>
</definitions>


### Wildcards and per-field boosts in the `fields` parameter

Fields can be specified with wildcards, eg:
```json

{
  "query": {
    "simple_query_string" : {
      "query":    "Will Smith",
      "fields": [ "title", "*_name" ] <1>
    }
  }
}
```

Individual fields can be boosted with the caret (`^`) notation:
```json

{
  "query": {
    "simple_query_string" : {
      "query" : "this is a test",
      "fields" : [ "subject^3", "message" ] <1>
    }
  }
}
```


### Multi-position tokens

By default, the `simple_query_string` query parser creates a [`match_phrase`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-match-query-phrase) query for each [multi-position token](https://www.elastic.co/docs/manage-data/data-store/text-analysis/token-graphs#token-graphs-multi-position-tokens) in the query string. For example, the parser creates a `match_phrase` query for the multi-word synonym `ny, new york`:
`(ny OR ("new york"))`
To match multi-position tokens with an `AND` conjunction instead, set `auto_generate_synonyms_phrase_query` to `false`:
```json

{
  "query": {
    "simple_query_string": {
      "query": "ny city",
      "auto_generate_synonyms_phrase_query": false
    }
  }
}
```

For the above example, the parser creates the following [`bool`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-bool-query) query:
`(ny OR (new AND york)) city)`
This `bool` query matches documents with the term `ny` or the conjunction `new AND york`.