﻿---
title: Nested query
description: Wraps another query to search nested fields. The nested query searches nested field objects as if they were indexed as separate documents. If an object...
url: https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-nested-query
products:
  - Elasticsearch
---

# Nested query
Wraps another query to search [nested](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/nested) fields.
The `nested` query searches nested field objects as if they were indexed as separate documents. If an object matches the search, the `nested` query returns the root parent document.

## Example request


### Index setup

To use the `nested` query, your index must include a [nested](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/nested) field mapping. For example:
```json

{
  "mappings": {
    "properties": {
      "obj1": {
        "type": "nested"
      }
    }
  }
}
```


### Example query

```json

{
  "query": {
    "nested": {
      "path": "obj1",
      "query": {
        "bool": {
          "must": [
            { "match": { "obj1.name": "blue" } },
            { "range": { "obj1.count": { "gt": 5 } } }
          ]
        }
      },
      "score_mode": "avg"
    }
  }
}
```


## Top-level parameters for `nested`

<definitions>
  <definition term="path">
    (Required, string) Path to the nested object you wish to search.
  </definition>
  <definition term="query">
    (Required, query object) Query you wish to run on nested objects in the `path`. If an object matches the search, the `nested` query returns the root parent document.
  </definition>
</definitions>

You can search nested fields using dot notation that includes the complete path, such as `obj1.name`.
Multi-level nesting is automatically supported, and detected, resulting in an inner nested query to automatically match the relevant nesting level, rather than root, if it exists within another nested query.
See [Multi-level nested queries](#multi-level-nested-query-ex) for an example.
<definitions>
  <definition term="score_mode">
    (Optional, string) Indicates how scores for matching child objects affect the root parent document’s [relevance score](/docs/reference/query-languages/query-dsl/query-filter-context#relevance-scores). Default is `avg`, but nested `knn` queries only support `score_mode=max`. Valid values are:
  </definition>
  <definition term="avg (Default)">
    Use the mean relevance score of all matching child objects.
  </definition>
  <definition term="max">
    Uses the highest relevance score of all matching child objects.
  </definition>
  <definition term="min">
    Uses the lowest relevance score of all matching child objects.
  </definition>
  <definition term="none">
    Do not use the relevance scores of matching child objects. The query assigns parent documents a score of `0`.
  </definition>
  <definition term="sum">
    Add together the relevance scores of all matching child objects.
  </definition>
  <definition term="ignore_unmapped">
    (Optional, Boolean) Indicates whether to ignore an unmapped `path` and not return any documents instead of an error. Defaults to `false`.
  </definition>
</definitions>

If `false`, Elasticsearch returns an error if the `path` is an unmapped field.
You can use this parameter to query multiple indices that may not contain the field `path`.

## Notes


### Context of script queries

If you run a [`script` query](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-script-query) within a nested query, you can only access doc values from the nested document, not the parent or root document.

### Multi-level nested queries

To see how multi-level nested queries work, first you need an index that has nested fields. The following request defines mappings for the `drivers` index with nested `make` and `model` fields.
```json

{
  "mappings": {
    "properties": {
      "driver": {
        "type": "nested",
        "properties": {
          "last_name": {
            "type": "text"
          },
          "vehicle": {
            "type": "nested",
            "properties": {
              "make": {
                "type": "text"
              },
              "model": {
                "type": "text"
              }
            }
          }
        }
      }
    }
  }
}
```

Next, index some documents to the `drivers` index.
```json

{
  "driver" : {
        "last_name" : "McQueen",
        "vehicle" : [
            {
                "make" : "Powell Motors",
                "model" : "Canyonero"
            },
            {
                "make" : "Miller-Meteor",
                "model" : "Ecto-1"
            }
        ]
    }
}


{
  "driver" : {
        "last_name" : "Hudson",
        "vehicle" : [
            {
                "make" : "Mifune",
                "model" : "Mach Five"
            },
            {
                "make" : "Miller-Meteor",
                "model" : "Ecto-1"
            }
        ]
    }
}
```

You can now use a multi-level nested query to match documents based on the `make` and `model` fields.
```json

{
  "query": {
    "nested": {
      "path": "driver",
      "query": {
        "nested": {
          "path": "driver.vehicle",
          "query": {
            "bool": {
              "must": [
                { "match": { "driver.vehicle.make": "Powell Motors" } },
                { "match": { "driver.vehicle.model": "Canyonero" } }
              ]
            }
          }
        }
      }
    }
  }
}
```

The search request returns the following response:
```json
{
  "took" : 5,
  "timed_out" : false,
  "_shards" : {
    "total" : 1,
    "successful" : 1,
    "skipped" : 0,
    "failed" : 0
  },
  "hits" : {
    "total" : {
      "value" : 1,
      "relation" : "eq"
    },
    "max_score" : 3.7349272,
    "hits" : [
      {
        "_index" : "drivers",
        "_id" : "1",
        "_score" : 3.7349272,
        "_source" : {
          "driver" : {
            "last_name" : "McQueen",
            "vehicle" : [
              {
                "make" : "Powell Motors",
                "model" : "Canyonero"
              },
              {
                "make" : "Miller-Meteor",
                "model" : "Ecto-1"
              }
            ]
          }
        }
      }
    ]
  }
}
```


### `must_not` clauses and `nested` queries

If a `nested` query matches one or more nested objects in a document, it returns the document as a hit. This applies even if other nested objects in the document don’t match the query. Keep this in mind when using a `nested` query that contains an inner [`must_not` clause](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-bool-query).
<tip>
  Use the [`inner_hits`](https://www.elastic.co/docs/reference/elasticsearch/rest-apis/retrieve-inner-hits) parameter to see which nested objects matched a `nested` query.
</tip>

For example, the following search uses an outer `nested` query with an inner `must_not` clause.
```json

{
  "mappings": {
    "properties": {
      "comments": {
        "type": "nested"
      }
    }
  }
}


{
  "comments": [
    {
      "author": "kimchy"
    }
  ]
}


{
  "comments": [
    {
      "author": "kimchy"
    },
    {
      "author": "nik9000"
    }
  ]
}


{
  "comments": [
    {
      "author": "nik9000"
    }
  ]
}


{
  "query": {
    "nested": {
      "path": "comments",
      "query": {
        "bool": {
          "must_not": [
            {
              "term": {
                "comments.author": "nik9000"
              }
            }
          ]
        }
      }
    }
  }
}
```

The search returns:
```json
{
  ...
  "hits" : {
    ...
    "hits" : [
      {
        "_index" : "my-index",
        "_id" : "1",
        "_score" : 0.0,
        "_source" : {
          "comments" : [
            {
              "author" : "kimchy"
            }
          ]
        }
      },
      {
        "_index" : "my-index",
        "_id" : "2",
        "_score" : 0.0,
        "_source" : {
          "comments" : [
            {
              "author" : "kimchy"              <1>
            },
            {
              "author" : "nik9000"             <2>
            }
          ]
        }
      }
    ]
  }
}
```

To exclude documents with any nested objects that match the `nested` query, use an outer `must_not` clause.
```json

{
  "query": {
    "bool": {
      "must_not": [
        {
          "nested": {
            "path": "comments",
            "query": {
              "term": {
                "comments.author": "nik9000"
              }
            }
          }
        }
      ]
    }
  }
}
```

The search returns:
```json
{
  ...
  "hits" : {
    ...
    "hits" : [
      {
        "_index" : "my-index",
        "_id" : "1",
        "_score" : 0.0,
        "_source" : {
          "comments" : [
            {
              "author" : "kimchy"
            }
          ]
        }
      }
    ]
  }
}
```