﻿---
title: Distance feature query
description: Boosts the relevance score of documents closer to a provided origin date or point. For example, you can use this query to give more weight to documents...
url: https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-distance-feature-query
products:
  - Elasticsearch
---

# Distance feature query
Boosts the [relevance score](/docs/reference/query-languages/query-dsl/query-filter-context#relevance-scores) of documents closer to a provided `origin` date or point. For example, you can use this query to give more weight to documents closer to a certain date or location.
You can use the `distance_feature` query to find the nearest neighbors to a location. You can also use the query in a [`bool`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-bool-query) search’s `should` filter to add boosted relevance scores to the `bool` query’s scores.

## Example request


### Index setup

To use the `distance_feature` query, your index must include a [`date`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date), [`date_nanos`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date_nanos) or [`geo_point`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/geo-point) field.
To see how you can set up an index for the `distance_feature` query, try the following example.
1. Create an `items` index with the following field mapping:
   - `name`, a [`keyword`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/keyword) field
- `production_date`, a [`date`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date) field
- `location`, a [`geo_point`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/geo-point) field
   ```json

   {
     "mappings": {
       "properties": {
         "name": {
           "type": "keyword"
         },
         "production_date": {
           "type": "date"
         },
         "location": {
           "type": "geo_point"
         }
       }
     }
   }
   ```
   
2. Index several documents to this index.
   ```json

   {
     "name" : "chocolate",
     "production_date": "2018-02-01",
     "location": [-71.34, 41.12]
   }


   {
     "name" : "chocolate",
     "production_date": "2018-01-01",
     "location": [-71.3, 41.15]
   }



   {
     "name" : "chocolate",
     "production_date": "2017-12-01",
     "location": [-71.3, 41.12]
   }
   ```


### Example queries


#### Boost documents based on date

The following `bool` search returns documents with a `name` value of `chocolate`. The search also uses the `distance_feature` query to increase the relevance score of documents with a `production_date` value closer to `now`.
```json

{
  "query": {
    "bool": {
      "must": {
        "match": {
          "name": "chocolate"
        }
      },
      "should": {
        "distance_feature": {
          "field": "production_date",
          "pivot": "7d",
          "origin": "now"
        }
      }
    }
  }
}
```


#### Boost documents based on location

The following `bool` search returns documents with a `name` value of `chocolate`. The search also uses the `distance_feature` query to increase the relevance score of documents with a `location` value closer to `[-71.3, 41.15]`.
```json

{
  "query": {
    "bool": {
      "must": {
        "match": {
          "name": "chocolate"
        }
      },
      "should": {
        "distance_feature": {
          "field": "location",
          "pivot": "1000m",
          "origin": [-71.3, 41.15]
        }
      }
    }
  }
}
```


## Top-level parameters for `distance_feature`

<definitions>
  <definition term="field">
    (Required, string) Name of the field used to calculate distances. This field must meet the following criteria:
    - Be a [`date`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date), [`date_nanos`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date_nanos) or [`geo_point`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/geo-point) field
    - Have an [`index`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-index) mapping parameter value of `true`, which is the default
    - Have an [`doc_values`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/doc-values) mapping parameter value of `true`, which is the default
  </definition>
  <definition term="origin">
    (Required, string) Date or point of origin used to calculate distances.
  </definition>
</definitions>

If the `field` value is a [`date`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date) or [`date_nanos`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date_nanos) field, the `origin` value must be a [date](/docs/reference/aggregations/search-aggregations-bucket-daterange-aggregation#date-format-pattern). [Date Math](/docs/reference/elasticsearch/rest-apis/common-options#date-math), such as `now-1h`, is supported.
If the `field` value is a [`geo_point`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/geo-point) field, the `origin` value must be a geopoint.
<definitions>
  <definition term="pivot">
    (Required, [time unit](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units) or [distance unit](/docs/reference/elasticsearch/rest-apis/api-conventions#distance-units)) Distance from the `origin` at which relevance scores receive half of the `boost` value.
  </definition>
</definitions>

If the `field` value is a [`date`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date) or [`date_nanos`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date_nanos) field, the `pivot` value must be a [time unit](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units), such as `1h` or `10d`.
If the `field` value is a [`geo_point`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/geo-point) field, the `pivot` value must be a [distance unit](/docs/reference/elasticsearch/rest-apis/api-conventions#distance-units), such as `1km` or `12m`.
<definitions>
  <definition term="boost">
    (Optional, float) Floating point number used to multiply the [relevance score](/docs/reference/query-languages/query-dsl/query-filter-context#relevance-scores) of matching documents. This value cannot be negative. Defaults to `1.0`.
  </definition>
</definitions>


## Notes


### How the `distance_feature` query calculates relevance scores

The `distance_feature` query dynamically calculates the distance between the `origin` value and a document’s field values. It then uses this distance as a feature to boost the [relevance score](/docs/reference/query-languages/query-dsl/query-filter-context#relevance-scores) of closer documents.
The `distance_feature` query calculates a document’s [relevance score](/docs/reference/query-languages/query-dsl/query-filter-context#relevance-scores) as follows:
```
relevance score = boost * pivot / (pivot + distance)
```

The `distance` is the absolute difference between the `origin` value and a document’s field value.

### Skip non-competitive hits

Unlike the [`function_score`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-function-score-query) query or other ways to change [relevance scores](/docs/reference/query-languages/query-dsl/query-filter-context#relevance-scores), the `distance_feature` query efficiently skips non-competitive hits when the [`track_total_hits`](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-search) parameter is **not** `true`.