﻿---
title: ES|QL spatial functions
description: ES|QL supports several spatial functions, all of which are aligned with the underlying spatial index grid used in Lucene: 
url: https://www.elastic.co/docs/reference/query-languages/esql/functions-operators/spatial-functions
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# ES|QL spatial functions
ES|QL supports several spatial functions, all of which are
[aligned with the underlying spatial index grid](/docs/reference/query-languages/esql/limitations#esql-limitations-spatial-precision)
used in Lucene:
- Geospatial predicates
  - [`ST_DISTANCE`](#esql-st_distance)
- [`ST_INTERSECTS`](#esql-st_intersects)
- [`ST_DISJOINT`](#esql-st_disjoint)
- [`ST_CONTAINS`](#esql-st_contains)
- [`ST_WITHIN`](#esql-st_within)
- Geometry functions
  - [`ST_X`](#esql-st_x)
- [`ST_Y`](#esql-st_y)
- [`ST_NPOINTS`](#esql-st_npoints) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
- [`ST_SIMPLIFY`](#esql-st_simplify) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
- [`ST_ENVELOPE`](#esql-st_envelope) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
  - [`ST_XMAX`](#esql-st_xmax) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
- [`ST_XMIN`](#esql-st_xmin) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
- [`ST_YMAX`](#esql-st_ymax) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
- [`ST_YMIN`](#esql-st_ymin) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
- Grid encoding functions
  - [`ST_GEOTILE`](#esql-st_geotile) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
- [`ST_GEOHEX`](#esql-st_geohex) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>
- [`ST_GEOHASH`](#esql-st_geohash) <applies-to>Elastic Stack: Preview</applies-to> <applies-to>Elastic Cloud Serverless: Preview</applies-to>


## Geospatial predicates


### `ST_DISTANCE`

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_distance.svg)

**Parameters**
<definitions>
  <definition term="geomA">
    Expression of type `geo_point` or `cartesian_point`. If `null`, the function returns `null`.
  </definition>
  <definition term="geomB">
    Expression of type `geo_point` or `cartesian_point`. If `null`, the function returns `null`. The second parameter must also have the same coordinate system as the first. This means it is not possible to combine `geo_point` and `cartesian_point` parameters.
  </definition>
</definitions>

**Description**
Computes the distance between two points. For cartesian geometries, this is the pythagorean distance in the same units as the original coordinates. For geographic geometries, this is the circular distance along the great circle in meters.
**Supported types**

| geomA           | geomB           | result |
|-----------------|-----------------|--------|
| cartesian_point | cartesian_point | double |
| geo_point       | geo_point       | double |

**Example**
```esql
FROM airports
| WHERE abbrev == "CPH"
| EVAL distance = ST_DISTANCE(location, city_location)
| KEEP abbrev, name, location, city_location, distance
```


| abbrev:k | name:text  | location:geo_point                       | city_location:geo_point | distance:d        |
|----------|------------|------------------------------------------|-------------------------|-------------------|
| CPH      | Copenhagen | POINT(12.6493508684508 55.6285017221528) | POINT(12.5683 55.6761)  | 7339.573896618216 |


### `ST_INTERSECTS`

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_intersects.svg)

**Parameters**
<definitions>
  <definition term="geomA">
    Expression that is either a geometry (`geo_point`, `cartesian_point`, `geo_shape` or `cartesian_shape`) or a geo-grid value (`geohash`, `geotile`, `geohex`). If `null`, the function returns `null`.
  </definition>
  <definition term="geomB">
    Expression that is either a geometry (`geo_point`, `cartesian_point`, `geo_shape` or `cartesian_shape`) or a geo-grid value (`geohash`, `geotile`, `geohex`). If `null`, the function returns `null`. The second parameter must also have the same coordinate system as the first. This means it is not possible to combine `geo_*` and `cartesian_*` parameters.
  </definition>
</definitions>

**Description**
Returns true if two geometries intersect. They intersect if they have any point in common, including their interior points (points along lines or within polygons). This is the inverse of the [ST_DISJOINT](#esql-st_disjoint) function. In mathematical terms: ST_Intersects(A, B) ⇔ A ⋂ B ≠ ∅
**Supported types**

| geomA           | geomB           | result  |
|-----------------|-----------------|---------|
| cartesian_point | cartesian_point | boolean |
| cartesian_point | cartesian_shape | boolean |
| cartesian_shape | cartesian_point | boolean |
| cartesian_shape | cartesian_shape | boolean |
| geo_point       | geo_point       | boolean |
| geo_point       | geo_shape       | boolean |
| geo_point       | geohash         | boolean |
| geo_point       | geohex          | boolean |
| geo_point       | geotile         | boolean |
| geo_shape       | geo_point       | boolean |
| geo_shape       | geo_shape       | boolean |
| geohash         | geo_point       | boolean |
| geohex          | geo_point       | boolean |
| geotile         | geo_point       | boolean |

**Example**
```esql
FROM airports
| WHERE ST_INTERSECTS(location, TO_GEOSHAPE("POLYGON((42 14, 43 14, 43 15, 42 15, 42 14))"))
```


| abbrev:keyword | city:keyword | city_location:geo_point | country:keyword | location:geo_point                     | name:text      | scalerank:i | type:k |
|----------------|--------------|-------------------------|-----------------|----------------------------------------|----------------|-------------|--------|
| HOD            | Al Ḩudaydah  | POINT(42.9511 14.8022)  | Yemen           | POINT(42.97109630194 14.7552534413725) | Hodeidah Int'l | 9           | mid    |


### `ST_DISJOINT`

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_disjoint.svg)

**Parameters**
<definitions>
  <definition term="geomA">
    Expression that is either a geometry (`geo_point`, `cartesian_point`, `geo_shape` or `cartesian_shape`) or a geo-grid value (`geohash`, `geotile`, `geohex`). If `null`, the function returns `null`.
  </definition>
  <definition term="geomB">
    Expression that is either a geometry (`geo_point`, `cartesian_point`, `geo_shape` or `cartesian_shape`) or a geo-grid value (`geohash`, `geotile`, `geohex`). If `null`, the function returns `null`. The second parameter must also have the same coordinate system as the first. This means it is not possible to combine `geo_*` and `cartesian_*` parameters.
  </definition>
</definitions>

**Description**
Returns whether the two geometries or geometry columns are disjoint. This is the inverse of the [ST_INTERSECTS](#esql-st_intersects) function. In mathematical terms: ST_Disjoint(A, B) ⇔ A ⋂ B = ∅
**Supported types**

| geomA           | geomB           | result  |
|-----------------|-----------------|---------|
| cartesian_point | cartesian_point | boolean |
| cartesian_point | cartesian_shape | boolean |
| cartesian_shape | cartesian_point | boolean |
| cartesian_shape | cartesian_shape | boolean |
| geo_point       | geo_point       | boolean |
| geo_point       | geo_shape       | boolean |
| geo_point       | geohash         | boolean |
| geo_point       | geohex          | boolean |
| geo_point       | geotile         | boolean |
| geo_shape       | geo_point       | boolean |
| geo_shape       | geo_shape       | boolean |
| geohash         | geo_point       | boolean |
| geohex          | geo_point       | boolean |
| geotile         | geo_point       | boolean |

**Example**
```esql
FROM airport_city_boundaries
| WHERE ST_DISJOINT(city_boundary, TO_GEOSHAPE("POLYGON((-10 -60, 120 -60, 120 60, -10 60, -10 -60))"))
| KEEP abbrev, airport, region, city, city_location
```


| abbrev:keyword | airport:text                 | region:text        | city:keyword       | city_location:geo_point  |
|----------------|------------------------------|--------------------|--------------------|--------------------------|
| ACA            | General Juan N Alvarez Int'l | Acapulco de Juárez | Acapulco de Juárez | POINT (-99.8825 16.8636) |


### `ST_CONTAINS`

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_contains.svg)

**Parameters**
<definitions>
  <definition term="geomA">
    Expression of type `geo_point`, `cartesian_point`, `geo_shape` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
  <definition term="geomB">
    Expression of type `geo_point`, `cartesian_point`, `geo_shape` or `cartesian_shape`. If `null`, the function returns `null`. The second parameter must also have the same coordinate system as the first. This means it is not possible to combine `geo_*` and `cartesian_*` parameters.
  </definition>
</definitions>

**Description**
Returns whether the first geometry contains the second geometry. This is the inverse of the [ST_WITHIN](#esql-st_within) function.
**Supported types**

| geomA           | geomB           | result  |
|-----------------|-----------------|---------|
| cartesian_point | cartesian_point | boolean |
| cartesian_point | cartesian_shape | boolean |
| cartesian_shape | cartesian_point | boolean |
| cartesian_shape | cartesian_shape | boolean |
| geo_point       | geo_point       | boolean |
| geo_point       | geo_shape       | boolean |
| geo_shape       | geo_point       | boolean |
| geo_shape       | geo_shape       | boolean |

**Example**
```esql
FROM airport_city_boundaries
| WHERE ST_CONTAINS(city_boundary, TO_GEOSHAPE("POLYGON((109.35 18.3, 109.45 18.3, 109.45 18.4, 109.35 18.4, 109.35 18.3))"))
| KEEP abbrev, airport, region, city, city_location
```


| abbrev:keyword | airport:text        | region:text | city:keyword | city_location:geo_point |
|----------------|---------------------|-------------|--------------|-------------------------|
| SYX            | Sanya Phoenix Int'l | 天涯区         | Sanya        | POINT(109.5036 18.2533) |


### `ST_WITHIN`

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_within.svg)

**Parameters**
<definitions>
  <definition term="geomA">
    Expression of type `geo_point`, `cartesian_point`, `geo_shape` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
  <definition term="geomB">
    Expression of type `geo_point`, `cartesian_point`, `geo_shape` or `cartesian_shape`. If `null`, the function returns `null`. The second parameter must also have the same coordinate system as the first. This means it is not possible to combine `geo_*` and `cartesian_*` parameters.
  </definition>
</definitions>

**Description**
Returns whether the first geometry is within the second geometry. This is the inverse of the [ST_CONTAINS](#esql-st_contains) function.
**Supported types**

| geomA           | geomB           | result  |
|-----------------|-----------------|---------|
| cartesian_point | cartesian_point | boolean |
| cartesian_point | cartesian_shape | boolean |
| cartesian_shape | cartesian_point | boolean |
| cartesian_shape | cartesian_shape | boolean |
| geo_point       | geo_point       | boolean |
| geo_point       | geo_shape       | boolean |
| geo_shape       | geo_point       | boolean |
| geo_shape       | geo_shape       | boolean |

**Example**
```esql
FROM airport_city_boundaries
| WHERE ST_WITHIN(city_boundary, TO_GEOSHAPE("POLYGON((109.1 18.15, 109.6 18.15, 109.6 18.65, 109.1 18.65, 109.1 18.15))"))
| KEEP abbrev, airport, region, city, city_location
```


| abbrev:keyword | airport:text        | region:text | city:keyword | city_location:geo_point |
|----------------|---------------------|-------------|--------------|-------------------------|
| SYX            | Sanya Phoenix Int'l | 天涯区         | Sanya        | POINT(109.5036 18.2533) |


## Geometry functions


### `ST_X`

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_x.svg)

**Parameters**
<definitions>
  <definition term="point">
    Expression of type `geo_point` or `cartesian_point`. If `null`, the function returns `null`.
  </definition>
</definitions>

**Description**
Extracts the `x` coordinate from the supplied point. If the point is of type `geo_point` this is equivalent to extracting the `longitude` value.
**Supported types**

| point           | result |
|-----------------|--------|
| cartesian_point | double |
| geo_point       | double |

**Example**
```esql
ROW point = TO_GEOPOINT("POINT(42.97109629958868 14.7552534006536)")
| EVAL x =  ST_X(point), y = ST_Y(point)
```


| point:geo_point                           | x:double          | y:double         |
|-------------------------------------------|-------------------|------------------|
| POINT(42.97109629958868 14.7552534006536) | 42.97109629958868 | 14.7552534006536 |


### `ST_Y`

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_y.svg)

**Parameters**
<definitions>
  <definition term="point">
    Expression of type `geo_point` or `cartesian_point`. If `null`, the function returns `null`.
  </definition>
</definitions>

**Description**
Extracts the `y` coordinate from the supplied point. If the point is of type `geo_point` this is equivalent to extracting the `latitude` value.
**Supported types**

| point           | result |
|-----------------|--------|
| cartesian_point | double |
| geo_point       | double |

**Example**
```esql
ROW point = TO_GEOPOINT("POINT(42.97109629958868 14.7552534006536)")
| EVAL x =  ST_X(point), y = ST_Y(point)
```


| point:geo_point                           | x:double          | y:double         |
|-------------------------------------------|-------------------|------------------|
| POINT(42.97109629958868 14.7552534006536) | 42.97109629958868 | 14.7552534006536 |


### `ST_NPOINTS`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Planned
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_npoints.svg)

**Parameters**
<definitions>
  <definition term="geometry">
    Expression of type `geo_point`, `geo_shape`, `cartesian_point` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
</definitions>

**Description**
Counts the number of points in the supplied geometry.
**Supported types**

| geometry        | result  |
|-----------------|---------|
| cartesian_point | integer |
| cartesian_shape | integer |
| geo_point       | integer |
| geo_shape       | integer |

**Example**
```esql
FROM airport_city_boundaries
| WHERE abbrev == "CPH"
| EVAL points = ST_NPOINTS(city_boundary)
| KEEP abbrev, airport, points
```


| abbrev:keyword | airport:text | points:integer |
|----------------|--------------|----------------|
| CPH            | Copenhagen   | 15             |


### `ST_SIMPLIFY`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Planned
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_simplify.svg)

**Parameters**
<definitions>
  <definition term="geometry">
    Expression of type `geo_point`, `geo_shape`, `cartesian_point` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
  <definition term="tolerance">
    Tolerance for the geometry simplification, in the units of the input SRS
  </definition>
</definitions>

**Description**
Simplifies the input geometry by applying the Douglas-Peucker algorithm with a specified tolerance. Vertices that fall within the tolerance distance from the simplified shape are removed. Note that the resulting geometry may be invalid, even if the original input was valid.
**Supported types**

| geometry        | tolerance | result          |
|-----------------|-----------|-----------------|
| cartesian_point | double    | cartesian_point |
| cartesian_shape | double    | cartesian_shape |
| geo_point       | double    | geo_point       |
| geo_shape       | double    | geo_shape       |

**Example**
```esql
ROW wkt = "POLYGON ((7.998 53.827, 9.470 53.068, 15.754 53.801, 16.523 57.160, 11.162 57.868, 8.064 57.445, 6.219 55.317, 7.998 53.827))"
| EVAL simplified = ST_SIMPLIFY(TO_GEOSHAPE(wkt), 0.7)
```


| wkt:keyword                                                                                                                   | simplified:geo_shape                                                                          |
|-------------------------------------------------------------------------------------------------------------------------------|-----------------------------------------------------------------------------------------------|
| POLYGON ((7.998 53.827, 9.470 53.068, 15.754 53.801, 16.523 57.160, 11.162 57.868, 8.064 57.445, 6.219 55.317, 7.998 53.827)) | POLYGON ((9.47 53.068, 15.754 53.801, 16.523 57.16, 8.064 57.445, 6.219 55.317, 9.47 53.068)) |


## `ST_ENVELOPE`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Preview
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_envelope.svg)

**Parameters**
<definitions>
  <definition term="geometry">
    Expression of type `geo_point`, `geo_shape`, `cartesian_point` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
</definitions>

**Description**
Determines the minimum bounding box of the supplied geometry.
**Supported types**

| geometry        | result          |
|-----------------|-----------------|
| cartesian_point | cartesian_shape |
| cartesian_shape | cartesian_shape |
| geo_point       | geo_shape       |
| geo_shape       | geo_shape       |

**Example**
```esql
FROM airport_city_boundaries
| WHERE abbrev == "CPH"
| EVAL envelope = ST_ENVELOPE(city_boundary)
| KEEP abbrev, airport, envelope
```


| abbrev:keyword | airport:text | envelope:geo_shape                                                                  |
|----------------|--------------|-------------------------------------------------------------------------------------|
| CPH            | Copenhagen   | BBOX (12.45299999602139, 12.639799928292632, 55.732699991203845, 55.63179998192936) |


### `ST_XMAX`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Preview
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_xmax.svg)

**Parameters**
<definitions>
  <definition term="point">
    Expression of type `geo_point`, `geo_shape`, `cartesian_point` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
</definitions>

**Description**
Extracts the maximum value of the `x` coordinates from the supplied geometry. If the geometry is of type `geo_point` or `geo_shape` this is equivalent to extracting the maximum `longitude` value.
**Supported types**

| point           | result |
|-----------------|--------|
| cartesian_point | double |
| cartesian_shape | double |
| geo_point       | double |
| geo_shape       | double |

**Example**
```esql
FROM airport_city_boundaries
| WHERE abbrev == "CPH"
| EVAL envelope = ST_ENVELOPE(city_boundary)
| EVAL xmin = ST_XMIN(envelope), xmax = ST_XMAX(envelope), ymin = ST_YMIN(envelope), ymax = ST_YMAX(envelope)
| KEEP abbrev, airport, xmin, xmax, ymin, ymax
```


| abbrev:keyword | airport:text | xmin:double       | xmax:double        | ymin:double       | ymax:double        |
|----------------|--------------|-------------------|--------------------|-------------------|--------------------|
| CPH            | Copenhagen   | 12.45299999602139 | 12.639799928292632 | 55.63179998192936 | 55.732699991203845 |


### `ST_XMIN`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Preview
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_xmin.svg)

**Parameters**
<definitions>
  <definition term="point">
    Expression of type `geo_point`, `geo_shape`, `cartesian_point` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
</definitions>

**Description**
Extracts the minimum value of the `x` coordinates from the supplied geometry. If the geometry is of type `geo_point` or `geo_shape` this is equivalent to extracting the minimum `longitude` value.
**Supported types**

| point           | result |
|-----------------|--------|
| cartesian_point | double |
| cartesian_shape | double |
| geo_point       | double |
| geo_shape       | double |

**Example**
```esql
FROM airport_city_boundaries
| WHERE abbrev == "CPH"
| EVAL envelope = ST_ENVELOPE(city_boundary)
| EVAL xmin = ST_XMIN(envelope), xmax = ST_XMAX(envelope), ymin = ST_YMIN(envelope), ymax = ST_YMAX(envelope)
| KEEP abbrev, airport, xmin, xmax, ymin, ymax
```


| abbrev:keyword | airport:text | xmin:double       | xmax:double        | ymin:double       | ymax:double        |
|----------------|--------------|-------------------|--------------------|-------------------|--------------------|
| CPH            | Copenhagen   | 12.45299999602139 | 12.639799928292632 | 55.63179998192936 | 55.732699991203845 |


### `ST_YMAX`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Preview
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_ymax.svg)

**Parameters**
<definitions>
  <definition term="point">
    Expression of type `geo_point`, `geo_shape`, `cartesian_point` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
</definitions>

**Description**
Extracts the maximum value of the `y` coordinates from the supplied geometry. If the geometry is of type `geo_point` or `geo_shape` this is equivalent to extracting the maximum `latitude` value.
**Supported types**

| point           | result |
|-----------------|--------|
| cartesian_point | double |
| cartesian_shape | double |
| geo_point       | double |
| geo_shape       | double |

**Example**
```esql
FROM airport_city_boundaries
| WHERE abbrev == "CPH"
| EVAL envelope = ST_ENVELOPE(city_boundary)
| EVAL xmin = ST_XMIN(envelope), xmax = ST_XMAX(envelope), ymin = ST_YMIN(envelope), ymax = ST_YMAX(envelope)
| KEEP abbrev, airport, xmin, xmax, ymin, ymax
```


| abbrev:keyword | airport:text | xmin:double       | xmax:double        | ymin:double       | ymax:double        |
|----------------|--------------|-------------------|--------------------|-------------------|--------------------|
| CPH            | Copenhagen   | 12.45299999602139 | 12.639799928292632 | 55.63179998192936 | 55.732699991203845 |


### `ST_YMIN`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Preview
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_ymin.svg)

**Parameters**
<definitions>
  <definition term="point">
    Expression of type `geo_point`, `geo_shape`, `cartesian_point` or `cartesian_shape`. If `null`, the function returns `null`.
  </definition>
</definitions>

**Description**
Extracts the minimum value of the `y` coordinates from the supplied geometry. If the geometry is of type `geo_point` or `geo_shape` this is equivalent to extracting the minimum `latitude` value.
**Supported types**

| point           | result |
|-----------------|--------|
| cartesian_point | double |
| cartesian_shape | double |
| geo_point       | double |
| geo_shape       | double |

**Example**
```esql
FROM airport_city_boundaries
| WHERE abbrev == "CPH"
| EVAL envelope = ST_ENVELOPE(city_boundary)
| EVAL xmin = ST_XMIN(envelope), xmax = ST_XMAX(envelope), ymin = ST_YMIN(envelope), ymax = ST_YMAX(envelope)
| KEEP abbrev, airport, xmin, xmax, ymin, ymax
```


| abbrev:keyword | airport:text | xmin:double       | xmax:double        | ymin:double       | ymax:double        |
|----------------|--------------|-------------------|--------------------|-------------------|--------------------|
| CPH            | Copenhagen   | 12.45299999602139 | 12.639799928292632 | 55.63179998192936 | 55.732699991203845 |


## Grid encoding functions


### `ST_GEOTILE`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Preview
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_geotile.svg)

**Parameters**
<definitions>
  <definition term="geometry">
    Expression of type `geo_point`. If `null`, the function returns `null`.
  </definition>
  <definition term="precision">
    Expression of type `integer`. If `null`, the function returns `null`. Valid values are between [0 and 29](https://wiki.openstreetmap.org/wiki/Zoom_levels).
  </definition>
  <definition term="bounds">
    Optional bounds to filter the grid tiles, a `geo_shape` of type `BBOX`. Use [`ST_ENVELOPE`](#esql-st_envelope) if the `geo_shape` is of any other type.
  </definition>
</definitions>

**Description**
Calculates the `geotile` of the supplied geo_point at the specified precision. The result is long encoded. Use [TO_STRING](#esql-to_string) to convert the result to a string, [TO_LONG](#esql-to_long) to convert it to a `long`, or [TO_GEOSHAPE](#esql-to_geoshape) to calculate the `geo_shape` bounding geometry.  These functions are related to the [`geo_grid` query](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-geo-grid-query) and the [`geotile_grid` aggregation](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-geotilegrid-aggregation).
**Supported types**

| geometry  | precision | bounds    | result  |
|-----------|-----------|-----------|---------|
| geo_point | integer   | geo_shape | geotile |
| geo_point | integer   |           | geotile |

**Example**
```esql
FROM airports
| EVAL geotile = ST_GEOTILE(location, 2)
| STATS
    count = COUNT(geotile),
    centroid = ST_CENTROID_AGG(location)
      BY geotile
| EVAL geotileString = TO_STRING(geotile)
| SORT count DESC, geotileString ASC
| KEEP count, centroid, geotileString
```


| count:long | centroid:geo_point                              | geotileString:keyword |
|------------|-------------------------------------------------|-----------------------|
| 286        | POINT (39.31202001609169 35.149993664386415)    | 2/2/1                 |
| 197        | POINT (-55.387361375756825 31.952955322292855)  | 2/1/1                 |
| 136        | POINT (-110.97162496141048 36.87185255084734)   | 2/0/1                 |
| 106        | POINT (119.35907618669827 25.46263281488791)    | 2/3/1                 |
| 67         | POINT (-58.031108492373754 -22.624166105151065) | 2/1/2                 |
| 46         | POINT (142.95455511274707 -20.581492295427978)  | 2/3/2                 |
| 34         | POINT (31.38476753634784 -14.64374022804858)    | 2/2/2                 |
| 8          | POINT (-160.0723083713092 -19.124013530672528)  | 2/0/2                 |
| 6          | POINT (23.95813101902604 70.17537698848173)     | 2/2/0                 |
| 3          | POINT (-133.4001641627401 72.06833167467266)    | 2/0/0                 |
| 2          | POINT (-68.47209956031293 66.77569948369637)    | 2/1/0                 |


### `ST_GEOHEX`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Preview
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_geohex.svg)

**Parameters**
<definitions>
  <definition term="geometry">
    Expression of type `geo_point`. If `null`, the function returns `null`.
  </definition>
  <definition term="precision">
    Expression of type `integer`. If `null`, the function returns `null`. Valid values are between [0 and 15](https://h3geo.org/docs/core-library/restable/).
  </definition>
  <definition term="bounds">
    Optional bounds to filter the grid tiles, a `geo_shape` of type `BBOX`. Use [`ST_ENVELOPE`](#esql-st_envelope) if the `geo_shape` is of any other type.
  </definition>
</definitions>

**Description**
Calculates the `geohex`, the H3 cell-id, of the supplied geo_point at the specified precision. The result is long encoded. Use [TO_STRING](#esql-to_string) to convert the result to a string, [TO_LONG](#esql-to_long) to convert it to a `long`, or [TO_GEOSHAPE](#esql-to_geoshape) to calculate the `geo_shape` bounding geometry.  These functions are related to the [`geo_grid` query](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-geo-grid-query) and the [`geohex_grid` aggregation](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-geohexgrid-aggregation).
**Supported types**

| geometry  | precision | bounds    | result |
|-----------|-----------|-----------|--------|
| geo_point | integer   | geo_shape | geohex |
| geo_point | integer   |           | geohex |

**Example**
```esql
FROM airports
| EVAL geohex = ST_GEOHEX(location, 1)
| STATS
    count = COUNT(geohex),
    centroid = ST_CENTROID_AGG(location)
      BY geohex
| WHERE count >= 10
| EVAL geohexString = TO_STRING(geohex)
| KEEP count, centroid, geohexString
| SORT count DESC, geohexString ASC
```


| count:long | centroid:geo_point                             | geohexString:keyword |
|------------|------------------------------------------------|----------------------|
| 22         | POINT (7.250850197689777 48.21363834643059)    | 811fbffffffffff      |
| 18         | POINT (-80.64959161449224 40.04119813675061)   | 812abffffffffff      |
| 17         | POINT (-0.7606179875266903 52.86413913565304)  | 81197ffffffffff      |
| 13         | POINT (22.53157936179867 41.98255742864254)    | 811efffffffffff      |
| 13         | POINT (78.30096947387435 26.073904778951636)   | 813dbffffffffff      |
| 12         | POINT (-76.39781514415517 45.16300531569868)   | 812bbffffffffff      |
| 12         | POINT (-100.30120467301458 20.114154297625646) | 8149bffffffffff      |
| 11         | POINT (18.037187419831753 48.66540593306788)   | 811e3ffffffffff      |
| 11         | POINT (-83.42379064553164 33.18388901439241)   | 8144fffffffffff      |
| 11         | POINT (-99.4237939513881 27.100012352774765)   | 8148bffffffffff      |
| 10         | POINT (128.01009018346667 35.8699960866943)    | 8130fffffffffff      |


### `ST_GEOHASH`

<applies-to>
  - Elastic Cloud Serverless: Preview
  - Elastic Stack: Preview
</applies-to>

**Syntax**
![Embedded](https://www.elastic.co/docs/reference/query-languages/esql/images/functions/st_geohash.svg)

**Parameters**
<definitions>
  <definition term="geometry">
    Expression of type `geo_point`. If `null`, the function returns `null`.
  </definition>
  <definition term="precision">
    Expression of type `integer`. If `null`, the function returns `null`. Valid values are between [1 and 12](https://en.wikipedia.org/wiki/Geohash).
  </definition>
  <definition term="bounds">
    Optional bounds to filter the grid tiles, a `geo_shape` of type `BBOX`. Use [`ST_ENVELOPE`](#esql-st_envelope) if the `geo_shape` is of any other type.
  </definition>
</definitions>

**Description**
Calculates the `geohash` of the supplied geo_point at the specified precision. The result is long encoded. Use [TO_STRING](#esql-to_string) to convert the result to a string, [TO_LONG](#esql-to_long) to convert it to a `long`, or [TO_GEOSHAPE](#esql-to_geoshape) to calculate the `geo_shape` bounding geometry.  These functions are related to the [`geo_grid` query](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-geo-grid-query) and the [`geohash_grid` aggregation](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-geohashgrid-aggregation).
**Supported types**

| geometry  | precision | bounds    | result  |
|-----------|-----------|-----------|---------|
| geo_point | integer   | geo_shape | geohash |
| geo_point | integer   |           | geohash |

**Example**
```esql
FROM airports
| EVAL geohash = ST_GEOHASH(location, 1)
| STATS
    count = COUNT(geohash),
    centroid = ST_CENTROID_AGG(location)
      BY geohash
| WHERE count >= 10
| EVAL geohashString = TO_STRING(geohash)
| KEEP count, centroid, geohashString
| SORT count DESC, geohashString ASC
```


| count:long | centroid:geo_point                             | geohashString:keyword |
|------------|------------------------------------------------|-----------------------|
| 118        | POINT (-77.41857436454018 26.96522968734409)   | d                     |
| 96         | POINT (23.181679135886952 27.295384635654045)  | s                     |
| 94         | POINT (70.94076107503807 25.691916451026547)   | t                     |
| 90         | POINT (-104.3941700803116 30.811849871650338)  | 9                     |
| 89         | POINT (18.71573683606942 53.165169130707305)   | u                     |
| 85         | POINT (114.3722876966657 24.908398092505248)   | w                     |
| 51         | POINT (-61.44522591713159 -22.87209844956284)  | 6                     |
| 38         | POINT (-9.429514887252529 25.497624435045413)  | e                     |
| 34         | POINT (-111.8071846965262 52.464381378993174)  | c                     |
| 30         | POINT (28.7045472683385 -14.706001980230212)   | k                     |
| 28         | POINT (159.52750137208827 -25.555616633001982) | r                     |
| 22         | POINT (-4.410395708612421 54.90304926367985)   | g                     |
| 21         | POINT (-69.40534970590046 50.93379438189523)   | f                     |
| 17         | POINT (114.05526293222519 -10.898114638950895) | q                     |
| 16         | POINT (147.40052131412085 21.054660080408212)  | x                     |
| 13         | POINT (63.64716878519035 54.37333276101317)    | v                     |
| 12         | POINT (-39.53510569408536 -11.72166372067295)  | 7                     |