﻿---
title: ES|QL RERANK command
description: 
url: https://www.elastic.co/docs/reference/query-languages/esql/commands/rerank
products:
  - Elasticsearch
---

# ES|QL RERANK command
<applies-to>
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Preview from 9.2 to 9.3
</applies-to>

The `RERANK` command uses an inference model to compute a new relevance score
for an initial set of documents, directly within your ESQL queries.
<tab-set>
  <tab-item title="9.3.0+">
    Starting in version 9.3.0, `RERANK` automatically limits processing to **1000
    rows by default** to prevent accidental high consumption. This limit is applied
    before the `RERANK` command executes.If you need to process more rows, you can adjust the limit using the cluster setting:
    ```
    PUT _cluster/settings
    {
      "persistent": {
        "esql.command.rerank.limit": 5000
      }
    }
    ```
    You can also disable the command entirely if needed:
    ```
    PUT _cluster/settings
    {
      "persistent": {
        "esql.command.rerank.enabled": false
      }
    }
    ```
  </tab-item>

  <tab-item title="9.2.x">
    No automatic row limit is applied. **You should always use `LIMIT` before or after `RERANK` to control the number of documents processed**, to avoid accidentally reranking large datasets which can result in high latency and increased costs.For example:
    ```esql
    FROM books
    | WHERE title:"search query"
    | SORT _score DESC
    | LIMIT 100 
    | RERANK "search query" ON title WITH { "inference_id" : "my_rerank_endpoint" }
    ```
  </tab-item>
</tab-set>

**Syntax**
```esql
RERANK [column =] query ON field [, field, ...] [WITH { "inference_id" : "my_inference_endpoint" }]
```

**Parameters**
<definitions>
  <definition term="column">
    (Optional) The name of the output column containing the reranked scores.
    If not specified, the results will be stored in a column named `_score`.
    If the specified column already exists, it will be overwritten with the new
    results.
  </definition>
  <definition term="query">
    The query text used to rerank the documents. This is typically the same
    query used in the initial search.
  </definition>
  <definition term="field">
    One or more fields to use for reranking. These fields should contain the
    text that the reranking model will evaluate.
  </definition>
  <definition term="my_inference_endpoint">
    The ID of
    the [inference endpoint](https://www.elastic.co/docs/explore-analyze/elastic-inference/inference-api)
    to use for the task.
    The inference endpoint must be configured with the `rerank` task type.
  </definition>
</definitions>

**Description**
The `RERANK` command uses an inference model to compute a new relevance score
for an initial set of documents, directly within your ESQL queries.
Typically, you first use a `WHERE` clause with a function like `MATCH` to
retrieve an initial set of documents. This set is often sorted by `_score` and
reduced to the top results (for example, 100) using `LIMIT`. The `RERANK`
command then processes this smaller, refined subset, which is a good balance
between performance and accuracy.
When using `RERANK` with a multivalue column, each value is ranked individually.
The score column is then assigned the maximum score resulting from ranking the
individual values.
**Requirements**
To use this command, you must deploy your reranking model in Elasticsearch as
an [inference endpoint](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-inference-put)
with the
task type `rerank`.

#### Handling timeouts

`RERANK` commands may time out when processing large datasets or complex
queries. The default timeout is 10 minutes, but you can increase this limit if
necessary.
How you increase the timeout depends on your deployment type:
<tab-set>
  <tab-item title="Elastic Cloud Hosted">
    - You can adjust Elasticsearch settings in
      the [Elastic Cloud Console](https://www.elastic.co/docs/deploy-manage/deploy/elastic-cloud/edit-stack-settings)
    - You can also adjust the `search.default_search_timeout` cluster setting
      using [Kibana's Advanced settings](https://www.elastic.co/docs/reference/kibana/advanced-settings#kibana-search-settings)
  </tab-item>

  <tab-item title="Self-managed">
    - You can configure at the cluster level by setting
      `search.default_search_timeout` in `elasticsearch.yml` or updating
      via [Cluster Settings API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-cluster-put-settings)
    - You can also adjust the `search:timeout` setting
      using [Kibana's Advanced settings](https://www.elastic.co/docs/reference/kibana/advanced-settings#kibana-search-settings)
    - Alternatively, you can add timeout parameters to individual queries
  </tab-item>

  <tab-item title="Elastic Cloud Serverless">
    - Requires a manual override from Elastic Support because you cannot modify
      timeout settings directly
  </tab-item>
</tab-set>

If you don't want to increase the timeout limit, try the following:
- Reduce data volume with `LIMIT` or more selective filters before the `RERANK`
  command
- Split complex operations into multiple simpler queries
- Configure your HTTP client's response timeout (Refer
  to [HTTP client configuration](/docs/reference/elasticsearch/configuration-reference/networking-settings#_http_client_configuration))

**Examples**
Rerank search results using a simple query and a single field:
```esql
FROM books METADATA _score
| WHERE MATCH(description, "hobbit")
| SORT _score DESC
| LIMIT 100
| RERANK "hobbit" ON description WITH { "inference_id" : "test_reranker" }
| LIMIT 3
| KEEP title, _score
```


| title:text                        | _score:double         |
|-----------------------------------|-----------------------|
| Poems from the Hobbit             | 0.0015673980815336108 |
| The Lord of the Rings - Boxed Set | 0.0011135857785120606 |
| Letters of J R R Tolkien          | 0.0024999999441206455 |

Rerank search results using a query and multiple fields, and store the new score
in a column named `rerank_score`:
```esql
FROM books METADATA _score
| WHERE MATCH(description, "hobbit") OR MATCH(author, "Tolkien")
| SORT _score DESC
| LIMIT 100
| RERANK rerank_score = "hobbit" ON description, author WITH { "inference_id" : "test_reranker" }
| SORT rerank_score DESC, book_no DESC
| LIMIT 3
```


| title:text                                                                 | _score:double      | rerank_score:double |
|----------------------------------------------------------------------------|--------------------|---------------------|
| The Lord of the Rings Poster Collection: Six Paintings by Alan Lee (No. 1) | 0.7976278066635132 | 0.125               |
| A Middle English Reader and Vocabulary                                     | 1.043297290802002  | 0.07692307978868484 |
| FINAL WITNESS                                                              | 1.4058171510696411 | 0.07692307978868484 |

Combine the original score with the reranked score:
```esql
FROM books METADATA _score
| WHERE MATCH(description, "hobbit") OR MATCH(author, "Tolkien")
| SORT _score DESC
| LIMIT 100
| RERANK rerank_score = "hobbit" ON title, description WITH { "inference_id" : "test_reranker" }
| EVAL original_score = _score, _score = rerank_score + original_score
| SORT _score
| LIMIT 3
```


| title:text                                                                 | _score:double | rerank_score:double |
|----------------------------------------------------------------------------|---------------|---------------------|
| The Lord of the Rings Poster Collection: Six Paintings by Alan Lee (No. 1) | 0.7976278     | 0.125               |
| A Middle English Reader and Vocabulary                                     | 1.043297      | 0.07692308          |
| FINAL WITNESS                                                              | 1.405817      | 0.07692308          |

Rerank using snippets extracted from the document with the `TOP_SNIPPETS`
function:
```esql
FROM books
| WHERE MATCH(title, "return")
| RERANK "Tolkien" ON TOP_SNIPPETS(description, "Tolkien", { "num_snippets": 3, "num_words": 25 }) WITH { "inference_id" : "test_reranker" }
```


| book_no:keyword | title:text                                                       | _score:double        |
|-----------------|------------------------------------------------------------------|----------------------|
| 2714            | Return of the King Being the Third Part of The Lord of the Rings | 0.007092198356986046 |
| 7350            | Return of the Shadow                                             | 0.012500000186264515 |