﻿---
title: Microsoft Exchange Online Message Trace
description: This integration is for Microsoft Exchange Online Message Trace logs. It includes the following datasets for receiving logs over the Microsoft Exchange...
url: https://www.elastic.co/docs/reference/integrations/microsoft_exchange_online_message_trace
products:
  - Elastic integrations
applies_to:
  - Serverless Observability projects: Generally available
  - Serverless Security projects: Generally available
  - Elastic Stack: Generally available since 9.1
---

# Microsoft Exchange Online Message Trace
|                                                                                                 |                                 |
|-------------------------------------------------------------------------------------------------|---------------------------------|
| **Version**                                                                                     | 1.29.1 ([View all](#changelog)) |
| **Subscription level**[What's this?](https://www.elastic.co/subscriptions)                      | Basic                           |
| **Developed by**[What's this?](https://www.elastic.co/docs/reference/integrations/developed-by) | Elastic                         |
| **Ingestion method(s)**                                                                         | API, File                       |
| **Minimum Kibana version(s)**                                                                   | 9.1.49.0.78.19.4                |

This integration is for Microsoft Exchange Online Message Trace logs. It includes the following datasets for receiving logs over the Microsoft Exchange Online Message Trace API or read from a file:
- `log` dataset: supports Microsoft Exchange Online Message Trace logs.


## Message Trace API Deprecation Notification

Microsoft has announced the deprecation on March 18th, 2026 of the legacy
Message Trace API support in the Reporting Webservice, which is used by this
integration.
However, that legacy API is already unavailable to new Microsoft tenants, and
the replacement Graph-based Message Trace API is not yet available. It is
tentatively expected in December 2025.
For the new Message Trace experience, there is an updated PowerShell cmdlet,
`Get-MessageTraceV2`, in General Availability since June 3rd, 2025.
- If your tenant was created before the API deprecation, you can use the
  current integration until March 2026.
- If your tenant was created since the API deprecation, you can use a
  PowerShell script to manully collect data using the new cmdlet, and use this
  integration to ingest it from a file. For more information, please see the
  "Logfile collection" section below.
- We're tracking Microsoft's new Graph-based Message Trace API (tentatively
  expected in December 2025) and plan to update this integration once it's
  available. Read Microsoft's announcement
  [here](https://techcommunity.microsoft.com/blog/exchange/announcing-general-availability-ga-of-the-new-message-trace-in-exchange-online/4420243).


## Basic Auth Deprecation notification

The basic authentication configuration fields have been removed from this integration as Microsoft has deprecated and disabled basic authentication for Exchange Online. See the [deprecation notification](https://learn.microsoft.com/en-us/exchange/clients-and-mobile-in-exchange-online/deprecation-of-basic-authentication-exchange-online) for details.

## Office 365 Account Requirements

At a minimum, your Office 365 service account should include a role with Message Tracking and View‑Only Recipients permissions, assigned to the Office 365 user account
that will be used for the integration. Assign these permissions using the [Exchange admin center](https://admin.exchange.microsoft.com).

## Logs

Logs are either gathered via the rest API or via a logfile. [Log Documentation](https://docs.microsoft.com/en-us/previous-versions/office/developer/o365-enterprise-developers/jj984335(v=office.15))

## Microsoft Exchange Online Message Trace API

The `log` dataset collects the Microsoft Exchange Online Message Trace logs. To search for ingested logs in Elasticsearch you need to query using `datastream.dataset: microsoft_exchange_online_message_trace.log`. This integration will poll the Microsoft Exchange Online Message Trace legacy API () to pull Message Trace logs and ingest them via the ingest pipelines.

## Configuring with OAuth2

In order to continue using the Microsoft Exchange Online Message Trace you will need to enable and configure OAuth2 authentication via your service app.
- ### Service App Configuration
  1. In the [Azure portal](https://portal.azure.com/), create a Microsoft Entra App (service app) Registration. For details please refer to the official [Microsoft Documentation](https://learn.microsoft.com/en-us/entra/identity-platform/howto-create-service-principal-portal).
2. In most cases under the `Redirect URI` section, you would want to configure the value `Web` for the `app type` and `http://localhost` for the `Redirect URI`, unless there are some specific requirements on your end.
3. Assign the application at least one Microsoft Entra (Azure AD) role that will enable it to access the Reporting Web Service:
   - Security Reader
- Global Reader
4. The App Registration should contain the following API permissions: Office 365 Exchange Online > `ReportingWebService.Read.All` (application). See [Specify the permissions your app requires to access the Reporting Web Service](https://learn.microsoft.com/en-gb/previous-versions/office/developer/o365-enterprise-developers/jj984325(v=office.15)#specify-the-permissions-your-app-requires-to-access-the-reporting-web-service).
- ### Configuring OAuth2 Credentials
  Once you have your service app registered and configured, you can now configure your OAuth2 credentials as follows:-
  1. Generate a client secret for your registered service app. Copy and store the `client secret value` with you as this will be required for your OAuth2 credentials.
2. Fill in the following fields with the appropriate values from your `configured service app`:-
   - **Client ID**: The `client_id` of your `service app` to pass in the OAuth request parameter.
- **Client secret**:  The `client_secret`  of your `service app` that you generated earlier, to pass in the OAuth request parameter.
- **Tenant ID**: The Directory ID (tenant identifier) of your `service app` in your Microsoft Entra ID(Azure Active Directory).
  With these values now configured, the OAuth2 configuration for the integration should be ideally complete. For more details, please check the
  official doc for [Getting Started with Reporting Web Service](https://learn.microsoft.com/en-gb/previous-versions/office/developer/o365-enterprise-developers/jj984325(v=office.15)#get-started-with-reporting-web-service).


### NOTE

- To configure `Local Domains` you can check your [Microsoft Admin Exchange Center](https://admin.exchange.microsoft.com/) for the domains
  available in your organization. They are usually under the sections [Accepted Domains](https://admin.exchange.microsoft.com/#/accepteddomains) and [Remote Domains](https://admin.exchange.microsoft.com/#/remotedomains).
- The default `Interval` is configured to `1h` and `Initial Interval` to `48h`, you can however change these to your required values. The look-back
  value of `Initial Interval` should not exceed `200 hours` as this might cause unexpected errors with the API.
- The default `Minimum Age` is configured to `1h`, you can however change these to your required values. The `Minimum Age` was introduced to allow a sliding
  window to exist in combination with the `Initial Interval`. If you do not require a sliding window you can set this to `0s` which will cause the `Minimum Age` to
  always default to the `current time (now)`.
- The default `Additional Look-back Time` value is configured for `1h`.
  This is intended to capture events that may not have been initially present due to eventual consistency.
  This value does not need to exceed [`24h`](https://learn.microsoft.com/en-us/previous-versions/office/developer/o365-enterprise-developers/jj984335(v=office.15)#data-granularity-persistence-and-availability).
  - <note>
  The larger this value is, the less likely events will be missed, however, this will cause the integration to take longer to pull all events, making newer events take longer to become present.
  </note>
- The default value of `Batch Size` is set to 1000. This means for every request Httpjson will paginate with a value of 1000 results per page. The
  maximum page size supported by the Message Trace API is `2000`. The API will return an empty `value` array when there are no more logs to pull and the
  pagination will terminate with an error that can be ignored.


## Logfile collection

**Disclaimer:** You may need to adapt the authentication method of the script
below to match your environment. For more information about authentication
methods available in PowerShell, please see the
[guides here](https://learn.microsoft.com/en-us/powershell/exchange/connect-to-exchange-online-powershell?view=exchange-ps).
Note that basic authentication (with `-Authentication Basic`) is no longer
supported.
The following example PowerShell script can be adapted to fetch the logs and
write them into a JSON file that the integration can consume (via the logfile
input).
Prerequisites:
Install the Exchange Online Management module by running the following command:
```powershell
Install-Module -Name ExchangeOnlineManagement
```

Import the Exchange Online Management module by running the following command:
```powershell
Import-Module -Name ExchangeOnlineManagement
```

This script would have to be triggered at a certain interval, in accordance
with the look-back interval specified.  In this example script, the look back
is 24 hours, so the interval would need to be daily. For more information about
the `Get-MessageTraceV2` cmdlet, please refer to its
[documentation](https://learn.microsoft.com/en-us/powershell/module/exchangepowershell/get-messagetracev2?view=exchange-ps).
```powershell
# Username and Password
$username = "USERNAME@DOMAIN.TLD"
$password = "PASSWORD"
# Lookback in Hours
$lookback = "-24"
# Results per request (maximum 5000)
$resultSize = "5000"
# Output of the json file
# This would then be ingested via the integration
$output_location = "C:\temp\messageTrace.json"

$password = ConvertTo-SecureString $password -AsPlainText -Force
$Credential = New-Object System.Management.Automation.PSCredential ($username, $password)
$startDate = (Get-Date).AddHours($lookback)
$endDate = Get-Date

Connect-ExchangeOnline -Credential $Credential

$paginate = 1
$output = @()

# Initialize V2-style pagination cursor values
$startingRecipientAddress = $null
$currentEndDate = $endDate

while ($paginate -eq 1)
{
    if ($startingRecipientAddress) {
        $messageTrace = Get-MessageTraceV2 -ResultSize $resultSize -StartDate $startDate -EndDate $currentEndDate -StartingRecipientAddress $startingRecipientAddress
    }
    else {
        $messageTrace = Get-MessageTraceV2 -ResultSize $resultSize -StartDate $startDate -EndDate $currentEndDate
    }

    if (!$messageTrace)
    {
        $paginate = 0
    }
    else
    {
        $output = $output + $messageTrace

        # If we got fewer than ResultSize rows, we've reached the end
        if ($messageTrace.Count -lt [int]$resultSize)
        {
            $paginate = 0
        }
        else
        {
            # Prepare the cursor data for the next query
            $last = $messageTrace[-1]
            $startingRecipientAddress = $last.RecipientAddress
            $currentEndDate = $last.Received
        }
    }
}

if (Test-Path $output_location)
{
    Remove-Item $output_location
}
foreach ($event in $output)
{
    $event.StartDate = [Xml.XmlConvert]::ToString(($event.StartDate), [Xml.XmlDateTimeSerializationMode]::Utc)
    $event.EndDate   = [Xml.XmlConvert]::ToString(($event.EndDate),   [Xml.XmlDateTimeSerializationMode]::Utc)
    $event.Received  = [Xml.XmlConvert]::ToString(($event.Received),  [Xml.XmlDateTimeSerializationMode]::Utc)
    $event = $event | ConvertTo-Json -Compress
    Add-Content $output_location $event -Encoding UTF8
}
```

<dropdown title="Example">
  ```json
  {
      "@timestamp": "2022-10-21T17:25:36.969Z",
      "agent": {
          "ephemeral_id": "11edfb81-b112-45ba-8f01-6e7483e450fa",
          "id": "1c0788e9-492a-441e-acab-fc8c56281cf1",
          "name": "elastic-agent-22259",
          "type": "filebeat",
          "version": "8.19.4"
      },
      "data_stream": {
          "dataset": "microsoft_exchange_online_message_trace.log",
          "namespace": "71098",
          "type": "logs"
      },
      "destination": {
          "domain": "contoso.com",
          "registered_domain": "contoso.com",
          "top_level_domain": "com",
          "user": {
              "domain": "contoso.com",
              "email": "linus@contoso.com",
              "id": "linus@contoso.com",
              "name": "linus"
          }
      },
      "ecs": {
          "version": "8.11.0"
      },
      "elastic_agent": {
          "id": "1c0788e9-492a-441e-acab-fc8c56281cf1",
          "snapshot": false,
          "version": "8.19.4"
      },
      "email": {
          "attachments": {
              "file": {
                  "size": 22761
              }
          },
          "delivery_timestamp": "2022-10-21T17:25:36.969376Z",
          "from": {
              "address": [
                  "noreply@azure.microsoft.com"
              ]
          },
          "local_id": "a5e6dc0f-23df-4b20-d240-08dab38944a1",
          "message_id": "<GVAP278MB037586A65EF1FB2F844B0258DA2D9@GVAP278MB0375.CHEP278.PROD.OUTLOOK.COM>",
          "subject": "testmail 2",
          "to": {
              "address": [
                  "linus@contoso.com"
              ]
          }
      },
      "event": {
          "agent_id_status": "verified",
          "category": [
              "email"
          ],
          "dataset": "microsoft_exchange_online_message_trace.log",
          "ingested": "2025-10-06T13:13:06Z",
          "original": "{\"Organization\":\"contoso.com\",\"MessageId\":\"\\u003cGVAP278MB037586A65EF1FB2F844B0258DA2D9@GVAP278MB0375.CHEP278.PROD.OUTLOOK.COM\\u003e\",\"Received\":\"2022-10-21T17:25:36.969376Z\",\"SenderAddress\":\"noreply@azure.microsoft.com\",\"RecipientAddress\":\"linus@contoso.com\",\"Subject\":\"testmail 2\",\"Status\":\"Delivered\",\"ToIP\":null,\"FromIP\":\"40.107.23.54\",\"Size\":22761,\"MessageTraceId\":\"a5e6dc0f-23df-4b20-d240-08dab38944a1\",\"StartDate\":\"2022-10-21T09:40:10Z\",\"EndDate\":\"2022-10-22T09:40:10Z\",\"Index\":0}",
          "outcome": "success",
          "type": [
              "info"
          ]
      },
      "input": {
          "type": "log"
      },
      "log": {
          "file": {
              "path": "/tmp/service_logs/microsoft_exchange_online_message_trace_test.ndjson.log"
          },
          "offset": 0
      },
      "microsoft": {
          "online_message_trace": {
              "EndDate": "2022-10-22T09:40:10Z",
              "FromIP": "40.107.23.54",
              "Index": 0,
              "MessageId": "<GVAP278MB037586A65EF1FB2F844B0258DA2D9@GVAP278MB0375.CHEP278.PROD.OUTLOOK.COM>",
              "MessageTraceId": "a5e6dc0f-23df-4b20-d240-08dab38944a1",
              "Organization": "contoso.com",
              "Received": "2022-10-21T17:25:36.969376Z",
              "RecipientAddress": "linus@contoso.com",
              "SenderAddress": "noreply@azure.microsoft.com",
              "Size": 22761,
              "StartDate": "2022-10-21T09:40:10Z",
              "Status": "Delivered",
              "Subject": "testmail 2"
          }
      },
      "related": {
          "user": [
              "linus@contoso.com",
              "noreply@azure.microsoft.com",
              "linus",
              "noreply"
          ]
      },
      "source": {
          "domain": "azure.microsoft.com",
          "ip": "40.107.23.54",
          "registered_domain": "microsoft.com",
          "subdomain": "azure",
          "top_level_domain": "com",
          "user": {
              "domain": "azure.microsoft.com",
              "email": "noreply@azure.microsoft.com",
              "id": "noreply@azure.microsoft.com",
              "name": "noreply"
          }
      },
      "tags": [
          "preserve_original_event",
          "microsoft-defender-endpoint",
          "forwarded"
      ]
  }
  ```
</dropdown>

<dropdown title="Exported fields">
  | Field                                           | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | Type             |
  |-------------------------------------------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------|
  | @timestamp                                      | Date/time when the event originated. This is the date/time extracted from the event, typically representing when the event was generated by the source. If the event source has no original timestamp, this value is typically populated by the first time the event was received by the pipeline. Required field for all events.                                                                                                                                                                                            | date             |
  | data_stream.dataset                             | The field can contain anything that makes sense to signify the source of the data. Examples include `nginx.access`, `prometheus`, `endpoint` etc. For data streams that otherwise fit, but that do not have dataset set we use the value "generic" for the dataset value. `event.dataset` should have the same value as `data_stream.dataset`. Beyond the Elasticsearch data stream naming criteria noted above, the `dataset` value has additional restrictions:   * Must not contain `-`   * No longer than 100 characters | constant_keyword |
  | data_stream.namespace                           | A user defined namespace. Namespaces are useful to allow grouping of data. Many users already organize their indices this way, and the data stream naming scheme now provides this best practice as a default. Many users will populate this field with `default`. If no value is used, it falls back to `default`. Beyond the Elasticsearch index naming criteria noted above, `namespace` value has the additional restrictions:   * Must not contain `-`   * No longer than 100 characters                                | constant_keyword |
  | data_stream.type                                | An overarching type for the data stream. Currently allowed values are "logs" and "metrics". We expect to also add "traces" and "synthetics" in the near future.                                                                                                                                                                                                                                                                                                                                                              | constant_keyword |
  | event.dataset                                   | Event dataset                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | constant_keyword |
  | input.type                                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | keyword          |
  | log.offset                                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | long             |
  | microsoft.online_message_trace.EndDate          | This field is used to limit the report period. Use this field in a $filter query option to set the end date and time of the reporting period. If you supply EndDate in the $filter option, you must also supply StartDate. In this report, this field corresponds to the date and time of the last processing step recorded for the message.                                                                                                                                                                                 | date_nanos       |
  | microsoft.online_message_trace.FromIP           | The IPv4 or IPv6 address that transmitted the message to the Office 365 email system.                                                                                                                                                                                                                                                                                                                                                                                                                                        | keyword          |
  | microsoft.online_message_trace.Index            |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | long             |
  | microsoft.online_message_trace.MessageId        | The Internet MessageID header of the message, if one was supplied. This value can also be explicitly null.</p></td>                                                                                                                                                                                                                                                                                                                                                                                                          | keyword          |
  | microsoft.online_message_trace.MessageTraceId   | An identifier used to get the detailed message transfer trace information.</p></td>                                                                                                                                                                                                                                                                                                                                                                                                                                          | keyword          |
  | microsoft.online_message_trace.Organization     | The fully qualified domain name that was processing the email.</p></td>                                                                                                                                                                                                                                                                                                                                                                                                                                                      | keyword          |
  | microsoft.online_message_trace.Received         | The date and time when the email was received by the Office 365 email system. This corresponds to the Date field of the first message trace detail entry.</p></td>                                                                                                                                                                                                                                                                                                                                                           | date_nanos       |
  | microsoft.online_message_trace.RecipientAddress | The SMTP email address of the user that the message was addressed to.</p></td>                                                                                                                                                                                                                                                                                                                                                                                                                                               | keyword          |
  | microsoft.online_message_trace.SenderAddress    | The SMTP email address of the user the message was purportedly from. Because sender addresses are commonly spoofed in spam email, they are not considered completely reliable.</p></td>                                                                                                                                                                                                                                                                                                                                      | keyword          |
  | microsoft.online_message_trace.Size             | The size of the message, in bytes.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | long             |
  | microsoft.online_message_trace.StartDate        | This field is used to limit the report period. Use this field in a $filter query option to set the start date and time of the reporting period. If you provide a StartDate in the $filter option, you must also specify an EndDate. In this report, this field corresponds to the date and time of the first processing step recorded for the message.</p></td>                                                                                                                                                              | date_nanos       |
  | microsoft.online_message_trace.Status           | The status of the message in the Office 365 email system. This corresponds to the Detail field of the last processing step recorded for the message.</p></td>                                                                                                                                                                                                                                                                                                                                                                | keyword          |
  | microsoft.online_message_trace.Subject          | The subject line of the message, if one was present for the message.</p></td>                                                                                                                                                                                                                                                                                                                                                                                                                                                | keyword          |
  | microsoft.online_message_trace.ToIP             | The IPv4 or IPv6 address that the Office 365 email system sent the message to.</p></td>                                                                                                                                                                                                                                                                                                                                                                                                                                      | keyword          |
</dropdown>


## Screenshots

This integration includes one or more Kibana dashboards that visualizes the data collected by the integration. The screenshots below illustrate how the ingested data is displayed.
<carousel>
  ![kibana dashboard](https://epr.elastic.co/package/microsoft_exchange_online_message_trace/1.29.1/img/dashboard.png)
</carousel>


## Changelog

<dropdown title="Changelog">
  | Version    | Details                                                                                                                                                                                                                                                                           | Minimum Kibana version |
  |------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------------|
  | **1.29.1** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/16620))Update the README's Logfile collection section.                                                                                                                                              | 9.1.49.0.78.19.4       |
  | **1.29.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15584))Prevent updating fleet health status to degraded.                                                                                                                                        | 9.1.49.0.78.19.4       |
  | **1.28.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15069))Improve documentation                                                                                                                                                                    | 9.0.08.13.0            |
  | **1.27.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12919))Update Kibana constraint to support 9.0.0.                                                                                                                                               | 9.0.08.13.0            |
  | **1.26.1** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/12781))Updated SSL description in package manifest.yml to be uniform and to include links to documentation.                                                                                         | 8.13.0                 |
  | **1.26.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12503))Allow the usage of deprecated log input and support for stack 9.0                                                                                                                        | 8.13.0                 |
  | **1.25.3** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/12446))Remove incorrect use of `event.start`, `event.end`.                                                                                                                                          | 8.13.0                 |
  | **1.25.2** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/12292))Fix handling of SenderAddress where the value is '<>'.                                                                                                                                       | 8.13.0                 |
  | **1.25.1** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/12239))Added the 'end_interval' input parameter to support sliding window logic with updates to some default values.                                                                                | 8.13.0                 |
  | **1.25.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12076))Do not remove `event.original` in main ingest pipeline.                                                                                                                                  | 8.13.0                 |
  | **1.24.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12046))Add "preserve_original_event" tag to documents with `event.kind` set to "pipeline_error".                                                                                                | 8.13.0                 |
  | **1.23.1** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11956))Improve document fingerprinting resolution.                                                                                                                                                  | 8.13.0                 |
  | **1.23.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10684))Add `event.category` and `event.type` to all events.                                                                                                                                     | 8.13.0                 |
  | **1.22.1** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/10547))Clarify configuration documentation.                                                                                                                                                         | 8.13.0                 |
  | **1.22.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10135))Update the kibana constraint to ^8.13.0. Modified the field definitions to remove ECS fields made redundant by the ecs@mappings component template.                                      | 8.13.0                 |
  | **1.21.2** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/10172))Fix template to not fail without local domains.                                                                                                                                              | 8.12.0                 |
  | **1.21.1** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/10136))Fix sample event.                                                                                                                                                                            | 8.12.0                 |
  | **1.21.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10120))Make `event.outcome` field conform to ECS field definition.                                                                                                                              | 8.12.0                 |
  | **1.20.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9974))Improve handling of empty responses.                                                                                                                                                      | 8.12.0                 |
  | **1.19.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9127))Set sensitive values as secret.                                                                                                                                                           | 8.12.0                 |
  | **1.18.1** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8943))Changed owners                                                                                                                                                                            | 8.7.1                  |
  | **1.18.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8717))Add ability to configure addition look-back time.**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8717))Fix issue with cursor potentially causing data gaps. | 8.7.1                  |
  | **1.17.2** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8635))Fix exclude_files pattern.                                                                                                                                                                    | 8.7.1                  |
  | **1.17.1** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8657))Updated httpjson template to fix a breaking issue due to an extra ']' present. Updated tests parameters to suit the change.                                                                   | 8.7.1                  |
  | **1.17.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8550))Updated httpjson template to parse data correctly, reworked the documentation and updated the default template values.                                                                    | 8.7.1                  |
  | **1.16.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8489))Limit request tracer log count to five.                                                                                                                                                   | 8.7.1                  |
  | **1.15.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8434))ECS version updated to 8.11.0.                                                                                                                                                            | 8.7.1                  |
  | **1.14.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8269))Improve 'event.original' check to avoid errors if set.                                                                                                                                    | 8.7.1                  |
  | **1.13.1** | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8130))Cleaned up documentation and fixed the default values for scope & login_url.                                                                                                                  | 8.7.1                  |
  | **1.13.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7919))ECS version updated to 8.10.0.                                                                                                                                                            | 8.7.1                  |
  | **1.12.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7883))The format_version in the package manifest changed from 2.11.0 to 3.0.0. Removed dotted YAML keys from package manifest. Added 'owner.type: elastic' to package manifest.                 | 8.7.1                  |
  | **1.11.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7789))Add tags.yml file so that integration's dashboards and saved searches are tagged with "Security Solution" and displayed in the Security Solution UI.                                      | 8.7.1                  |
  | **1.10.0** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7110))Update package to ECS 8.9.0.                                                                                                                                                              | 8.7.1                  |
  | **1.9.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6992))Document duration units.                                                                                                                                                                  | 8.7.1                  |
  | **1.8.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7128))Update package-spec to 2.9.0.                                                                                                                                                             | 8.7.1                  |
  | **1.7.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6931))Convert dashboards to Lens.                                                                                                                                                               | 8.7.1                  |
  | **1.6.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6706))Document valid duration units.                                                                                                                                                            | 8.7.1                  |
  | **1.5.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6661))Ensure event.kind is correctly set for pipeline errors.                                                                                                                                   | 8.7.1                  |
  | **1.4.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6327))Update package to ECS 8.8.0.                                                                                                                                                              | 8.7.1                  |
  | **1.3.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6115))Add a new flag to enable request tracing                                                                                                                                                  | 8.7.1                  |
  | **1.2.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5765))Update package to ECS 8.7.0.                                                                                                                                                              | 8.0.0                  |
  | **1.1.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5775))Deprecated basic auth fields and added support for OAuth2.                                                                                                                                | 8.0.0                  |
  | **1.0.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5677))Release Microsoft Exchange Online Message Trace as GA.                                                                                                                                    | 8.0.0                  |
  | **0.4.1**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5123))Added categories and/or subcategories.                                                                                                                                                    |                        |
  | **0.4.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5318))Added the ability to configure request.timeout**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/5318))Add Missing event.created and event.start ECS mappings  |                        |
  | **0.3.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5159))Retain original data in structured format.**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5159))Map fields to ECS.                                      |                        |
  | **0.2.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4576))Update package to ECS 8.6.0.                                                                                                                                                              |                        |
  | **0.1.0**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4285))Update package to ECS 8.5.0.                                                                                                                                                              |                        |
  | **0.0.1**  | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4135))Initial draft of the package for Microsoft Exchange Online Message Trace logs                                                                                                             |                        |
</dropdown>