﻿---
title: AWS Health
description: AWS Health metrics provide insights into the health of your AWS environment by monitoring various aspects such as open issues, scheduled maintenance events,...
url: https://www.elastic.co/docs/reference/integrations/aws/awshealth
products:
  - Elastic integrations
applies_to:
  - Serverless Observability projects: Generally available
  - Serverless Security projects: Generally available
  - Elastic Stack: Generally available since 9.2
---

# AWS Health
|                                                                            |                                |
|----------------------------------------------------------------------------|--------------------------------|
| **Version**                                                                | 6.1.0 ([View all](#changelog)) |
| **Subscription level**[What's this?](https://www.elastic.co/subscriptions) | Basic                          |
| **Ingestion method(s)**                                                    | API, AWS CloudWatch, AWS S3    |
| **Minimum Kibana version(s)**                                              | 9.2.18.19.4                    |

AWS Health metrics provide insights into the health of your AWS environment by monitoring various aspects such as open issues, scheduled maintenance events, security advisories, compliance status, notification counts, and service disruptions. These metrics help you proactively identify and address issues impacting your AWS resources, ensuring the reliability, security, and compliance of your infrastructure.

## Data streams

The AWS Health integration collects one type of data: metrics.
Metrics provide insight into the operational health of your AWS environment, including the status of AWS services, scheduled changes, and notifications about potential issues that could impact your resources. Metrics are gathered with the [AWS Health API](https://docs.aws.amazon.com/health/latest/APIReference/Welcome.html)
Check more details in the *Metrics reference*.

## What do I need to use this integration?

You need Elasticsearch for storing and searching your data and Kibana for visualizing and managing it.
You can use our hosted Elasticsearch Service on Elastic Cloud, which is recommended, or self-manage the Elastic Stack on your own hardware.
Before using any AWS integration you will need:
- **AWS Credentials** to connect with your AWS account.
- **AWS Permissions** to make sure the user you're using to connect has permission to share the relevant data.

For more details about these requirements, refer to the [AWS integration documentation](https://docs.elastic.co/integrations/aws#requirements).
To collect AWS Health metrics, you would need specific AWS permissions to access the necessary data. Here's a list of permissions required for an IAM user to collect AWS Health metrics:
- `health:DescribeAffectedEntities`
- `health:DescribeEventDetails`
- `health:DescribeEvents`


## Setup

If you want to collect data from two or more AWS services, consider using the **AWS** integration. When you configure the AWS integration, you can collect data from as many AWS services as you'd like.
For step-by-step instructions on how to set up an integration, refer to the
[Quickstart: Monitor hosts with Elastic Agent](https://www.elastic.co/docs/solutions/observability/get-started/quickstart-monitor-hosts-with-elastic-agent).

### Data stream specific configuration notes

`Period`:: (*string*) Reporting interval. Recommended value is `24h`.

## Metrics reference

The `awshealth` data stream collects AWS Health metrics from AWS.
<dropdown title="Example">
  ```json
  {
      "@timestamp": "2024-06-09T07:26:10.847Z",
      "agent": {
          "ephemeral_id": "4c4c2973-500d-46d4-b41f-e6e02bb792e4",
          "id": "f2c63d7a-e4e0-4646-b76b-b6ba80fb1be8",
          "name": "service-integration-dev-idc-1",
          "type": "metricbeat",
          "version": "8.14.0"
      },
      "aws": {
          "awshealth": {
              "affected_entities": [
                  {
                      "aws_account_id": "00000000000",
                      "entity_arn": "arn:aws:health:us-west-2:00000000000:entity/g1boW0WfWEdh7qa18fGj5UZaAEFeA_2Ol3YtjyQ1IWcI4=1g",
                      "entity_url": "",
                      "entity_value": "arn:aws:ecs:us-west-2:00000000000:cluster/EC2BasedCluster",
                      "last_updated_time": "2024-06-07T01:53:31.728Z"
                  },
                  {
                      "aws_account_id": "00000000000",
                      "entity_arn": "arn:aws:health:us-west-2:00000000000:entity/g1ct5TryWpLSufg8DM-B5Wp_lAicG_0BlW9Zuh1m2YZrg=1g",
                      "entity_url": "",
                      "entity_value": "arn:aws:ecs:us-west-2:00000000000:cluster/DEMOGO-ECS",
                      "last_updated_time": "2024-06-07T01:53:31.728Z"
                  },
                  {
                      "aws_account_id": "00000000000",
                      "entity_arn": "arn:aws:health:us-west-2:00000000000:entity/g1mkYYnoO_cHYLqqY-p5H5owH5HTnTTu3ZmqnieDfbf_o=1g",
                      "entity_url": "",
                      "entity_value": "arn:aws:ecs:us-west-2:00000000000:cluster/elasticAgentCluster",
                      "last_updated_time": "2024-06-07T01:53:31.728Z"
                  }
              ],
              "affected_entities_others": 0,
              "affected_entities_pending": 0,
              "affected_entities_resolved": 0,
              "event_arn": "arn:aws:health:us-west-2::event/ECS/AWS_ECS_SECURITY_NOTIFICATION/AWS_ECS_SECURITY_NOTIFICATION_de761fb8c443f11306d764547c7efb11ee0a33361f8f172d0ac6f6c4159cf646",
              "event_description": "We are following up on a previous notification regarding changes to the Docker version used in the Amazon Elastic Container Service (Amazon ECS) Optimized Amazon Machine Imagines (AMI) based on Amazon Linux 2 (AL2). If you are running the Amazon ECS-Optimized AL2023 AMIs, which are on the latest Docker version, you can disregard this message.\n\nDocker version 20.10.25 reached its end-of-life in December 2023 and will not receive any further security updates. We are changing the default version of Docker in the ECS-Optimized AL2 AMI to 25.0.3 with a staged release. This is a major version bump in Docker and contains backwards incompatible changes. Please make sure to test your application after applying this update.\n\nOn June 12, 2024, as communicated previously, we will disable automatic security updates at instance launch for ECS-Optimized AL2 AMIs. On June 19, 2024, we will release Docker 25.0.3 to the Amazon ECS Extras Repository without being tagged as a security advisory. This will allow customers who update in place, to manually update their instances with the command 'yum update docker' to complete any necessary acceptance testing. On July 11, 2024, we will release a new version of the ECS-Optimized AL2 AMI with Docker 25.0.3. For customers that choose not to update in place, we recommend moving to this AMI to get the update. ECS-Optimized AL2 AMI versions released between June 12, 2024 and July 11, 2024 will not receive the new Docker version and your action will be required to receive security updates to Docker. We recommend you update to an ECS-Optimized AL2 AMI released on July 11, 2024 or later.\n\nOn August 28, 2024, we will tag Docker 25.0.3 as a security update. This will trigger AMIs released prior to June 12, 2024 to automatically update the Docker version to Docker 25.0.3 at instance launch unless you have taken steps to disable this functionality [1].\n\nUpdating in place to Docker 25.0.3 will also update the ecs-init package to the latest available. Docker 25.0.3 is only compatible with ecs-init version 1.81 or later. As a best practice, and to ensure you are receiving the latest package versions and updates, we recommend customers update to the latest Amazon ECS-Optimized AMIs, which are released weekly.\n\nA list of your impacted resources can be found in the 'Affected resources' tab.\n\nIf you have any further questions, please contact AWS Support [2].\n\n[1] https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html\n[2] https://aws.amazon.com/support",
              "event_scope_code": "ACCOUNT_SPECIFIC",
              "event_type_category": "accountNotification",
              "event_type_code": "AWS_ECS_SECURITY_NOTIFICATION",
              "last_updated_time": "2024-06-07T05:25:47.913Z",
              "region": "us-west-2",
              "service": "ECS",
              "start_time": "2024-06-06T20:25:00.000Z",
              "status_code": "open"
          }
      },
      "cloud": {
          "account": {
              "id": "elastic-obs-integrations-dev"
          },
          "availability_zone": "asia-south1-c",
          "instance": {
              "id": "000000000000000000",
              "name": "service-integration-dev-idc-1"
          },
          "machine": {
              "type": "n1-standard-8"
          },
          "project": {
              "id": "elastic-obs-integrations-dev"
          },
          "provider": "gcp",
          "region": "asia-south1",
          "service": {
              "name": "GCE"
          }
      },
      "cloud.provider": "aws",
      "data_stream": {
          "dataset": "aws.awshealth",
          "namespace": "ep",
          "type": "metrics"
      },
      "ecs": {
          "version": "8.0.0"
      },
      "elastic_agent": {
          "id": "f2c63d7a-e4e0-4646-b76b-b6ba80fb1be8",
          "snapshot": false,
          "version": "8.14.0"
      },
      "event": {
          "agent_id_status": "verified",
          "dataset": "aws.awshealth",
          "duration": 11035786308,
          "ingested": "2024-06-09T07:26:31Z",
          "module": "aws"
      },
      "host": {
          "architecture": "x86_64",
          "containerized": false,
          "hostname": "service-integration-dev-idc-1",
          "id": "1bfc9b2d8959f75a520a3cb94cf035c8",
          "ip": [
              "10.160.0.4",
              "fe80::4001:aff:fea0:4",
              "192.168.0.1",
              "192.168.32.1",
              "fe80::7caf:a6ff:febd:c3a9",
              "172.22.0.1",
              "172.26.0.1",
              "172.20.0.1",
              "172.30.0.1",
              "172.29.0.1",
              "172.17.0.1",
              "fe80::42:a5ff:fe15:d63c",
              "172.1.0.1",
              "172.19.0.1",
              "172.27.0.1",
              "172.31.0.1",
              "192.168.80.1",
              "192.168.224.1",
              "172.23.0.1",
              "192.168.49.1",
              "fe80::42:beff:fe39:f457",
              "fe80::4c7d:f2ff:fece:af8e",
              "fe80::3cbb:3ff:fe9a:20db",
              "fe80::d4d6:36ff:fef2:6468",
              "fe80::cc52:29ff:fe19:ed74",
              "192.168.16.1",
              "fe80::42:2fff:fe16:b2ba",
              "fe80::1c7d:aff:fe87:6f65",
              "fe80::2861:f4ff:fe68:8801",
              "fe80::5c94:eaff:fea6:b831"
          ],
          "mac": [
              "02-42-0D-A6-43-C0",
              "02-42-23-32-CF-25",
              "02-42-27-90-E6-54",
              "02-42-2F-16-B2-BA",
              "02-42-34-10-CA-62",
              "02-42-4F-1D-94-1B",
              "02-42-50-2E-CB-58",
              "02-42-5D-42-F3-1D",
              "02-42-66-9B-25-B2",
              "02-42-99-B7-1B-26",
              "02-42-A5-15-D6-3C",
              "02-42-A6-68-F8-E9",
              "02-42-BE-39-F4-57",
              "02-42-CE-31-B7-A3",
              "02-42-E8-F3-CF-7A",
              "02-42-F1-35-B0-41",
              "02-42-F4-2F-0F-22",
              "1E-7D-0A-87-6F-65",
              "2A-61-F4-68-88-01",
              "3E-BB-03-9A-20-DB",
              "42-01-0A-A0-00-04",
              "4E-7D-F2-CE-AF-8E",
              "5E-94-EA-A6-B8-31",
              "7E-AF-A6-BD-C3-A9",
              "CE-52-29-19-ED-74",
              "D6-D6-36-F2-64-68"
          ],
          "name": "service-integration-dev-idc-1",
          "os": {
              "codename": "bionic",
              "family": "debian",
              "kernel": "5.4.0-1106-gcp",
              "name": "Ubuntu",
              "platform": "ubuntu",
              "type": "linux",
              "version": "18.04.6 LTS (Bionic Beaver)"
          }
      },
      "metricset": {
          "name": "awshealth",
          "period": 86400000
      },
      "service": {
          "type": "aws-health"
      }
  }
  ```
</dropdown>

<dropdown title="Exported fields">
  | Field                                             | Description                                                                                                                                                                                                                                                                                              | Type             | Metric Type |
  |---------------------------------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------|-------------|
  | @timestamp                                        | Event timestamp.                                                                                                                                                                                                                                                                                         | date             |             |
  | agent.id                                          | Unique identifier of this agent (if one exists). Example: For Beats this would be beat.id.                                                                                                                                                                                                               | keyword          |             |
  | aws.awshealth.affected_entities                   | The details of the affected entities related to the event.                                                                                                                                                                                                                                               | object           |             |
  | aws.awshealth.affected_entities.aws_account_id    | The Amazon Web Services account number that contains the affected entity.                                                                                                                                                                                                                                | keyword          |             |
  | aws.awshealth.affected_entities.entity_arn        | The unique identifier for the affected entities. The entity ARN has the format: `arn:aws:health:\<entity-region\>:\<aws-account\>:entity/\<entity-id\>`. For example, `arn:aws:health:us-east-1:111222333444:entity/AVh5GGT7ul1arKr1sE1K`.                                                               | keyword          |             |
  | aws.awshealth.affected_entities.entity_url        | The URL of the affected entity.                                                                                                                                                                                                                                                                          | keyword          |             |
  | aws.awshealth.affected_entities.entity_value      | The ID of the affected entity.                                                                                                                                                                                                                                                                           | keyword          |             |
  | aws.awshealth.affected_entities.last_updated_time | The last updated time of the affected entity.                                                                                                                                                                                                                                                            | date             |             |
  | aws.awshealth.affected_entities.status_code       | The most recent status of the entity affected by the event. The possible values include `IMPAIRED`, `UNIMPAIRED`, `UNKNOWN`, `PENDING`, `RESOLVED`.                                                                                                                                                      | keyword          |             |
  | aws.awshealth.affected_entities_others            | The number of affected resources related to the event whose status cannot be verified.                                                                                                                                                                                                                   | float            | gauge       |
  | aws.awshealth.affected_entities_pending           | The number of affected resources that may require action.                                                                                                                                                                                                                                                | float            | gauge       |
  | aws.awshealth.affected_entities_resolved          | The number of affected resources that do not require any action.                                                                                                                                                                                                                                         | float            | gauge       |
  | aws.awshealth.end_time                            | The date and time when the event ended. Some events may not have an end date.                                                                                                                                                                                                                            | date             |             |
  | aws.awshealth.event_arn                           | The unique identifier for the event. The event ARN has the format `arn:aws:health:\<event-region\>::event/\<SERVICE\>/\<EVENT_TYPE_CODE\>/\<EVENT_TYPE_PLUS_ID\>`. For example, `arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456`. | keyword          |             |
  | aws.awshealth.event_description                   | The detailed description of the event.                                                                                                                                                                                                                                                                   | text             |             |
  | aws.awshealth.event_scope_code                    | The `event_scope_code` parameter specifies whether the Health event is a public Amazon Web Service event or an account-specific event. Allowed values are `PUBLIC`, `ACCOUNT_SPECIFIC`, or `NONE`."                                                                                                      | keyword          |             |
  | aws.awshealth.event_type_category                 | The event type category code. The possible values are `issue`, `accountNotification`, `investigation` or `scheduledChange`.                                                                                                                                                                              | keyword          |             |
  | aws.awshealth.event_type_code                     | The unique identifier for the event type. The format is `AWS_\<SERVICE_DESCRIPTION\>`. For example, `AWS_EC2_SYSTEM_MAINTENANCE_EVENT`.                                                                                                                                                                  | keyword          |             |
  | aws.awshealth.last_updated_time                   | The most recent date and time when the event was updated.                                                                                                                                                                                                                                                | date             |             |
  | aws.awshealth.region                              | The Amazon Web Services Region name of the event.                                                                                                                                                                                                                                                        | keyword          |             |
  | aws.awshealth.service                             | The Amazon Web Service affected by the event. For example, EC2 or RDS.                                                                                                                                                                                                                                   | keyword          |             |
  | aws.awshealth.start_time                          | The date and time when the event began.                                                                                                                                                                                                                                                                  | date             |             |
  | aws.awshealth.status_code                         | The most recent status of the event. Possible values are `open`, `closed`, and `upcoming`.                                                                                                                                                                                                               | keyword          |             |
  | aws.linked_account.id                             | ID used to identify linked account.                                                                                                                                                                                                                                                                      | keyword          |             |
  | aws.linked_account.name                           | Name or alias used to identify linked account.                                                                                                                                                                                                                                                           | keyword          |             |
  | aws.tags                                          | Tag key value pairs from aws resources.                                                                                                                                                                                                                                                                  | flattened        |             |
  | cloud.account.id                                  | The cloud account or organization id used to identify different entities in a multi-tenant environment. Examples: AWS account id, Google Cloud ORG Id, or other unique identifier.                                                                                                                       | keyword          |             |
  | cloud.region                                      | Region in which this host, resource, or service is located.                                                                                                                                                                                                                                              | keyword          |             |
  | data_stream.dataset                               | Data stream dataset.                                                                                                                                                                                                                                                                                     | constant_keyword |             |
  | data_stream.namespace                             | Data stream namespace.                                                                                                                                                                                                                                                                                   | constant_keyword |             |
  | data_stream.type                                  | Data stream type.                                                                                                                                                                                                                                                                                        | constant_keyword |             |
</dropdown>


## Screenshots

This integration includes one or more Kibana dashboards that visualizes the data collected by the integration. The screenshots below illustrate how the ingested data is displayed.
<carousel>
  ![AWS Health overview](https://epr.elastic.co/package/aws/6.1.0/img/metricbeat-aws-awshealth.png)
</carousel>


## Changelog

<dropdown title="Changelog">
  | Version              | Details                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | Minimum Kibana version  |
  |----------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------------------|
  | **6.1.0**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/17084))Remove fixed value `event.dataset` mapping for lambda_logs data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 9.2.18.19.4             |
  | **6.0.0**            | **Breaking change** ([View pull request](https://github.com/elastic/integrations/pull/16133))Remove third-party pipeline for previously removed 'third-party REST API' input from 'AWS Cloudtrail'.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 9.2.18.19.4             |
  | **5.6.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/16991))Fix Cloudtrail's Lambda event parsing of `vpcConfig.securityGroupIds` and `vpcConfig.subnetIds` fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 9.2.18.19.4             |
  | **5.6.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/16755))Update AWS ALB Grok patterns to support the conn_trace_id field and parse malformed requests missing HTTP versions.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 9.2.18.19.4             |
  | **5.5.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/16905))Added support for extracting user name from AWS GovCloud STS ARNs and fixed management_event field type conversion.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 9.2.18.19.4             |
  | **5.5.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/16713))Add dashboards for AWS Web Application Firewall data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 9.2.18.19.4             |
  | **5.4.1**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/16695))S3 Access logs, improve S3KEY regex to handle special characters.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 9.2.18.19.4             |
  | **5.4.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15346))Add ec2_metrics, lambda, sqs and sns alert rule templates.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 9.2.18.19.4             |
  | **5.3.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/16122))Add `cookies` field in cloudfront logs datastream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 9.1.48.19.4             |
  | **5.3.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/16579))Added rate limiting and retry configuration for AWS Config data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 9.1.48.19.4             |
  | **5.2.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15601))Normalize user fields for AWS CloudTrail events.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 9.1.48.19.4             |
  | **5.1.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/16515))Set `event.kind` to `alert` on AWS WAF events.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 9.1.48.19.4             |
  | **5.0.0**            | **Breaking change** ([View pull request](https://github.com/elastic/integrations/pull/16195))Rebrand AWS Security Hub as AWS Security Hub CSPM to align with the vendor's current branding.This change involves updating the `observer.vendor` value to `AWS Security Hub CSPM`.                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 9.1.48.19.4             |
  | **4.7.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15929))Add type-specific entity fields to data stream.This enables better entity classification while maintaining backward compatibility with existing `actor.entity.id` and `target.entity.id` fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 9.1.48.19.4             |
  | **4.6.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/16098))Retain CloudTrail `sessionCredentialFromConsole` field as `aws.cloudtrail.session_credential_from_console`.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 9.1.48.19.4             |
  | **4.5.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15945))Prevent updating fleet health status to degraded when the HTTPJSON template value evaluation is empty.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 9.1.48.19.4             |
  | **4.4.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15279))Prefer set with copy_from.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 9.1.08.19.0             |
  | **4.3.2**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/15722))Update transform to filter out document containing an error.message from AWS Config, AWS Inspector, and AWS Security Hub latest indexes.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 9.1.08.19.0             |
  | **4.3.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/15737))Update the AWS CloudWatch documentation.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 9.1.08.19.0             |
  | **4.3.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15320))Improve documentation to align with new guidelines.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 9.1.08.19.0             |
  | **4.2.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15436))Add `user.name` to Cloudtrail's `UserAuthentication` events.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 9.1.08.19.0             |
  | **4.1.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15151))Add `vulnerability_workflow` and `misconfiguration_workflow` sub category labels.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 9.1.08.19.0             |
  | **4.0.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15230))Add latest transform to `AWS Config` and `AWS Inspector`.This enables support for extended protections for `AWS Config` and `AWS Inspector`.**Breaking change** ([View pull request](https://github.com/elastic/integrations/pull/15230))The latest transforms requires `transform` node and necessary permissions to use the transform.The transforms stores the latest copy of vulnerabilities and misconfigurations in the destination indices, which will require additional storage.Due to change in fingeprinting on source indices, duplicates occur on source indices of `AWS Config` and `AWS Inspector`.                                 | 9.1.08.19.0             |
  | **3.17.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/17064))Fix Cloudtrail's Lambda event parsing of `vpcConfig.securityGroupIds` and `vpcConfig.subnetIds` fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 9.0.08.18.08.17.48.16.6 |
  | **3.17.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/16998))Disable lastSync start_position configuration for CloudWatch as it's incompatible with versions below 9.x.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 9.0.08.18.08.17.48.16.6 |
  | **3.17.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15312))Enable Agentless deployment for AWS GuardDuty.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 9.0.08.18.08.17.48.16.6 |
  | **3.16.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15217))Map `recipient_account_id` to `cloud.account.id` for AWS CloudTrail.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 9.0.08.18.08.17.48.16.6 |
  | **3.15.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/15077))Add support for VPC Flow logs versions 6, 7, and 8.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 9.0.08.18.08.17.48.16.6 |
  | **3.14.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14995))Remove unused agent files.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 9.0.08.18.08.17.48.16.6 |
  | **3.14.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14958))Fixed issue where empty DescribeConfigRules responses caused 'index out of bounds' errors in AWS Config integration.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 9.0.08.18.08.17.48.16.6 |
  | **3.14.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14947))Prevent logging expected agent HTTP JSON template errors.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 9.0.08.18.08.17.48.16.6 |
  | **3.13.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14733))Add description on lastSync start_position configuration for CloudWatch.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 9.0.08.18.08.17.48.16.6 |
  | **3.13.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14734))Remove redundant code from config data stream agent configuration.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 9.0.08.18.08.17.48.16.6 |
  | **3.13.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14573))Remove Include Linked Accounts config option from AWS Health.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 9.0.08.18.08.17.48.16.6 |
  | **3.13.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14525))Avoid updating fleet health status to degraded when Guardduty has no findings.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 9.0.08.18.08.17.48.16.6 |
  | **3.13.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14441))Reduce unnecessary work done in cloudtrail data stream when flattened fields are not required.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 9.0.08.18.08.17.48.16.6 |
  | **3.12.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14306))Mapping changes in `inspector` datastream for Cloud Detection and Response (CDR) vulnerability workflow.Parse and map newly introduced fields in the `inspector` data stream.Enable request trace log removal and Agentless deployment in the `inspector` data stream.                                                                                                                                                                                                                                                                                                                                                                             | 9.0.08.18.08.17.48.16.6 |
  | **3.11.0**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/13345))Fix `tlsVersion` parsing when not properly defined in cloudtrail event.**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13345))Add empty value removal script to cloudtrail data stream ingest pipeline.                                                                                                                                                                                                                                                                                                                                                                                                              | 9.0.08.18.08.17.48.16.6 |
  | **3.10.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14439))Fix configuration template typo.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 9.0.08.18.08.17.48.16.6 |
  | **3.10.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14236))Allow user-specification of fields to retain in the cloudtrail data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 9.0.08.18.08.17.48.16.6 |
  | **3.9.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14354))Ingest managed insights from Security Hub.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 9.0.08.18.08.17.48.16.6 |
  | **3.8.2**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14264))Fix null reference in securityhub_findings* data streams when extracting `host.ip`.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 9.0.08.18.08.17.48.16.6 |
  | **3.8.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14251))Modify the data type of `event.kind` from a constant_keyword to a keyword to handle pipeline errors that send `event.kind` as `pipeline_error`.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 9.0.08.18.08.17.48.16.6 |
  | **3.8.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14137))Standardize user fields processing across integrations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 9.0.08.18.08.17.48.16.6 |
  | **3.7.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14172))Add temporary processor to remove the fields added by the Agentless policy.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 9.0.08.18.08.17.48.16.6 |
  | **3.7.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14181))Add AWS lambda logs overview dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 9.0.08.18.08.17.48.16.6 |
  | **3.6.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14070))Add AWS lambda logs dataset.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 9.0.08.18.08.17.48.16.6 |
  | **3.5.2**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/14179))Add Redshift `InstanceType` dimension.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 9.0.08.18.08.17.48.16.6 |
  | **3.5.1**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/14169))Change Redshift terraform node type because `dc2.large` is deprecated and leads to system tests fails.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 9.0.08.18.08.17.48.16.6 |
  | **3.5.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13956))Add the `external_id` field to data streams collecting data from S3.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 9.0.08.18.08.17.48.16.6 |
  | **3.4.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13830))Add new AWS Config datastream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 9.0.08.18.08.17.48.16.6 |
  | **3.3.3**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13978))Update README - Document ingested log types of AWS Network Firewall.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 9.0.08.18.08.17.48.16.6 |
  | **3.3.2**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13434))Update README - EC2 Instance IAM Role for AWS Authentication                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 9.0.08.18.08.17.48.16.6 |
  | **3.3.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/13676))Fix handling of duplicate fields in Network Firewall Logs data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 9.0.08.18.08.17.48.16.6 |
  | **3.3.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13246))Remove deprecated httpjson input for cloudtrail data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 9.0.08.18.08.17.48.16.6 |
  | **3.2.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13367))Add agentless deployment for AWS Security Hub.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 9.0.08.18.08.17.48.16.6 |
  | **3.1.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13542))Enhancements for Guardduty dashboards.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 9.0.08.18.08.17.48.16.6 |
  | **3.0.0**            | **Breaking change** ([View pull request](https://github.com/elastic/integrations/pull/13372))Add new Security Hub Findings Full Posture data stream. If you rely on Findings > Misconfigurations view, enable this new data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 9.0.08.18.08.17.48.16.6 |
  | **2.45.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13486))Update grok pattern for AWS S3 access ingest pipeline                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 9.0.08.16.5             |
  | **2.45.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/13350))Fix handling of SQS worker count configuration.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 9.0.08.16.5             |
  | **2.45.0**           | **Breaking change** ([View pull request](https://github.com/elastic/integrations/pull/13370))Update default data_stream.dataset to aws.cloudwatch_logs for cloudwatch_logs data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 9.0.08.16.5             |
  | **2.44.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12685))Add `actor.entity.id` and `target.entity.id`                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 9.0.08.16.5             |
  | **2.43.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/13099))Set `event.type` and `event.action` fields in vpcflow logs.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 9.0.08.16.5             |
  | **2.42.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12645))Add support to configure start_timestamp and ignore_older configurations for AWS S3 backed inputs.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 9.0.08.16.5             |
  | **2.41.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/12727))Updated SSL description to be uniform and to include links to documentation.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 9.0.08.16.2             |
  | **2.41.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12755))Ignore long `cloudtrail.request_parameters` and `cloudtrail.response_elements` fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 9.0.08.16.2             |
  | **2.40.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12637))Add support for Kibana `9.0.0`.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 9.0.08.16.2             |
  | **2.39.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12503))Allow the usage of deprecated log input and support for stack 9.0                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.16.2                  |
  | **2.38.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/12145))Update links to getting started docs                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.16.2                  |
  | **2.38.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12223))Add missing category.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.16.2                  |
  | **2.38.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12154))Add support for Access Point ARN when collecting logs via the AWS S3 Bucket.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.16.2                  |
  | **2.37.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11883))Map aws.dimensions as object instead of flattened in CloudWatch metrics.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.16.0                  |
  | **2.36.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12110))Include pipeline test examples to accommodate the new Cloudtrail format for the `CreateGroup` and `UpdateGroup` event types.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.16.0                  |
  | **2.36.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12072))Add SQS API calls documentation and required S3 permissions.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.16.0                  |
  | **2.36.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12030))Add ELB connection logs dashboards for application load balancers.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.16.0                  |
  | **2.35.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11933))Add the support for connection logs for AWS ELB dataset for Application Load Balancers.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 8.16.0                  |
  | **2.34.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11907))Add Lambda Event Source Mapping metrics and improve Lambda dashboard to display the new metrics.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.16.0                  |
  | **2.33.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11484))Add option to check linked accounts when using log group prefixes to derive matching log groups                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.16.0                  |
  | **2.32.0**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11829))Implemented grok processor based parsing for ipv6 & ipv4 addresses in the AWS CloudFront logs.**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11829))Auto formatted various text descriptions and newlines across all data streams via elastic-package.                                                                                                                                                                                                                                                                                                                                                              | 8.16.0                  |
  | **2.31.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11794))Update documentation with required permissions for AWS Inspector.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.16.0                  |
  | **2.31.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11654))Removed the `reducedTimeRange` filter from the AWS Billing Total Estimated Charges lens to ensure value is displayed.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.16.0                  |
  | **2.31.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11590))Add the support for listeners with ALPN policy extension in ELB dataset for Network Load Balancers.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.16.0                  |
  | **2.31.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11608))Add `cloud.provider`, `event.kind`, and `observer.vendor` fields to _source as needed by CDR workflows.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.16.0                  |
  | **2.31.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11158))Improve support for Cloud Detection and Response (CDR) workflows in securityhub_findings data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.16.0                  |
  | **2.30.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/12237))Map aws.dimensions as object instead of flattened in CloudWatch metrics (backported from 2.37.0)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.15.2                  |
  | **2.30.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11706))Add the support for listeners with ALPN policy extension in ELB dataset for Network Load Balancers.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.15.2                  |
  | **2.30.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11498))Update the AWS dashboard panels.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.15.2                  |
  | **2.30.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11442))Support configuring the Owning Account                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.15.2                  |
  | **2.29.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11392))Add mapping for the service.runtimeDetails fields in GuardDuty events.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.15.0                  |
  | **2.28.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11416))Add reference to AWS API requests and pricing information.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.15.0                  |
  | **2.27.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11112))Improve ingest pipeline error reporting.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.15.0                  |
  | **2.26.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11397))Add more data to related.entity field.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.15.0                  |
  | **2.26.0-preview01** | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/11115))Add related.entity field.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |                         |
  | **2.25.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11368))Add aws.metrics_names_fingerprint field and mark it as a dimension.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.14.0                  |
  | **2.25.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10888))Allow @custom pipeline access to event.original without setting preserve_original_event.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.14.0                  |
  | **2.24.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11369))Fix aws.metrics_names_fingerprint field.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.14.0                  |
  | **2.24.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/11281))Add aws.metrics_names_fingerprint.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.14.0                  |
  | **2.24.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/10216))Fixed and refactored AWS cloudfront log parsing.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.14.0                  |
  | **2.24.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10683))Add dot_expander processor into metrics ingest pipeline.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.14.0                  |
  | **2.23.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10444))Split the current AWS ELB dashboard into 3 separate dashboards, each focusing on a specific type of load balancer ELB, ALB, and NLB.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.14.0                  |
  | **2.22.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/10480))Update max_number_of_messages parameter description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.14.0                  |
  | **2.22.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10324))Add global dataset filter for dashboards to improve performance.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.14.0                  |
  | **2.21.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9249))Fix route53 public logs grok pattern.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.14.0                  |
  | **2.20.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10460))Add S3 polling option to data streams use aws-s3 input                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.14.0                  |
  | **2.19.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10488))Add a visualization panel to display the Inbound and Outbound traffic of Application load balancers.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.14.0                  |
  | **2.18.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/issues/10119))Add AWS Health integration.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.14.0                  |
  | **2.17.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10223))ECS version updated to 8.11.0. Update the kibana constraint to ^8.13.0. Modified the field definitions to remove ECS fields made redundant by the ecs@mappings component template.                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.13.0                  |
  | **2.16.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/10181))Add TargetResponseTime metric to ELB Application metrics.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.12.0                  |
  | **2.15.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/10082))Fix AWS Network Firewall title and description.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.12.0                  |
  | **2.15.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9865))Add endpoint + region variables to all SQS based AWS integrations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.12.0                  |
  | **2.15.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/9763))Fix AWS Cloudtrail resources field processing.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.12.0                  |
  | **2.15.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/9705))Don't index empty AWS Security Hub responses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.12.0                  |
  | **2.15.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9782))Adds async event age and drops metrics, and implements sum aggregation for existing lambda metrics.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.12.0                  |
  | **2.14.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/9669))Update aggregation function for AWS lambda invocation metric.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.12.0                  |
  | **2.14.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9290))Document billing data stream limitations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.12.0                  |
  | **2.14.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8414))Add ability to set processors and leader election on AWS Billing.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.12.0                  |
  | **2.13.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9346))Update latency parameter description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.12.0                  |
  | **2.13.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9133))Add Amazon MSK integration                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.12.0                  |
  | **2.12.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/9253))Fix an issue were the "_id" field was being used to aggregate data in Severity Over Time dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.12.0                  |
  | **2.12.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9213))Add cloudsecurity_cdr sub category label.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.12.0                  |
  | **2.12.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/9009))Enable 'secret' for the sensitive fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.12.0                  |
  | **2.11.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8882))Fix query range calculation for GuardDuty datastream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.10.2                  |
  | **2.11.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8811))Remove hardcoded event.dataset field and use ecs instead.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.10.2                  |
  | **2.11.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8703))Improve wording on milliseconds.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.10.2                  |
  | **2.11.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8509))Convert Total Estimated Charges panel to new metric visualization.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.10.2                  |
  | **2.10.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8709))Fix dimensions fingerprint field                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.10.2                  |
  | **2.10.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8635))Fix exclude_files pattern.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.10.2                  |
  | **2.10.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8513))Convert "AWS Redshift metrics overview" visualizations to new metric.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.10.2                  |
  | **2.9.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8521))Change SQS metrics statistic method, which includes changing ApproximateAgeOfOldestMessage from average to max, changing NumberOfMessagesDeleted, NumberOfEmptyReceives, NumberOfMessagesReceived and NumberOfMessagesSent from average to sum.                                                                                                                                                                                                                                                                                                                                                                                                         | 8.9.0                   |
  | **2.9.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8489))Limit request tracer log count to five.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.9.0                   |
  | **2.8.6**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8398))Add missing fields from beats input                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.9.0                   |
  | **2.8.5**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8332))Update donut charts with pie for better representation                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.9.0                   |
  | **2.8.4**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8140))Remove unused aws.*.metrics.*.* and aws.s3.bucket.name                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.9.0                   |
  | **2.8.3**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8115))Include documentation and mappings for subfields of dns.answers**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8115))Fix mapping for tags and dynamic metric fields                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.9.0                   |
  | **2.8.2**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7933))Add null checks and ignore_missing checks to the rename processor                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.9.0                   |
  | **2.8.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7659))Fix incorrect billing metrics displayed under AWS Billing overview dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.9.0                   |
  | **2.8.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8027))Allow configuration of TLD for guardduty, inspector, and security hub datastreams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.9.0                   |
  | **2.7.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7955))Add tags.yml file so that integration's dashboards and saved searches are tagged with "Security Solution" and displayed in the Security Solution UI.**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7955))Upgrade package spec to 3.0.0.**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7955))Fix duplicated and invalid field definitions.**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7955))Add missing dashboard filters.                                                                                                                       | 8.9.0                   |
  | **2.6.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7990))Fix AWS API Gateway logs dashboard lens                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.9.0                   |
  | **2.6.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/8002))ECS version updated to 8.10.0.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.9.0                   |
  | **2.5.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6352))Update Cloudtrail datastream to support tlsDetails field                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 8.9.0                   |
  | **2.4.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7900))Fix Security Hub Findings to abide by ECS allowed values.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.9.0                   |
  | **2.4.0**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7876))Add AWS API Gateway metrics dashboards for each API type and additional filters which ensure data consistency                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.9.0                   |
  | **2.3.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7856))Change include_linked_accounts default to true                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.9.0                   |
  | **2.2.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7785))Fix GuardDuty API call parameter.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.9.0                   |
  | **2.2.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7702))Add AWS API Gateway metrics dashboard Stage filter, control groups and clean up                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.9.0                   |
  | **2.1.2**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7675))Fix AWS API Gateway metrics dashboard                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.9.0                   |
  | **2.1.1**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7541))Improve AWS API Gateway dashboard                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.9.0                   |
  | **2.1.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7497))Enable TSDB by default for EC2 metrics data stream. This improves storage usage and query performance. For more details, see [https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html](https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html).                                                                                                                                                                                                                                                                                                                                                                | 8.9.0                   |
  | **2.0.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6247))Remove deprecated option for "Cloudwatch via S3"from the AWS CloudWatch integration. If you are using it take note that logs WON'T BE ingested via this route anymore once you update.                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.9.0                   |
  | **1.53.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7490))Set metric type in EC2 data stream fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.9.0                   |
  | **1.53.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7487))Add dimension fields to EC2 data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 8.9.0                   |
  | **1.53.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7425))Add missing fields definition for ec2                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.9.0                   |
  | **1.53.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/7331))Remove the remove processor since rename processor removes old field already.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.9.0                   |
  | **1.53.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7435))Disable TSDB on AWS Billing.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.9.0                   |
  | **1.53.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7151))Add AWS API Gateway custom acccess logging fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.9.0                   |
  | **1.52.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7129))Use default color for AWS dashboards metric charts.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.9.0                   |
  | **1.52.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7262))Enable TSDB by default for cloudwatch metrics data stream. This improves storage usage and query performance. For more details, see [https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html](https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html).                                                                                                                                                                                                                                                                                                                                                         | 8.9.0                   |
  | **1.51.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/8888))Remove hardcoded event.dataset field and use ecs instead.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.8.1                   |
  | **1.51.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7438))Disable TSDB on AWS Billing.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.8.1                   |
  | **1.51.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7327))Use object metric type for the cloudwatch metrics                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.8.1                   |
  | **1.51.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6936))Add standalone S3 option for vpcflow                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.8.1                   |
  | **1.50.6**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6999))Add metric_type metadata to the cloudwatch data_stream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.8.1                   |
  | **1.50.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6871))Migrate AWS Security Hub dashboards to lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.8.1                   |
  | **1.50.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6962))Migrate AWS VPC dashboard visualizations to lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.8.1                   |
  | **1.50.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/7075))Add EMR logs dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.8.1                   |
  | **1.50.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6644))Migrate AWS Billing dashboard visualizations to lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.8.1                   |
  | **1.50.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6683))Add AWS API Gateway logs dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.8.1                   |
  | **1.50.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6895))Add EMR logs data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.8.1                   |
  | **1.49.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6530))Add API Gateway logs datastream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.8.1                   |
  | **1.48.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6915))Adding missing fields for the CloudTrail datastream - add option for standalone S3 bucket                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.8.1                   |
  | **1.47.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6865))Migrate AWS Redshift dashboard input controls.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.8.1                   |
  | **1.47.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6487))Migrate AWS S3 Server Access Log Overview dashboard visualizations to lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.8.1                   |
  | **1.46.9**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6995))Migrate AWS Network Firewall dashboard input controls.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.8.1                   |
  | **1.46.8**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6827))Add dimensions metadata to the cloudwatch data_stream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.8.1                   |
  | **1.46.7**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6969))Enable time series data streams for the API Gateway and EMR data streams. This improves storage usage and query performance. For more details, see [https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html](https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html).                                                                                                                                                                                                                                                                                                                                          | 8.8.1                   |
  | **1.46.6**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6964))Update metric type and set dimension fields for AWS EMR data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.8.1                   |
  | **1.46.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6952))Fix metric type for API Gateway metric fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.8.1                   |
  | **1.46.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6950))Set dimensions fields for API Gateway data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.46.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6916))Add missing S3 fields for vpcflow                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.8.1                   |
  | **1.46.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6887))Enable time series data streams for the S3 daily storage and S3 request datasets. This improves storage usage and query performance. For more details, see [https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html](https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html).                                                                                                                                                                                                                                                                                                                                  | 8.8.1                   |
  | **1.46.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6886))Enable time series data streams for the Usage dataset. This improves storage usage and query performance. For more details, see [https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html](https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html).                                                                                                                                                                                                                                                                                                                                                             | 8.8.1                   |
  | **1.46.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6782))Enable time series data streams for the metrics datasets Billing, DynamoDB, EBS, ECS, ELB, Firewall, Kinesis, Lambda, NAT gateway, RDS, Redshift, S3 Storage Lens, SNS, SQS, Transit Gateway and VPN. This improves storage usage and query performance. For more details, see [https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html](https://www.elastic.co/guide/en/elasticsearch/reference/current/tsds.html).                                                                                                                                                                                                              | 8.8.1                   |
  | **1.45.9**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6749))Add new fingerprint dimension to AWS Billing.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.8.1                   |
  | **1.45.8**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6667))Add metric_type metadata to s3_daily_storage and s3_request data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.8.1                   |
  | **1.45.7**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6666))Add dimension fields metadata to s3_request and s3_data_storage data streams to support TSDB                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.8.1                   |
  | **1.45.6**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6584))Add metric type to S3 Storage Lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.8.1                   |
  | **1.45.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6583))Set dimension fields for S3 Storage Lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.8.1                   |
  | **1.45.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/6648))Remove aws.dimensions.* from package-fields.yml                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.8.1                   |
  | **1.45.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6606))Add AWS EMR metrics dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.8.1                   |
  | **1.45.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6476))Add AWS API Gateway dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.8.1                   |
  | **1.45.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6120))Add AWS EMR metrics data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.8.1                   |
  | **1.44.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6543))Migrate AWS Metric Overview dashboard visualizations to lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.8.1                   |
  | **1.44.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6521))Migrate AWS ELB Access Log dashboard visualizations to lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.8.1                   |
  | **1.44.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/6561))Fix image link in readme                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.8.1                   |
  | **1.44.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5651))Migrate AWS TransitGateway metrics dashboard to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.8.1                   |
  | **1.44.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6340))Add permissions to reroute events to logs-*-* for cloudwatch_logs and ec2_logs datastream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.8.1                   |
  | **1.43.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6231))Add documentation for latency parameter                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.8.1                   |
  | **1.43.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6526))Add tags_filter and include_linked_accounts config parameter in missing metric data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.8.1                   |
  | **1.43.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6459))Add include_linked_accounts config parameter for metrics data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.8.1                   |
  | **1.42.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6421))Add field agent.id to be set as dimension for TSDB migration.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.7.1                   |
  | **1.41.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5426))Migrate AWS NATGateway metrics dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.7.1                   |
  | **1.40.9**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5643))Migrate AWS ELB metrics dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.7.1                   |
  | **1.40.8**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5626))Migrate EC2 metrics dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.7.1                   |
  | **1.40.7**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5650))Add AWS Firewall metrics dashboard input control groups.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 8.7.1                   |
  | **1.40.6**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6455))Migrate AWS S3 Storage Lens dashboard visualizations to lens.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.7.1                   |
  | **1.40.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6368))Migrate Usage Overview dashboard to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.7.1                   |
  | **1.40.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6374))Migrate AWS CloudTrail dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.7.1                   |
  | **1.40.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6296))Add fields metric type to usage, dynamoDB and ELB data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.7.1                   |
  | **1.40.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6440))Replace aws.rds.db_instance.identifier with aws.dimensions.DBInstanceIdentifier in RDS dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.7.1                   |
  | **1.40.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6480))Add link to main AWS requirements in all integrations page.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.7.1                   |
  | **1.40.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6319))Add metric type to SNS, SQS and Billing data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.7.1                   |
  | **1.39.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6188))Add AWS API Gateway data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.7.1                   |
  | **1.38.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6311))Add dimension fields to billing, sns and sqs data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.7.1                   |
  | **1.38.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6339))Add dimension fields to firewall, transit gateway and vpn data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.7.1                   |
  | **1.38.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6365))Add metric type to vpn, firewall and transit gateway data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.7.1                   |
  | **1.38.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6094))Add metric type to RDS data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.7.1                   |
  | **1.38.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6079))Add dimensions to RDS data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.7.1                   |
  | **1.37.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/6353))Fix incorrect fields on multiple visualizations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.7.1                   |
  | **1.37.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5640))Migrate AWS RDS metrics dashboard to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.7.1                   |
  | **1.37.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5642))Migrate AWS SNS dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.7.1                   |
  | **1.37.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5644))Migrate AWS SQS metrics dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.7.1                   |
  | **1.36.9**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5649))Migrate AWS VPN metrics dashboard to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.7.1                   |
  | **1.36.8**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6213))Add dimension fields to usage, dynamoDB and ELB data streams.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.7.1                   |
  | **1.36.7**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5973))Add dimension fields to Lambda data stream for TSDB support.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.7.1                   |
  | **1.36.6**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6105))Add metric type to natgateway data stream fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.7.1                   |
  | **1.36.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6063))Add metric type to EBS fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.7.1                   |
  | **1.36.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5866))Add support for TSDB on kinesis data stream (metric type).                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.7.1                   |
  | **1.36.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6083))Add dimensions to Redshift data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.7.1                   |
  | **1.36.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6087))Add metric type mapping to Redshift data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.7.1                   |
  | **1.36.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6102))Add dimension fields to natgateway data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.7.1                   |
  | **1.36.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6045))Add metric type to Lambda fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.7.1                   |
  | **1.35.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/6213))Fix typo in field name causing erroneous timestamp detection on the s3access data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.7.1                   |
  | **1.35.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6161))Add a new flag to enable request tracing on `httpjson` based input                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.7.1                   |
  | **1.34.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5455))Migrate AWS Lambda metrics dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.6.0                   |
  | **1.34.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5288))Migrate AWS DynamoDB metrics dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.6.0                   |
  | **1.34.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5968))Add field metric type to ECS data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.6.0                   |
  | **1.34.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5891))Add dimension fields to Kinesis datastream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.6.0                   |
  | **1.34.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5895))Add dimension fields to ECS datastream for TSDB support.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | 8.6.0                   |
  | **1.34.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/6062))Add dimensions to EBS data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.6.0                   |
  | **1.33.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5794))Add number_of_workers and latency to all CloudWatch Logs based integrations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.6.0                   |
  | **1.33.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/5791))Add missing permissions in the AWS Billing integration documentation.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.6.0                   |
  | **1.33.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/5790))Add missing permissions in the AWS CloudWatch Logs integration documentation.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.6.0                   |
  | **1.33.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5777))Add latency configuration option on the CloudWatch Logs integration.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.6.0                   |
  | **1.32.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/5646))Fix a minor documentation format issue.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.6.0                   |
  | **1.32.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5123))Added categories and/or subcategories.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.6.0                   |
  | **1.32.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5212))Migrate AWS EBS dashboard visualizations to lenses.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.6.0                   |
  | **1.31.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4915))Add a data stream for Amazon GuardDuty.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.6.0                   |
  | **1.30.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/5178))Add dashboards data streams filters.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.6.0                   |
  | **1.29.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/5017))Drop comments from CloudFront loglines                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.6.0                   |
  | **1.29.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4515))Add data_granularity parameter and rename period title to Collection Period.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.6.0                   |
  | **1.28.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4857))Remove quotes from VPC flow log message field and move dot_expander processor to top                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.4.0                   |
  | **1.28.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4830))Add dot_expander processor to expand all fields with dot into object fields**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4830))Support VPC flow log with message field                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.4.0                   |
  | **1.28.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4794))Adjust kinesis integration to kinesis data stream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.4.0                   |
  | **1.28.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4641))Enhance S3 integration dashboard                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.4.0                   |
  | **1.27.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4676))Support multiple forwarded IPs in cloudfront integration                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.4.0                   |
  | **1.27.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4703))Update the pagination termination condition.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | 8.4.0                   |
  | **1.27.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4596))Added a Summary Dashboard for AWS Security Hub.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.4.0                   |
  | **1.27.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4604))Add Inspector data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.4.0                   |
  | **1.25.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4657))Remove duplicate fields from agent.yml and use ecs.yml for ECS fields                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.3.0                   |
  | **1.25.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4567/))Update ec2 fields.yml doc                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | 8.3.0                   |
  | **1.25.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4592))Remove duplicate 'content_type' config that causes errors while configurating the integration.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.3.0                   |
  | **1.25.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4532))Force content type where json content is expected                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.3.0                   |
  | **1.24.6**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4551))Enhance Kinesis integration dashboard                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.3.0                   |
  | **1.24.5**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4437))Allow adding multiple processors in cloudfront logs.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.3.0                   |
  | **1.24.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4430))Do not rely on dynamodb lightweight module metricset.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.3.0                   |
  | **1.24.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4395))Fix adding processors in cloudfront logs.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.3.0                   |
  | **1.24.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4308))Fix billing datastream agent template.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.3.0                   |
  | **1.24.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4281))Fix aws.cloudtrail.request_id parsing                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.3.0                   |
  | **1.24.0**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4158))Expose Default Region setting to UI                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.3.0                   |
  | **1.23.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4103))Set default endpoint to empty string                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 8.3.0                   |
  | **1.23.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4083))Fix Billing Dashboard                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.3.0                   |
  | **1.23.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4057))Fix EC2 dashboard                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.3.0                   |
  | **1.23.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3907))Update all AWS documentation.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.1.0                   |
  | **1.23.0**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/4049))Fix file.path field in cloudtrail data stream to use json.digestS3Object                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.1.0                   |
  | **1.22.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/4024))Update `cloud.region` parsing                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 8.1.0                   |
  | **1.21.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3553))Add Security Hub Findings and Insights data streams                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.1.0                   |
  | **1.20.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3688))Improve dashboards by removing individual visualizations from library                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.1.0                   |
  | **1.19.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3856))Move ebs metrics config from beats to integrations                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.0.07.15.0             |
  | **1.19.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3881))Fix proxy URL documentation rendering.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.0.07.15.0             |
  | **1.19.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3874))Update sample_event.json in kinesis data stream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.0.07.15.0             |
  | **1.19.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3873))Move NATGateway metrics config from beats to integrations                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.0.07.15.0             |
  | **1.19.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3840))Move Transit Gateway metrics config from beats to integrations                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.0.07.15.0             |
  | **1.19.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3829))Add Kinesis metrics datastream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.0.07.15.0             |
  | **1.18.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3838))Move s3_request metrics config from beats to integrations**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3838))Move s3_daily_storage metrics config from beats to integrations**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3838))Move SQS metrics config from beats to integrations**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3838))Move SNS metrics config from beats to integrations**Enhancement** ([View pull request](https://github.com/elastic/integrations/issues/3599))Move lambda metrics config from beats to integrations | 8.0.07.15.0             |
  | **1.18.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3860))Release AWS billing integration as GA                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.0.07.15.0             |
  | **1.18.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3799))Add ECS metricset**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3799))Fix incorrect fields on multiple visualizations                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.0.07.15.0             |
  | **1.17.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3851))Release Amazon Redshift integration as GA                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.0.07.15.0             |
  | **1.17.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3844))Fix data_stream.dataset indentation on cloudwatch_logs integration                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.0.07.15.0             |
  | **1.17.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3795))Add missing endpoint config to metrics datasets.**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3764))Move usage metrics config from beats to integrations**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3764))Move dynamodb metrics config from beats to integrations                                                                                                                                                                                                                                                                                                             | 8.0.07.15.0             |
  | **1.17.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3682))Improve support for event.original field from upstream forwarders.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.0.07.15.0             |
  | **1.17.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3610))Fix misspelling of Log Stream Prefix variable in manifest for aws-cloudwatch input                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 8.0.07.15.0             |
  | **1.17.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3569))Added Redshift integration                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | 8.0.07.15.0             |
  | **1.16.6**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3308))Update documentation with additional context for new users.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 8.0.07.15.0             |
  | **1.16.5**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3474))Move ELB metrics config from beats to integrations                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.16.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3499))Fix ELB dataset to parse URLs with spaces**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3499))Upgrade ECS to 8.2.0                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.0.07.15.0             |
  | **1.16.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3467))Move RDS metrics config from beats to integrations                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.16.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3357))Move EC2 metrics config from beats to integrations                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.16.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3342))Fix invalid values for ECS fields in vpcflow                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |                         |
  | **1.16.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3305))Move VPN configuration file into integrations and add tag collection                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 8.0.07.15.0             |
  | **1.15.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3304))Deprecate s3 input in cloudwatch integration**Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3304))Improve description for cloudwatch integration                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.14.8**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3285))Fix http.response.status_code to accept 000                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | 8.0.07.15.0             |
  | **1.14.7**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3253))Fix aws.dimensions.* for rds data stream**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3253))Fix aws.dimensions.* for sns data stream**Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3253))Add aws.dimensions.* for dynamodb data stream                                                                                                                                                                                                                                                                                                                                                   | 8.0.07.15.0             |
  | **1.14.6**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/3240))Improve s3 integration tile title and description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |                         |
  | **1.14.5**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3226))Fix duplicate titles for integrations                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.0.07.15.0             |
  | **1.14.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/3192))Fix cloudfront integration grok pattern                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |                         |
  | **1.14.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2912))Add new pattern to VPC Flow logs including all 29 v5 fields                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |                         |
  | **1.14.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2946))Fix billing dashboard.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.14.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2916))Add documentation for multi-fields                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.14.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2790))Add configuration for max_number_of_messages to the aws.firewall_logs S3 input.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.0.07.15.0             |
  | **1.13.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2838))Fix metricbeat- reference in dashboard                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.0.07.15.0             |
  | **1.13.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2767))Compress dashboard screenshots.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 8.0.07.15.0             |
  | **1.12.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2687))Fix field mapping conflicts in the elb_logs data stream relating to ECS fields (`trace.id`, `source.port`, and a few others).                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.0.07.15.0             |
  | **1.12.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2476))Add CloudFront Logs Datastream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |                         |
  | **1.11.4**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2470))Add Ingest Pipeline script to map IANA Protocol Numbers                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |                         |
  | **1.11.3**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2642))Changing missing ecs versions to 8.0.0                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.11.2**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2560))Add data_stream.dataset option for custom aws-cloudwatch log input                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |                         |
  | **1.11.1**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2635))Update permission list                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.11.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2478))Update to ECS 8.0                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 8.0.07.15.0             |
  | **1.10.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2536))Change cloudwatch metrics and logs default to false                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 8.0.07.15.0             |
  | **1.10.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2530))Add description of supported vpcflow formats                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |                         |
  | **1.10.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2323))Add cloudwatch input into AWS package for log collection                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |                         |
  | **1.9.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2341))Add Route 53 Resolver Logs Datastream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | 8.0.07.15.0             |
  | **1.8.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2316))Add Route 53 Public Zone Logs Datastream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |                         |
  | **1.7.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2339))Regenerate test files using the new GeoIP database                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |                         |
  | **1.7.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2199))Add integration for AWS Network Firewall                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |                         |
  | **1.6.2**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/2327))Change test public IPs to the supported subset                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |                         |
  | **1.6.1**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2325))Fix the value of event.created in CloudTrail data stream.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.0.07.15.0             |
  | **1.6.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2299))Add max_number_of_messages config option to AWS S3 input config.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |                         |
  | **1.5.1**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2282))Add missing sample events                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 8.0.07.15.0             |
  | **1.5.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2206))Support Kibana 8.0                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 8.0.07.15.0             |
  | **1.4.1**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2153))Add Overview dashboard for AWS S3 Storage Lens                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | 7.15.0                  |
  | **1.4.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2142))Add integration for AWS S3 Storage Lens                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |                         |
  | **1.3.2**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/2003))Uniform with guidelines                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |                         |
  | **1.3.1**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1917))Add config parameter descriptions                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |                         |
  | **1.3.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1886))Add WAF datastream                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **1.2.2**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1871))Prevent pipeline script error                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |                         |
  | **1.2.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1797))Fix logic that checks for the 'forwarded' tag                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |                         |
  | **1.2.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1687))Update to ECS 1.12.0                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |                         |
  | **1.1.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1772))vpcflow sync with filebeat fileset                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 7.14.0                  |
  | **1.0.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1623))Release AWS as GA                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | 7.14.0                  |
  | **0.10.7**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1648))Add proxy config                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |                         |
  | **0.10.6**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1606))Fix aws.billing.EstimatedCharges field name                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |                         |
  | **0.10.5**           | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1590))Add event.created field                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |                         |
  | **0.10.4**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1449))Improve RDS dashboard                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |                         |
  | **0.10.3**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1465))Convert to generated ECS fields                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |                         |
  | **0.10.2**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1371))update to ECS 1.11.0                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |                         |
  | **0.10.1**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1405))Escape special characters in docs                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |                         |
  | **0.10.0**           | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1364))Update integration description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |                         |
  | **0.9.3**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1357))Fix categories for each policy template                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |                         |
  | **0.9.2**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1334))Add linked account information into billing metricset                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |                         |
  | **0.9.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1333))Fix `aws.s3access` pipeline when remote IP is a `-`                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |                         |
  | **0.9.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1320))Change default credential options to access keys                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |                         |
  | **0.8.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1245))Set "event.module" and "event.dataset"                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |                         |
  | **0.7.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/767))Introduce granularity using input_groups                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |                         |
  | **0.6.4**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1147))Add support for Splunk authorization tokens                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |                         |
  | **0.6.3**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1201))Fix bug in Third Party ingest pipeline                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **0.6.2**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1212))Removed incorrect `http.request.referrer` field from elb logs                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |                         |
  | **0.6.1**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1079))Add support for CloudTrail Digest & Insight logs                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |                         |
  | **0.6.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/1070))Update ECS version, add event.original and preparing for package GA                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |                         |
  | **0.5.6**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/1000))Fix stack compatability                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |                         |
  | **0.5.5**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/979))Allow role_arn work with access keys for AWS                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |                         |
  | **0.5.4**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/631))Rename s3 input to aws-s3.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |                         |
  | **0.5.3**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/919))Add missing "geo" fields                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |                         |
  | **0.5.2**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/834))update to ECS 1.9.0                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |                         |
  | **0.5.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/791))Ignore missing "json" field in ingest pipeline                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |                         |
  | **0.5.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/782))Moving edge processors to ingest pipeline                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |                         |
  | **0.4.2**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/766))Updating package owner                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |                         |
  | **0.4.1**            | **Bug fix** ([View pull request](https://github.com/elastic/integrations/pull/754))Correct sample event file.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |                         |
  | **0.4.0**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/721))Add changes to use ECS 1.8 fields.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |                         |
  | **0.0.3**            | **Enhancement** ([View pull request](https://github.com/elastic/integrations/pull/21))initial release                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |                         |
</dropdown>