﻿---
title: Script processor
description: Runs an inline or stored script on incoming documents. The script runs in the ingest context. The script processor uses the script cache to avoid recompiling...
url: https://www.elastic.co/docs/reference/enrich-processor/script-processor
products:
  - Elasticsearch
---

# Script processor
Runs an inline or stored [script](https://www.elastic.co/docs/explore-analyze/scripting) on incoming documents. The script runs in the [`ingest`](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-ingest-processor-context) context.
The script processor uses the [script cache](https://www.elastic.co/docs/explore-analyze/scripting/scripts-search-speed) to avoid recompiling the script for each incoming document. To improve performance, ensure the script cache is properly sized before using a script processor in production.


| Name             | Required | Default    | Description                                                                                                                                                                                                 |
|------------------|----------|------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `lang`           | no       | "painless" | [Script language](https://www.elastic.co/docs/explore-analyze/scripting#scripting-available-languages).                                                                                                     |
| `id`             | no       | -          | ID of a [stored script](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-put-script).                                         If no `source` is specified, this parameter is required. |
| `source`         | no       | -          | Inline script.                                         If no `id` is specified, this parameter is required.                                                                                                 |
| `params`         | no       | -          | Object containing parameters for the script.                                                                                                                                                                |
| `description`    | no       | -          | Description of the processor. Useful for describing the purpose of the processor or its configuration.                                                                                                      |
| `if`             | no       | -          | Conditionally execute the processor. See [Conditionally run a processor](https://www.elastic.co/docs/manage-data/ingest/transform-enrich/ingest-pipelines#conditionally-run-processor).                     |
| `ignore_failure` | no       | `false`    | Ignore failures for the processor. See [Handling pipeline failures](https://www.elastic.co/docs/manage-data/ingest/transform-enrich/ingest-pipelines#handling-pipeline-failures).                           |
| `on_failure`     | no       | -          | Handle failures for the processor. See [Handling pipeline failures](https://www.elastic.co/docs/manage-data/ingest/transform-enrich/ingest-pipelines#handling-pipeline-failures).                           |
| `tag`            | no       | -          | Identifier for the processor. Useful for debugging and metrics.                                                                                                                                             |


## Access source fields

The script processor parses each incoming document’s JSON source fields into a set of maps, lists, and primitives. To access these fields with a Painless script, use the [map access operator](/docs/reference/scripting-languages/painless/painless-operators-reference#map-access-operator): `ctx['my-field']`. You can also use the shorthand `ctx.<my-field>` syntax.
<note>
  The script processor does not support the `ctx['_source']['my-field']` or `ctx._source.<my-field>` syntaxes.
</note>

The following processor uses a Painless script to extract the `tags` field from the `env` source field.
```json

{
  "pipeline": {
    "processors": [
      {
        "script": {
          "description": "Extract 'tags' from 'env' field",
          "lang": "painless",
          "source": """
            String[] envSplit = ctx['env'].splitOnToken(params['delimiter']);
            ArrayList tags = new ArrayList();
            tags.add(envSplit[params['position']].trim());
            ctx['tags'] = tags;
          """,
          "params": {
            "delimiter": "-",
            "position": 1
          }
        }
      }
    ]
  },
  "docs": [
    {
      "_source": {
        "env": "es01-prod"
      }
    }
  ]
}
```

The processor produces:
```json
{
  "docs": [
    {
      "doc": {
        ...
        "_source": {
          "env": "es01-prod",
          "tags": [
            "prod"
          ]
        }
      }
    }
  ]
}
```


## Access metadata fields

You can also use a script processor to access metadata fields. The following processor uses a Painless script to set an incoming document’s `_index`.
```json

{
  "pipeline": {
    "processors": [
      {
        "script": {
          "description": "Set index based on `lang` field and `dataset` param",
          "lang": "painless",
          "source": """
            ctx['_index'] = ctx['lang'] + '-' + params['dataset'];
          """,
          "params": {
            "dataset": "catalog"
          }
        }
      }
    ]
  },
  "docs": [
    {
      "_index": "generic-index",
      "_source": {
        "lang": "fr"
      }
    }
  ]
}
```

The processor changes the document’s `_index` to `fr-catalog` from `generic-index`.
```json
{
  "docs": [
    {
      "doc": {
        ...
        "_index": "fr-catalog",
        "_source": {
          "lang": "fr"
        }
      }
    }
  ]
}
```