﻿---
title: Text field type
description: A field to index full-text values, such as the body of an email or the description of a product. These fields are analyzed, that is they are passed through...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/text
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Text field type
A field to index full-text values, such as the body of an email or the description of a product. These fields are `analyzed`, that is they are passed through an [analyzer](https://www.elastic.co/docs/manage-data/data-store/text-analysis) to convert the string into a list of individual terms before being indexed. The analysis process allows Elasticsearch to search for individual words *within* each full text field. Text fields are not used for sorting and seldom used for aggregations (although the [significant text aggregation](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-significanttext-aggregation) is a notable exception).
`text` fields are best suited for unstructured but human-readable content. If you need to index unstructured machine-generated content, see [Mapping unstructured content](/docs/reference/elasticsearch/mapping-reference/keyword#mapping-unstructured-content).
If you need to index structured content such as email addresses, hostnames, status codes, or tags, it is likely that you should rather use a [`keyword`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/keyword) field.
Below is an example of a mapping for a text field:
```json

{
  "mappings": {
    "properties": {
      "full_name": {
        "type":  "text"
      }
    }
  }
}
```


## Use a field as both text and keyword

Sometimes it is useful to have both a full text (`text`) and a keyword (`keyword`) version of the same field: one for full text search and the other for aggregations and sorting. This can be achieved with [multi-fields](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/multi-fields).

## Parameters for text fields

The following parameters are accepted by `text` fields:
<definitions>
  <definition term="analyzer">
    The [analyzer](https://www.elastic.co/docs/manage-data/data-store/text-analysis) which should be used for the `text` field, both at index-time and at search-time (unless overridden by the  [`search_analyzer`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/search-analyzer)). Defaults to the default index analyzer, or the [`standard` analyzer](https://www.elastic.co/docs/reference/text-analysis/analysis-standard-analyzer).
  </definition>
  <definition term="eager_global_ordinals">
    Should global ordinals be loaded eagerly on refresh? Accepts `true` or `false` (default). Enabling this is a good idea on fields that are frequently used for (significant) terms aggregations.
  </definition>
  <definition term="fielddata">
    Can the field use in-memory fielddata for sorting, aggregations, or scripting? Accepts `true` or `false` (default).
  </definition>
  <definition term="fielddata_frequency_filter">
    Expert settings which allow to decide which values to load in memory when `fielddata` is enabled. By default all values are loaded.
  </definition>
  <definition term="fields">
    Multi-fields allow the same string value to be indexed in multiple ways for different purposes, such as one field for search and a multi-field for sorting and aggregations, or the same string value analyzed by different analyzers.
  </definition>
  <definition term="index">
    Should the field be searchable? Accepts `true` (default) or `false`.
  </definition>
  <definition term="index_options">
    What information should be stored in the index, for search and highlighting purposes. Defaults to `positions`.
  </definition>
  <definition term="index_prefixes">
    If enabled, term prefixes of between 2 and 5 characters are indexed into a separate field. This allows prefix searches to run more efficiently, at the expense of a larger index.
  </definition>
  <definition term="index_phrases">
    If enabled, two-term word combinations (*shingles*) are indexed into a separate field. This allows exact phrase queries (no slop) to run more efficiently, at the expense of a larger index. Note that this works best when stopwords are not removed, as phrases containing stopwords will not use the subsidiary field and will fall back to a standard phrase query. Accepts `true` or `false` (default).
  </definition>
  <definition term="norms">
    Whether field-length should be taken into account when scoring queries. Accepts `true` (default) or `false`.
  </definition>
  <definition term="position_increment_gap">
    The number of fake term position which should be inserted between each element of an array of strings. Defaults to the `position_increment_gap` configured on the analyzer which defaults to `100`. `100` was chosen because it prevents phrase queries with reasonably large slops (less than 100) from matching terms across field values.
  </definition>
  <definition term="store">
    Whether the field value should be stored and retrievable separately from the [`_source`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-source-field) field. Accepts `true` or `false` (default).
  </definition>
  <definition term="search_analyzer">
    The [`analyzer`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/analyzer) that should be used at search time on the `text` field. Defaults to the `analyzer` setting.
  </definition>
  <definition term="search_quote_analyzer">
    The [`analyzer`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/analyzer) that should be used at search time when a phrase is encountered. Defaults to the `search_analyzer` setting.
  </definition>
  <definition term="similarity">
    Which scoring algorithm or *similarity* should be used. Defaults to `BM25`.
  </definition>
  <definition term="term_vector">
    Whether term vectors should be stored for the field. Defaults to `no`.
  </definition>
  <definition term="meta">
    Metadata about the field.
  </definition>
</definitions>


## Synthetic `_source`

`text` fields can use a [`keyword`](/docs/reference/elasticsearch/mapping-reference/keyword#keyword-synthetic-source) sub-field to support [synthetic `_source`](/docs/reference/elasticsearch/mapping-reference/mapping-source-field#synthetic-source) without storing values of the text field itself.
In this case, the synthetic source of the `text` field will have the same [modifications](/docs/reference/elasticsearch/mapping-reference/mapping-source-field#synthetic-source) as a `keyword` field.
These modifications can impact usage of `text` fields:
- Reordering text fields can have an effect on [phrase](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-match-query-phrase) and [span](https://www.elastic.co/docs/reference/query-languages/query-dsl/span-queries) queries. See the discussion about [`position_increment_gap`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/position-increment-gap) for more details. You can avoid this by making sure the `slop` parameter on the phrase queries is lower than the `position_increment_gap`. This is the default.
- Handling of `null` values is different. `text` fields ignore `null` values, but `keyword` fields support replacing `null` values with a value specified in the `null_value` parameter. This replacement is represented in synthetic source.

For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "text": {
        "type": "text",
        "fields": {
          "kwd": {
            "type": "keyword",
            "null_value": "NA"
          }
        }
      }
    }
  }
}

{
  "text": [
    null,
    "the quick brown fox",
    "the quick brown fox",
    "jumped over the lazy dog"
  ]
}
```

Will become:
```json
{
  "text": [
    "NA",
    "jumped over the lazy dog",
    "the quick brown fox"
  ]
}
```

If the `text` field sets `store` to `true` then the sub-field is not used and no modifications are applied. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "text": {
        "type": "text",
        "store": true,
        "fields": {
          "raw": {
            "type": "keyword"
          }
        }
      }
    }
  }
}

{
  "text": [
    "the quick brown fox",
    "the quick brown fox",
    "jumped over the lazy dog"
  ]
}
```

Will become:
```json
{
  "text": [
    "the quick brown fox",
    "the quick brown fox",
    "jumped over the lazy dog"
  ]
}
```


## `fielddata` mapping parameter

`text` fields are searchable by default, but by default are not available for aggregations, sorting, or scripting. If you try to sort, aggregate, or access values from a `text` field using a script, you’ll see an exception indicating that field data is disabled by default on text fields. To load field data in memory, set `fielddata=true` on your field.
<note>
  Loading field data in memory can consume significant memory.
</note>

Field data is the only way to access the analyzed tokens from a full text field in aggregations, sorting, or scripting. For example, a full text field like `New York` would get analyzed as `new` and `york`. To aggregate on these tokens requires field data.

## Before enabling fielddata

It usually doesn’t make sense to enable fielddata on text fields. Field data is stored in the heap with the [field data cache](https://www.elastic.co/docs/reference/elasticsearch/configuration-reference/field-data-cache-settings) because it is expensive to calculate. Calculating the field data can cause latency spikes, and increasing heap usage is a cause of cluster performance issues.
Most users who want to do more with text fields use [multi-field mappings](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/multi-fields) by having both a `text` field for full text searches, and an unanalyzed [`keyword`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/keyword) field for aggregations, as follows:
```json

{
  "mappings": {
    "properties": {
      "my_field": { <1>
        "type": "text",
        "fields": {
          "keyword": { <2>
            "type": "keyword"
          }
        }
      }
    }
  }
}
```


## Enabling fielddata on `text` fields

You can enable fielddata on an existing `text` field using the [update mapping API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-indices-put-mapping) as follows:
```json

{
  "properties": {
    "my_field": { <1>
      "type":     "text",
      "fielddata": true
    }
  }
}
```


## `fielddata_frequency_filter` mapping parameter

Fielddata filtering can be used to reduce the number of terms loaded into memory, and thus reduce memory usage. Terms can be filtered by *frequency*:
The frequency filter allows you to only load terms whose document frequency falls between a `min` and `max` value, which can be expressed an absolute number (when the number is bigger than 1.0) or as a percentage (eg `0.01` is `1%` and `1.0` is `100%`). Frequency is calculated **per segment**. Percentages are based on the number of docs which have a value for the field, as opposed to all docs in the segment.
Small segments can be excluded completely by specifying the minimum number of docs that the segment should contain with `min_segment_size`:
```json

{
  "mappings": {
    "properties": {
      "tag": {
        "type": "text",
        "fielddata": true,
        "fielddata_frequency_filter": {
          "min": 0.001,
          "max": 0.1,
          "min_segment_size": 500
        }
      }
    }
  }
}
```