﻿---
title: Ingest data with semantic_text fields
description: This page provides instructions for ingesting data into semantic_text fields. Learn how to index pre-chunked content, use copy_to and multi-fields to...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/semantic-text-ingestions
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available since 9.0
---

# Ingest data with semantic_text fields
This page provides instructions for ingesting data into `semantic_text` fields. Learn how to index pre-chunked content, use `copy_to` and multi-fields to collect values from multiple fields, and perform updates and partial updates to optimize ingestion costs.

## Index pre-chunked content

<applies-to>
  - Elastic Stack: Generally available since 9.1
</applies-to>

To index pre-chunked content, provide your text as an array of strings. Each element in the array represents a single chunk that will be sent directly to the inference service without further chunking.
<stepper>
  <step title="Disable automatic chunking">
    Disable automatic chunking in your index mapping by setting `chunking_settings.strategy` to `none`:
    ```json

    {
      "mappings": {
        "properties": {
          "my_semantic_field": {
            "type": "semantic_text",
            "chunking_settings": {
              "strategy": "none"    <1>
            }
          }
        }
      }
    }
    ```
  </step>

  <step title="Index documents">
    Index documents with pre-chunked text as an array:
    ```json

    {
        "my_semantic_field": ["my first chunk", "my second chunk", ...]    <1>
        ...
    }
    ```
  </step>
</stepper>

<important>
  When providing pre-chunked input:
  - Ensure that you set the chunking strategy to `none` to avoid additional processing.
  - Size each chunk carefully, staying within the token limit of the inference service and the underlying model.
  - If a chunk exceeds the model's token limit, the behavior depends on the service:
    - Some services (such as OpenAI) will return an error.
  - Others (such as `elastic` and `elasticsearch`) will automatically truncate the input.
</important>


## Use `copy_to` and multi-fields with `semantic_text`

You can use a single `semantic_text` field to collect values from multiple fields for semantic search. The `semantic_text` field type can serve as the target of [copy_to fields](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/copy-to), be part of a [multi-field](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/multi-fields) structure, or contain [multi-fields](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/multi-fields) internally.

### Use copy_to

Use `copy_to` to copy values from source fields to a `semantic_text` field:
```json

{
    "mappings": {
        "properties": {
            "source_field": {
                "type": "text",
                "copy_to": "infer_field"
            },
            "infer_field": {
                "type": "semantic_text",
                "inference_id": ".elser-2-elasticsearch"
            }
        }
    }
}
```


### Use multi-fields

Declare `semantic_text` as a multi-field:
```json

{
    "mappings": {
        "properties": {
            "source_field": {
                "type": "text",
                "fields": {
                    "infer_field": {
                        "type": "semantic_text",
                        "inference_id": ".elser-2-elasticsearch"
                    }
                }
            }
        }
    }
}
```


## Updates and partial updates

When updating documents that contain `semantic_text` fields, it's important to understand how inference is triggered:
<definitions>
  <definition term="Full document updates">
    Full document updates re-run inference on all `semantic_text` fields, even if their values did not change. This ensures that embeddings remain consistent with the current document state but can increase ingestion costs.
  </definition>
  <definition term="Partial updates using the Bulk API">
    Partial updates submitted through the [Bulk API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-bulk) reuse existing embeddings when you omit `semantic_text` fields. inference does not run for omitted fields, which can significantly reduce processing time and cost.
  </definition>
  <definition term="Partial updates using the Update API">
    Partial updates submitted through the [Update API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-update) re-run inference on all `semantic_text` fields, even when you omit them from the `doc` object. Embeddings are re-generated regardless of whether field values changed.
  </definition>
</definitions>

To preserve existing embeddings and avoid unnecessary inference costs:
- Use partial updates with the Bulk API.
- Omit any `semantic_text` fields that did not change from the `doc` object in your request.


### Scripted updates

For indices containing `semantic_text` fields, updates that use scripts have the
following behavior:
- ✅ **Supported:** [Update API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-update)
- ❌ **Not supported:** [Bulk API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-bulk-1). Scripted updates will fail even if the script targets non-`semantic_text` fields.