﻿---
title: Pass-through object field type
description: Pass-through objects extend the functionality of objects by allowing to access their subfields without including the name of the pass-through object as...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/passthrough
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Pass-through object field type
Pass-through objects extend the functionality of [objects](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/object) by allowing to access their subfields without including the name of the pass-through object as prefix. For instance:
```json

{
  "mappings": {
    "properties": {
      "attributes": {
        "type": "passthrough", <1>
        "priority": 10,
        "properties": {
          "id": {
            "type": "keyword"
          }
        }
      }
    }
  }
}


{
  "attributes" : {  <2>
    "id": "foo",
    "zone": 10
  }
}


{
  "query": {
    "bool": {
      "must": [
        { "match": { "id": "foo" }},  <3>
        { "match": { "zone": 10 }}
      ]
    }
  }
}


{
  "query": {
    "bool": {
      "must": [
        { "match": { "attributes.id": "foo" }}, <4>
        { "match": { "attributes.zone": 10 }}
      ]
    }
  }
}
```


## Conflict resolution

It’s possible for conflicting names to arise, for fields that are defined within different scopes:
1. A pass-through object is defined next to a field that has the same name as one of the pass-through object sub-fields, e.g.
   ```json

   {
     "attributes" : {
       "id": "foo"
     },
     "id": "bar"
   }
   ```
   In this case, references to `id` point to the field at the root level, while field `attributes.id` can only be accessed using the full path.
2. Two (or more) pass-through objects are defined within the same object and contain fields with the same name, e.g.
   ```json

   {
     "mappings": {
       "properties": {
         "attributes": {
           "type": "passthrough",
           "priority": 10,
           "properties": {
             "id": {
               "type": "keyword"
             }
           }
         },
         "resource.attributes": {
           "type": "passthrough",
           "priority": 20,
           "properties": {
             "id": {
               "type": "keyword"
             }
           }
         }
       }
     }
   }
   ```
   In this case, param `priority` is used for conflict resolution, with the higher values taking precedence. In the example above, `resource.attributes` has higher priority than `attributes`, so references to `id` point to the field within `resource.attributes`. `attributes.id` can still be accessed using its full path.


## Defining sub-fields as time-series dimensions

It is possible to configure a pass-through field as a container for  [time-series dimensions](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds#time-series-dimension). In this case, all sub-fields get annotated with the same parameter under the covers, and they’re also included in [routing path](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds) and [tsid](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds#tsid) calculations, thus simplifying the [TSDS](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds) setup:
```json

{
  "index_patterns": ["metrics-mymetrics-*"],
  "priority": 200,
  "data_stream": { },
  "template": {
    "settings": {
      "index.mode": "time_series"
    },
    "mappings": {
      "properties": {
        "attributes": {
          "type": "passthrough",
          "priority": 10,
          "time_series_dimension": true,
          "properties": {
            "host.name": {
              "type": "keyword"
            }
          }
        },
        "cpu": {
          "type": "integer",
          "time_series_metric": "counter"
        }
      }
    }
  }
}


{
  "@timestamp": "2020-01-01T00:00:00.000Z",
  "attributes" : {
    "host.name": "foo",
    "zone": "bar"
  },
  "cpu": 10
}
```

In the example above, `attributes` is defined as a dimension container. Its sub-fields `host.name` (static) and `zone` (dynamic) get included in the routing path and tsid, and can be referenced in queries without the `attributes.` prefix.

## Sub-field auto-flattening

Pass-through fields apply [auto-flattening](/docs/reference/elasticsearch/mapping-reference/subobjects#subobjects-auto-flattening) to sub-fields by default, to reduce dynamic mapping conflicts. As a consequence, no sub-object definitions are allowed within pass-through fields.

## Parameters for `passthrough` fields

The following parameters are accepted by `passthrough` fields:
<definitions>
  <definition term="priority">
    (Required) used for naming conflict resolution between pass-through fields. The field with the highest value wins. Accepts non-negative integer values.
  </definition>
  <definition term="time_series_dimension">
    Whether or not to treat sub-fields as [time-series dimensions](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds#time-series-dimension). Accepts `false` (default) or `true`.
  </definition>
  <definition term="dynamic">
    Whether or not new `properties` should be added dynamically to an existing object. Accepts `true` (default), `runtime`, `false` and `strict`.
  </definition>
  <definition term="enabled">
    Whether the JSON value given for the object field should be parsed and indexed (`true`, default) or completely ignored (`false`).
  </definition>
  <definition term="properties">
    The fields within the object, which can be of any [data type](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/field-data-types), including `object`. New properties may be added to an existing object.
  </definition>
</definitions>

<important>
  If you need to index arrays of objects instead of single objects, read [Nested](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/nested) first.
</important>