﻿---
title: Join field type
description: The join data type is a special field that creates parent/child relation within documents of the same index. The relations section defines a set of possible...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/parent-join
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Unavailable
---

# Join field type
The `join` data type is a special field that creates parent/child relation within documents of the same index. The `relations` section defines a set of possible relations within the documents, each relation being a parent name and a child name.
<warning>
  We don’t recommend using multiple levels of relations to replicate a relational model. Each level of relation adds an overhead at query time in terms of memory and computation. For better search performance, denormalize your data instead.
</warning>

A parent/child relation can be defined as follows:
```json

{
  "mappings": {
    "properties": {
      "my_id": {
        "type": "keyword"
      },
      "my_join_field": { <1>
        "type": "join",
        "relations": {
          "question": "answer" <2>
        }
      }
    }
  }
}
```

To index a document with a join, the name of the relation and the optional parent of the document must be provided in the `source`. For instance the following example creates two `parent` documents in the `question` context:
```json

{
  "my_id": "1",
  "text": "This is a question",
  "my_join_field": {
    "name": "question" <1>
  }
}


{
  "my_id": "2",
  "text": "This is another question",
  "my_join_field": {
    "name": "question"
  }
}
```

When indexing parent documents, you can choose to specify just the name of the relation as a shortcut instead of encapsulating it in the normal object notation:
```json

{
  "my_id": "1",
  "text": "This is a question",
  "my_join_field": "question" <1>
}


{
  "my_id": "2",
  "text": "This is another question",
  "my_join_field": "question"
}
```

When indexing a child, the name of the relation as well as the parent id of the document must be added in the `_source`.
<warning>
  It is required to index the lineage of a parent in the same shard so you must always route child documents using their greater parent id.
</warning>

For instance the following example shows how to index two `child` documents:
```json

{
  "my_id": "3",
  "text": "This is an answer",
  "my_join_field": {
    "name": "answer", <2>
    "parent": "1" <3>
  }
}


{
  "my_id": "4",
  "text": "This is another answer",
  "my_join_field": {
    "name": "answer",
    "parent": "1"
  }
}
```


## Parent-join and performance

The join field shouldn’t be used like joins in a relation database. In Elasticsearch the key to good performance is to de-normalize your data into documents. Each join field, `has_child` or `has_parent` query adds a significant tax to your query performance. It can also trigger [global ordinals](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/eager-global-ordinals) to be built.
The only case where the join field makes sense is if your data contains a one-to-many relationship where one entity significantly outnumbers the other entity. An example of such case is a use case with products and offers for these products. In the case that offers significantly outnumbers the number of products then it makes sense to model the product as parent document and the offer as child document.

## Parent-join restrictions

- Only one `join` field mapping is allowed per index.
- Parent and child documents must be indexed on the same shard. This means that the same `routing` value needs to be provided when [getting](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-get), [deleting](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-delete), or [updating](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-update) a child document.
- An element can have multiple children but only one parent.
- It is possible to add a new relation to an existing `join` field.
- It is also possible to add a child to an existing element but only if the element is already a parent.


## Searching with parent-join

The parent-join creates one field to index the name of the relation within the document (`my_parent`, `my_child`, … ).
It also creates one field per parent/child relation. The name of this field is the name of the `join` field followed by `#` and the name of the parent in the relation. So for instance for the `my_parent` → [`my_child`, `another_child`] relation, the `join` field creates an additional field named `my_join_field#my_parent`.
This field contains the parent `_id` that the document links to if the document is a child (`my_child` or `another_child`) and the `_id` of document if it’s a parent (`my_parent`).
When searching an index that contains a `join` field, these two fields are always returned in the search response:
```json

{
  "query": {
    "match_all": {}
  },
  "sort": ["my_id"]
}
```

Will return:
```json
{
  ...,
  "hits": {
    "total": {
      "value": 4,
      "relation": "eq"
    },
    "max_score": null,
    "hits": [
      {
        "_index": "my-index-000001",
        "_id": "1",
        "_score": null,
        "_source": {
          "my_id": "1",
          "text": "This is a question",
          "my_join_field": "question"         
        },
        "sort": [
          "1"
        ]
      },
      {
        "_index": "my-index-000001",
        "_id": "2",
        "_score": null,
        "_source": {
          "my_id": "2",
          "text": "This is another question",
          "my_join_field": "question"          
        },
        "sort": [
          "2"
        ]
      },
      {
        "_index": "my-index-000001",
        "_id": "3",
        "_score": null,
        "_routing": "1",
        "_source": {
          "my_id": "3",
          "text": "This is an answer",
          "my_join_field": {
            "name": "answer",                 
            "parent": "1"                     
          }
        },
        "sort": [
          "3"
        ]
      },
      {
        "_index": "my-index-000001",
        "_id": "4",
        "_score": null,
        "_routing": "1",
        "_source": {
          "my_id": "4",
          "text": "This is another answer",
          "my_join_field": {
            "name": "answer",
            "parent": "1"
          }
        },
        "sort": [
          "4"
        ]
      }
    ]
  }
}
```


## Parent-join queries and aggregations

See the [`has_child`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-has-child-query) and [`has_parent`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-has-parent-query) queries, the [`children`](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-children-aggregation) aggregation, and [inner hits](/docs/reference/elasticsearch/rest-apis/retrieve-inner-hits#parent-child-inner-hits) for more information.
The value of the `join` field is accessible in aggregations and scripts, and may be queried with the [`parent_id` query](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-parent-id-query):
```json

{
  "query": {
    "parent_id": { <1>
      "type": "answer",
      "id": "1"
    }
  },
  "aggs": {
    "parents": {
      "terms": {
        "field": "my_join_field#question", <2>
        "size": 10
      }
    }
  },
  "runtime_mappings": {
    "parent": {
      "type": "long",
      "script": """
        emit(Integer.parseInt(doc['my_join_field#question'].value)) <3>
      """
    }
  },
  "fields": [
    { "field": "parent" }
  ]
}
```


## Global ordinals

The `join` field uses [global ordinals](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/eager-global-ordinals) to speed up joins. Global ordinals need to be rebuilt after any change to a shard. The more parent id values are stored in a shard, the longer it takes to rebuild the global ordinals for the `join` field.
Global ordinals, by default, are built eagerly: if the index has changed, global ordinals for the `join` field will be rebuilt as part of the refresh. This can add significant time to the refresh. However most of the times this is the right trade-off, otherwise global ordinals are rebuilt when the first parent-join query or aggregation is used. This can introduce a significant latency spike for your users and usually this is worse as multiple global ordinals for the `join` field may be attempt rebuilt within a single refresh interval when many writes are occurring.
When the `join` field is used infrequently and writes occur frequently it may make sense to disable eager loading:
```json

{
  "mappings": {
    "properties": {
      "my_join_field": {
        "type": "join",
        "relations": {
           "question": "answer"
        },
        "eager_global_ordinals": false
      }
    }
  }
}
```

The amount of heap used by global ordinals can be checked per parent relation as follows:
```json
# Per-index


# Per-node per-index
```


## Multiple children per parent

It is also possible to define multiple children for a single parent:
```json

{
  "mappings": {
    "properties": {
      "my_join_field": {
        "type": "join",
        "relations": {
          "question": ["answer", "comment"]  <1>
        }
      }
    }
  }
}
```


## Multiple levels of parent join

<warning>
  We don’t recommend using multiple levels of relations to replicate a relational model. Each level of relation adds an overhead at query time in terms of memory and computation. For better search performance, denormalize your data instead.
</warning>

Multiple levels of parent/child:
```json

{
  "mappings": {
    "properties": {
      "my_join_field": {
        "type": "join",
        "relations": {
          "question": ["answer", "comment"],  <1>
          "answer": "vote" <2>
        }
      }
    }
  }
}
```

The mapping above represents the following tree:
```
   question
    /    \
   /      \
comment  answer
           |
           |
          vote
```

Indexing a grandchild document requires a `routing` value equals to the grand-parent (the greater parent of the lineage):
```json

{
  "text": "This is a vote",
  "my_join_field": {
    "name": "vote",
    "parent": "2" <2>
  }
}
```