﻿---
title: Numeric field types
description: The following numeric types are supported: Below is an example of configuring a mapping with numeric fields: As far as integer types (byte, short, integer...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/number
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Numeric field types
The following numeric types are supported:
<definitions>
  <definition term="long">
    A signed 64-bit integer with a minimum value of `-2^63` and a maximum value of `2^63 - 1`.
  </definition>
  <definition term="integer">
    A signed 32-bit integer with a minimum value of `-2^31` and a maximum value of `2^31 - 1`.
  </definition>
  <definition term="short">
    A signed 16-bit integer with a minimum value of `-32,768` and a maximum value of `32,767`.
  </definition>
  <definition term="byte">
    A signed 8-bit integer with a minimum value of `-128` and a maximum value of `127`.
  </definition>
  <definition term="double">
    A double-precision 64-bit IEEE 754 floating point number, restricted to finite values.
  </definition>
  <definition term="float">
    A single-precision 32-bit IEEE 754 floating point number, restricted to finite values.
  </definition>
  <definition term="half_float">
    A half-precision 16-bit IEEE 754 floating point number, restricted to finite values.
  </definition>
  <definition term="scaled_float">
    A floating point number that is backed by a `long`, scaled by a fixed `double` scaling factor.
  </definition>
  <definition term="unsigned_long">
    An unsigned 64-bit integer with a minimum value of 0 and a maximum value of `2^64 - 1`.
  </definition>
</definitions>

Below is an example of configuring a mapping with numeric fields:
```json

{
  "mappings": {
    "properties": {
      "number_of_bytes": {
        "type": "integer"
      },
      "time_in_seconds": {
        "type": "float"
      },
      "price": {
        "type": "scaled_float",
        "scaling_factor": 100
      }
    }
  }
}
```

<note>
  The `double`, `float` and `half_float` types consider that `-0.0` and `+0.0` are different values. As a consequence, doing a `term` query on `-0.0` will not match `+0.0` and vice-versa. Same is true for range queries: if the upper bound is `-0.0` then `+0.0` will not match, and if the lower bound is `+0.0` then `-0.0` will not match.
</note>


## Which type should I use?

As far as integer types (`byte`, `short`, `integer` and `long`) are concerned, you should pick the smallest type which is enough for your use-case. This will help indexing and searching be more efficient. Note however that storage is optimized based on the actual values that are stored, so picking one type over another one will have no impact on storage requirements.
For floating-point types, it is often more efficient to store floating-point data into an integer using a scaling factor, which is what the `scaled_float` type does under the hood. For instance, a `price` field could be stored in a `scaled_float` with a `scaling_factor` of `100`. All APIs would work as if the field was stored as a double, but under the hood Elasticsearch would be working with the number of cents, `price*100`, which is an integer. This is mostly helpful to save disk space since integers are way easier to compress than floating points. `scaled_float` is also fine to use in order to trade accuracy for disk space. For instance imagine that you are tracking cpu utilization as a number between `0` and `1`. It usually does not matter much whether cpu utilization is `12.7%` or `13%`, so you could use a `scaled_float` with a `scaling_factor` of `100` in order to round cpu utilization to the closest percent in order to save space.
If `scaled_float` is not a good fit, then you should pick the smallest type that is enough for the use-case among the floating-point types: `double`, `float` and `half_float`. Here is a table that compares these types in order to help make a decision.


| Type         | Minimum value | Maximum value          | Significant                                                      bits / digits | Example precision loss                   |
|--------------|---------------|------------------------|--------------------------------------------------------------------------------|------------------------------------------|
| `double`     | `2^-1074`     | `(2 - 2^-52) * 2^1023` | `53 / 15.95`                                                                   | `1.2345678912345678`→`1.234567891234568` |
| `float`      | `2^-149`      | `(2 - 2^-23) * 2^127`  | `24 / 7.22`                                                                    | `1.23456789`→`1.2345679`                 |
| `half_float` | `2^-24`       | `65504`                | `11 / 3.31`                                                                    | `1.2345`→`1.234375`                      |

<admonition title="Mapping numeric identifiers">
  Not all numeric data should be mapped as a `numeric` field data type. Elasticsearch optimizes numeric fields, such as `integer` or `long`, for [`range`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-range-query) queries. However, [`keyword`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/keyword) fields are better for [`term`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-term-query) and other [term-level](https://www.elastic.co/docs/reference/query-languages/query-dsl/term-level-queries) queries.Identifiers, such as an ISBN or a product ID, are rarely used in `range` queries. However, they are often retrieved using term-level queries.Consider mapping a numeric identifier as a `keyword` if:
  - You don’t plan to search for the identifier data using [`range`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-range-query) queries.
  - Fast retrieval is important. `term` query searches on `keyword` fields are often faster than `term` searches on numeric fields.
  If you’re unsure which to use, you can use a [multi-field](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/multi-fields) to map the data as both a `keyword` *and* a numeric data type.
</admonition>


## Parameters for numeric fields

The following parameters are accepted by numeric types:
<definitions>
  <definition term="coerce">
    Try to convert strings to numbers and truncate fractions for integers. Accepts `true` (default) and `false`. Not applicable for `unsigned_long`. Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="doc_values">
    Should the field be stored on disk in a column-stride fashion, so that it can later be used for sorting, aggregations, or scripting? Accepts `true` (default) or `false`.
  </definition>
  <definition term="ignore_malformed">
    If `true`, malformed numbers are ignored. If `false` (default), malformed numbers throw an exception and reject the whole document.  Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="index">
    Should the field be quickly searchable? Accepts `true` (default) and `false`. Numeric fields that only have [`doc_values`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/doc-values) enabled can also be queried, albeit slower.
  </definition>
  <definition term="meta">
    Metadata about the field.
  </definition>
  <definition term="null_value">
    Accepts a numeric value of the same `type` as the field which is substituted for any explicit `null` values. Defaults to `null`, which means the field is treated as missing. Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="on_script_error">
    Defines what to do if the script defined by the `script` parameter throws an error at indexing time. Accepts `fail` (default), which will cause the entire document to be rejected, and `continue`, which will register the field in the document’s [`_ignored`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-ignored-field) metadata field and continue indexing. This parameter can only be set if the `script` field is also set.
  </definition>
  <definition term="script">
    If this parameter is set, then the field will index values generated by this script, rather than reading the values directly from the source. If a value is set for this field on the input document, then the document will be rejected with an error. Scripts are in the same format as their [runtime equivalent](https://www.elastic.co/docs/manage-data/data-store/mapping/map-runtime-field). Scripts can only be configured on `long` and `double` field types.
  </definition>
  <definition term="store">
    Whether the field value should be stored and retrievable separately from the [`_source`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-source-field) field. Accepts `true` or `false` (default).
  </definition>
  <definition term="time_series_dimension">
    (Optional, Boolean)
    Marks the field as a [time series dimension](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds#time-series-dimension). Defaults to `false`.
    The `index.mapping.dimension_fields.limit` [index setting](https://www.elastic.co/docs/reference/elasticsearch/index-settings/time-series) limits the number of dimensions in an index.
    Dimension fields have the following constraints:
    - The `doc_values` and `index` mapping parameters must be `true`.
    Of the numeric field types, only `byte`, `short`, `integer`, `long`, and `unsigned_long` fields support this parameter.
    A numeric field can’t be both a time series dimension and a time series metric.
  </definition>
  <definition term="time_series_metric">
    (Optional, string) Marks the field as a [time series metric](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds#time-series-metric). The value is the metric type. You can’t update this parameter for existing fields.
    **Valid `time_series_metric` values for numeric fields**:
    <definitions>
      <definition term="counter">
        A cumulative metric that only monotonically increases or resets to `0` (zero). For example, a count of errors or completed tasks.
      </definition>
      <definition term="gauge">
        A metric that represents a single numeric that can arbitrarily increase or decrease. For example, a temperature or available disk space.
      </definition>
      <definition term="null (Default)">
        Not a time series metric.
      </definition>
    </definitions>
    For a numeric time series metric, the `doc_values` parameter must be `true`. A numeric field can’t be both a time series dimension and a time series metric.
  </definition>
</definitions>


## Parameters for `scaled_float`

`scaled_float` accepts an additional parameter:
<definitions>
  <definition term="scaling_factor">
    The scaling factor to use when encoding values. Values will be multiplied by this factor at index time and rounded to the closest long value. For instance, a `scaled_float` with a `scaling_factor` of `10` would internally store `2.34` as `23` and all search-time operations (queries, aggregations, sorting) will behave as if the document had a value of `2.3`. High values of `scaling_factor` improve accuracy but also increase space requirements. This parameter is required.
  </definition>
</definitions>


## `scaled_float` saturation

`scaled_float` is stored as a single `long` value, which is the product of multiplying the original value by the scaling factor. If the multiplication results in a value that is outside the range of a `long`, the value is saturated to the minimum or maximum value of a `long`. For example, if the scaling factor is `100` and the value is `92233720368547758.08`, the expected value is `9223372036854775808`. However, the value that is stored is `9223372036854775807`, the maximum value for a `long`.
This can lead to unexpected results with [range queries](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-range-query) when the scaling factor or provided `float` value are exceptionally large.

## Synthetic `_source`

All numeric fields support [synthetic `_source`](/docs/reference/elasticsearch/mapping-reference/mapping-source-field#synthetic-source) in their default configuration. Synthetic `_source` cannot be used together with [`copy_to`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/copy-to), or with [`doc_values`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/doc-values) disabled.
Synthetic source may sort numeric field values. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "long": { "type": "long" }
    }
  }
}

{
  "long": [0, 0, -123466, 87612]
}
```

Will become:
```json
{
  "long": [-123466, 0, 0, 87612]
}
```

Scaled floats will always apply their scaling factor so:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "f": { "type": "scaled_float", "scaling_factor": 0.01 }
    }
  }
}

{
  "f": 123
}
```

Will become:
```json
{
  "f": 100.0
}
```