﻿---
title: Nested field type
description: The nested type is a specialised version of the object data type that allows arrays of objects to be indexed in a way that they can be queried independently...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/nested
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Nested field type
The `nested` type is a specialised version of the [`object`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/object) data type that allows arrays of objects to be indexed in a way that they can be queried independently of each other.
<tip>
  When ingesting key-value pairs with a large, arbitrary set of keys, you might consider modeling each key-value pair as its own nested document with `key` and `value` fields. Instead, consider using the [flattened](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/flattened) data type, which maps an entire object as a single field and allows for simple searches over its contents. Nested documents and queries are typically expensive, so using the `flattened` data type for this use case is a better option.
</tip>

<warning>
  Nested fields have incomplete support in Kibana. While they are visible and searchable in Discover, they cannot be used to build visualizations in Lens.
</warning>

<warning>
  If a parent object field mapping has `subobject` set to `false` or the root object `subobject` set to `false`, then nested field mappings are also auto flattened and only leaf field mappings are retained. This means that the nested field mapping's functionality is disabled.
</warning>


## How arrays of objects are flattened

Elasticsearch has no concept of inner objects. Therefore, it flattens object hierarchies into a simple list of field names and values. For instance, consider the following document:
```json

{
  "group" : "fans",
  "user" : [ <1>
    {
      "first" : "John",
      "last" :  "Smith"
    },
    {
      "first" : "Alice",
      "last" :  "White"
    }
  ]
}
```

The previous document would be transformed internally into a document that looks more like this:
```js
{
  "group" :        "fans",
  "user.first" : [ "alice", "john" ],
  "user.last" :  [ "smith", "white" ]
}
```

The `user.first` and `user.last` fields are flattened into multi-value fields, and the association between `alice` and `white` is lost. This document would incorrectly match a query for `alice AND smith`:
```json

{
  "query": {
    "bool": {
      "must": [
        { "match": { "user.first": "Alice" }},
        { "match": { "user.last":  "Smith" }}
      ]
    }
  }
}
```


## Using `nested` fields for arrays of objects

If you need to index arrays of objects and to maintain the independence of each object in the array, use the `nested` data type instead of the [`object`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/object) data type.
Internally, nested objects index each object in the array as a separate hidden document, meaning that each nested object can be queried independently of the others with the [`nested` query](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-nested-query):
```json

{
  "mappings": {
    "properties": {
      "user": {
        "type": "nested" <1>
      }
    }
  }
}


{
  "group" : "fans",
  "user" : [
    {
      "first" : "John",
      "last" :  "Smith"
    },
    {
      "first" : "Alice",
      "last" :  "White"
    }
  ]
}


{
  "query": {
    "nested": {
      "path": "user",
      "query": {
        "bool": {
          "must": [
            { "match": { "user.first": "Alice" }},
            { "match": { "user.last":  "Smith" }} <2>
          ]
        }
      }
    }
  }
}


{
  "query": {
    "nested": {
      "path": "user",
      "query": {
        "bool": {
          "must": [
            { "match": { "user.first": "Alice" }},
            { "match": { "user.last":  "White" }} <3>
          ]
        }
      },
      "inner_hits": { <4>
        "highlight": {
          "fields": {
            "user.first": {}
          }
        }
      }
    }
  }
}
```


## Interacting with `nested` documents

Nested documents can be:
- queried with the [`nested`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-nested-query) query.
- analyzed with the [`nested`](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-nested-aggregation) and [`reverse_nested`](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-reverse-nested-aggregation) aggregations.
- sorted with [nested sorting](/docs/reference/elasticsearch/rest-apis/sort-search-results#nested-sorting).
- retrieved and highlighted with [nested inner hits](/docs/reference/elasticsearch/rest-apis/retrieve-inner-hits#nested-inner-hits).

<important>
  Because nested documents are indexed as separate documents, they can only be accessed within the scope of the `nested` query, the `nested`/`reverse_nested` aggregations, or [nested inner hits](/docs/reference/elasticsearch/rest-apis/retrieve-inner-hits#nested-inner-hits).For instance, if a string field within a nested document has [`index_options`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/index-options) set to `offsets` to allow use of the postings during the highlighting, these offsets will not be available during the main highlighting phase. Instead, highlighting needs to be performed via [nested inner hits](/docs/reference/elasticsearch/rest-apis/retrieve-inner-hits#nested-inner-hits). The same consideration applies when loading fields during a search through [`docvalue_fields`](/docs/reference/elasticsearch/rest-apis/retrieve-selected-fields#docvalue-fields) or [`stored_fields`](/docs/reference/elasticsearch/rest-apis/retrieve-selected-fields#stored-fields).
</important>


## Parameters for `nested` fields

The following parameters are accepted by `nested` fields:
<definitions>
  <definition term="dynamic">
    (Optional, string) Whether or not new `properties` should be added dynamically to an existing nested object. Accepts `true` (default), `false` and `strict`.
  </definition>
  <definition term="properties">
    (Optional, object) The fields within the nested object, which can be of any [data type](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/field-data-types), including `nested`. New properties may be added to an existing nested object.
  </definition>
  <definition term="include_in_parent">
    (Optional, Boolean) If `true`, all fields in the nested object are also added to the parent document as standard (flat) fields. Defaults to `false`.
  </definition>
  <definition term="include_in_root">
    (Optional, Boolean) If `true`, all fields in the nested object are also added to the root document as standard (flat) fields. Defaults to `false`.
  </definition>
</definitions>


## Limits on `nested` mappings and objects

As described earlier, each nested object is indexed as a separate Lucene document. Continuing with the previous example, if we indexed a single document containing 100 `user` objects, then 101 Lucene documents would be created: one for the parent document, and one for each nested object. Because of the expense associated with `nested` mappings, Elasticsearch puts settings in place to guard against performance problems:
<definitions>
  <definition term="index.mapping.nested_fields.limit">
    The maximum number of distinct `nested` mappings in an index. The `nested` type should only be used in special cases, when arrays of objects need to be queried independently of each other. To safeguard against poorly designed mappings, this setting limits the number of unique `nested` types per index. Default is `100`.
  </definition>
</definitions>

In the previous example, the `user` mapping would count as only 1 towards this limit.
<definitions>
  <definition term="index.mapping.nested_parents.limit">
    The maximum number of nested fields that act as parents of other nested fields. Each nested parent requires its own in-memory parent bitset. Root-level nested fields share a parent bitset, but nested fields under other nested fields require additional bitsets. This setting limits the number of unique nested parents to prevent excessive memory usage. Default is `50`.
  </definition>
  <definition term="index.mapping.nested_objects.limit">
    The maximum number of nested JSON objects that a single document can contain across all `nested` types. This limit helps to prevent out of memory errors when a document contains too many nested objects. Default is `10000`.
  </definition>
</definitions>

To illustrate how this setting works, consider adding another `nested` type called `comments` to the previous example mapping. For each document, the combined number of `user` and `comment` objects it contains must be below the limit.
See [Prevent mapping explosions](https://www.elastic.co/docs/manage-data/data-store/mapping#mapping-limit-settings) regarding additional settings for preventing mappings explosion.