﻿---
title: Keyword type family
description: The keyword family includes the following field types: keyword, which is used for structured content such as IDs, email addresses, hostnames, status codes,...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/keyword
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Keyword type family
The keyword family includes the following field types:
- [`keyword`](#keyword-field-type), which is used for structured content such as IDs, email addresses, hostnames, status codes, zip codes, or tags.
- [`constant_keyword`](#constant-keyword-field-type) for keyword fields that always contain the same value.
- [`wildcard`](#wildcard-field-type) for unstructured machine-generated content. The `wildcard` type is optimized for fields with large values or high cardinality.

Keyword fields are often used in [sorting](https://www.elastic.co/docs/reference/elasticsearch/rest-apis/sort-search-results), [aggregations](https://www.elastic.co/docs/reference/aggregations), and [term-level queries](https://www.elastic.co/docs/reference/query-languages/query-dsl/term-level-queries), such as [`term`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-term-query).
<tip>
  Avoid using keyword fields for full-text search. Use the [`text`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/text) field type instead.
</tip>


## Keyword field type

Below is an example of a mapping for a basic `keyword` field:
```json

{
  "mappings": {
    "properties": {
      "tags": {
        "type":  "keyword"
      }
    }
  }
}
```

<admonition title="Mapping numeric identifiers">
  Not all numeric data should be mapped as a [numeric](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/number) field data type. Elasticsearch optimizes numeric fields, such as `integer` or `long`, for [`range`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-range-query) queries. However, `keyword` fields are better for [`term`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-term-query) and other [term-level](https://www.elastic.co/docs/reference/query-languages/query-dsl/term-level-queries) queries.Identifiers, such as an ISBN or a product ID, are rarely used in `range` queries. However, they are often retrieved using term-level queries.Consider mapping a numeric identifier as a `keyword` if:
  - You don’t plan to search for the identifier data using [`range`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-range-query) queries.
  - Fast retrieval is important. `term` query searches on `keyword` fields are often faster than `term` searches on numeric fields.
  If you’re unsure which to use, you can use a [multi-field](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/multi-fields) to map the data as both a `keyword` *and* a numeric data type.
</admonition>


### Parameters for basic keyword fields

The following parameters are accepted by `keyword` fields:
<definitions>
  <definition term="doc_values">
    Should the field be stored on disk in a column-stride fashion, so that it can later be used for sorting, aggregations, or scripting? Accepts `true` (default) or `false`.
  </definition>
  <definition term="eager_global_ordinals">
    Should global ordinals be loaded eagerly on refresh? Accepts `true` or `false` (default). Enabling this is a good idea on fields that are frequently used for terms aggregations.
  </definition>
  <definition term="fields">
    Multi-fields allow the same string value to be indexed in multiple ways for different purposes, such as one field for search and a multi-field for sorting and aggregations.
  </definition>
  <definition term="ignore_above">
    Do not index any field containing a string with more characters than this value. This is important because Elasticsearch
    will reject entire documents if they contain keyword fields that exceed `32766` UTF-8 encoded bytes.
    To avoid any risk of document rejection, set this value to `8191` or less. Fields with strings exceeding this
    length will be excluded from indexing.
    The defaults are complicated:
    | Index type                                                                                                      | Default                                                                                                                                                       | Effect                                                                                                                                                                                                                    |
    |-----------------------------------------------------------------------------------------------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
    | Standard indices                                                                                                | `2147483647` (effectively unbounded)                                                                                                                          | Documents will be rejected if this keyword exceeds `32766` UTF-8 encoded bytes.                                                                                                                                           |
    | `logsdb` indices                                                                                                | `8191`                                                                                                                                                        | This `keyword` field will never cause documents to be rejected. If this field is longer than `8191` characters it won't be indexed but its values are still available from `_source`.                                     |
    | [dynamic mapping](https://www.elastic.co/docs/manage-data/data-store/mapping/dynamic-mapping) for string fields | `text` field with a [sub](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/multi-fields)-`keyword` field with an `ignore_above` of `256` | All string fields are available. Values longer than 256 characters are only available for full text search and won't have a value in their `.keyword` sub-field, so they can not be used for exact matching over _search. |
  </definition>
  <definition term="index">
    Should the field be quickly searchable? Accepts `true` (default) and `false`. `keyword` fields that only have [`doc_values`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/doc-values) enabled can still be queried, albeit slower.
  </definition>
  <definition term="index_options">
    What information should be stored in the index, for scoring purposes. Defaults to `docs` but can also be set to `freqs` to take term frequency into account when computing scores.
  </definition>
  <definition term="meta">
    Metadata about the field.
  </definition>
  <definition term="norms">
    Whether field-length should be taken into account when scoring queries. Accepts `true` or `false` (default).
  </definition>
  <definition term="null_value">
    Accepts a string value which is substituted for any explicit `null` values. Defaults to `null`, which means the field is treated as missing. Note that this cannot be set if the `script` value is used.
  </definition>
  <definition term="on_script_error">
    Defines what to do if the script defined by the `script` parameter throws an error at indexing time. Accepts `fail` (default), which will cause the entire document to be rejected, and `continue`, which will register the field in the document’s [`_ignored`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-ignored-field) metadata field and continue indexing. This parameter can only be set if the `script` field is also set.
  </definition>
  <definition term="script">
    If this parameter is set, then the field will index values generated by this script, rather than reading the values directly from the source. If a value is set for this field on the input document, then the document will be rejected with an error. Scripts are in the same format as their [runtime equivalent](https://www.elastic.co/docs/manage-data/data-store/mapping/map-runtime-field). Values emitted by the script are normalized as usual, and will be ignored if they are longer that the value set on `ignore_above`.
  </definition>
  <definition term="store">
    Whether the field value should be stored and retrievable separately from the [`_source`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-source-field) field. Accepts `true` or `false` (default).
  </definition>
  <definition term="similarity">
    Which scoring algorithm or *similarity* should be used. Defaults to `BM25`.
  </definition>
  <definition term="normalizer">
    How to pre-process the keyword prior to indexing. Defaults to `null`, meaning the keyword is kept as-is.
  </definition>
  <definition term="split_queries_on_whitespace">
    Whether [full text queries](https://www.elastic.co/docs/reference/query-languages/query-dsl/full-text-queries) should split the input on whitespace when building a query for this field. Accepts `true` or `false` (default).
  </definition>
  <definition term="time_series_dimension">
    (Optional, Boolean)
    Marks the field as a [time series dimension](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds#time-series-dimension). Defaults to `false`.
    The `index.mapping.dimension_fields.limit` [index setting](https://www.elastic.co/docs/reference/elasticsearch/index-settings/time-series) limits the number of dimensions in an index.
    Dimension fields have the following constraints:
    - The `doc_values` and `index` mapping parameters must be `true`.
    - Dimension values are used to identify a document’s time series. If dimension values are altered in any way during indexing, the document will be stored as belonging to different from intended time series. As a result there are additional constraints: the field cannot use a [`normalizer`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/normalizer).
  </definition>
</definitions>


## Synthetic `_source`

Synthetic source may sort `keyword` fields and remove duplicates. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "kwd": { "type": "keyword" }
    }
  }
}

{
  "kwd": ["foo", "foo", "bar", "baz"]
}
```

Will become:
```json
{
  "kwd": ["bar", "baz", "foo"]
}
```

If a `keyword` field sets `store` to `true` then order and duplicates are preserved. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "kwd": { "type": "keyword", "store": true }
    }
  }
}

{
  "kwd": ["foo", "foo", "bar", "baz"]
}
```

Will become:
```json
{
  "kwd": ["foo", "foo", "bar", "baz"]
}
```

Values longer than `ignore_above` are preserved but sorted to the end. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "kwd": { "type": "keyword", "ignore_above": 3 }
    }
  }
}

{
  "kwd": ["foo", "foo", "bang", "bar", "baz"]
}
```

Will become:
```json
{
  "kwd": ["bar", "baz", "foo", "bang"]
}
```

If `null_value` is configured, `null` values are replaced with the `null_value` in synthetic source:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "kwd": { "type": "keyword", "null_value": "NA" }
    }
  }
}

{
  "kwd": ["foo", null, "bar"]
}
```

Will become:
```json
{
  "kwd": ["NA", "bar", "foo"]
  }
```


## Constant keyword field type

Constant keyword is a specialization of the `keyword` field for the case that all documents in the index have the same value.
```json

{
  "mappings": {
    "properties": {
      "@timestamp": {
        "type": "date"
      },
      "message": {
        "type": "text"
      },
      "level": {
        "type": "constant_keyword",
        "value": "debug"
      }
    }
  }
}
```

`constant_keyword` supports the same queries and aggregations as `keyword` fields do, but takes advantage of the fact that all documents have the same value per index to execute queries more efficiently.
It is both allowed to submit documents that don’t have a value for the field or that have a value equal to the value configured in mappings. The two below indexing requests are equivalent:
```json

{
  "@timestamp": "2019-12-12",
  "message": "Starting up Elasticsearch",
  "level": "debug"
}


{
  "@timestamp": "2019-12-12",
  "message": "Starting up Elasticsearch"
}
```

However providing a value that is different from the one configured in the mapping is disallowed.
In case no `value` is provided in the mappings, the field will automatically configure itself based on the value contained in the first indexed document. While this behavior can be convenient, note that it means that a single poisonous document can cause all other documents to be rejected if it had a wrong value.
Before a value has been provided (either through the mappings or from a document), queries on the field will not match any documents. This includes [`exists`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-exists-query) queries.
The `value` of the field cannot be changed after it has been set.

### Parameters for constant keyword fields

The following mapping parameters are accepted:
<definitions>
  <definition term="meta">
    Metadata about the field.
  </definition>
  <definition term="value">
    The value to associate with all documents in the index. If this parameter is not provided, it is set based on the first document that gets indexed.
  </definition>
</definitions>


## Wildcard field type

The `wildcard` field type is a specialized keyword field for unstructured machine-generated content you plan to search using grep-like [`wildcard`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-wildcard-query) and [`regexp`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-regexp-query) queries. The `wildcard` type is optimized for fields with large values or high cardinality.
<admonition title="Mapping unstructured content">
  You can map a field containing unstructured content to either a `text` or keyword family field. The best field type depends on the nature of the content and how you plan to search the field.Use the `text` field type if:
  - The content is human-readable, such as an email body or product description.
  - You plan to search the field for individual words or phrases, such as `the brown fox jumped`, using [full text queries](https://www.elastic.co/docs/reference/query-languages/query-dsl/full-text-queries). Elasticsearch [analyzes](https://www.elastic.co/docs/manage-data/data-store/text-analysis) `text` fields to return the most relevant results for these queries.
  Use a keyword family field type if:
  - The content is machine-generated, such as a log message or HTTP request information.
  - You plan to search the field for exact full values, such as `org.foo.bar`, or partial character sequences, such as `org.foo.*`, using [term-level queries](https://www.elastic.co/docs/reference/query-languages/query-dsl/term-level-queries).
  **Choosing a keyword family field type**If you choose a keyword family field type, you can map the field as a `keyword` or `wildcard` field depending on the cardinality and size of the field’s values. Use the `wildcard` type if you plan to regularly search the field using a [`wildcard`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-wildcard-query) or [`regexp`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-regexp-query) query and meet one of the following criteria:
  - The field contains more than a million unique values. AND You plan to regularly search the field using a pattern with leading wildcards, such as `*foo` or `*baz`.
  - The field contains values larger than 32KB. AND You plan to regularly search the field using any wildcard pattern.
  Otherwise, use the `keyword` field type for faster searches, faster indexing, and lower storage costs. For an in-depth comparison and decision flowchart, see our [related blog post](https://www.elastic.co/blog/find-strings-within-strings-faster-with-the-new-elasticsearch-wildcard-field).**Switching from a `text` field to a keyword field**If you previously used a `text` field to index unstructured machine-generated content, you can [reindex to update the mapping](https://www.elastic.co/docs/manage-data/data-store/mapping/explicit-mapping#update-mapping) to a `keyword` or `wildcard` field. We also recommend you update your application or workflow to replace any word-based [full text queries](https://www.elastic.co/docs/reference/query-languages/query-dsl/full-text-queries) on the field to equivalent [term-level queries](https://www.elastic.co/docs/reference/query-languages/query-dsl/term-level-queries).
</admonition>

Internally the `wildcard` field indexes the whole field value using ngrams and stores the full string. The index is used as a rough filter to cut down the number of values that are then checked by retrieving and checking the full values. This field is especially well suited to run grep-like queries on log lines. Storage costs are typically lower than those of `keyword` fields but search speeds for exact matches on full terms are slower. If the field values share many prefixes, such as URLs for the same website, storage costs for a `wildcard` field may be higher than an equivalent `keyword` field.
You index and search a wildcard field as follows
```json

{
  "mappings": {
    "properties": {
      "my_wildcard": {
        "type": "wildcard"
      }
    }
  }
}


{
  "my_wildcard" : "This string can be quite lengthy"
}


{
  "query": {
    "wildcard": {
      "my_wildcard": {
        "value": "*quite*lengthy"
      }
    }
  }
}
```


### Parameters for wildcard fields

The following parameters are accepted by `wildcard` fields:
<definitions>
  <definition term="null_value">
    Accepts a string value which is substituted for any explicit `null` values. Defaults to `null`, which means the field is treated as missing.
  </definition>
  <definition term="ignore_above">
    Do not index any string longer than this value. Defaults to `2147483647` in standard indices so that all values would be accepted, and `8191` in logsdb indices to protect against Lucene's term byte-length limit of `32766`.
  </definition>
</definitions>


### Limitations

- `wildcard` fields are untokenized like keyword fields, so do not support queries that rely on word positions such as phrase queries.
- When running `wildcard` queries any `rewrite` parameter is ignored. The scoring is always a constant score.


## Synthetic `_source`

Synthetic source may sort `wildcard` field values. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "card": { "type": "wildcard" }
    }
  }
}

{
  "card": ["king", "ace", "ace", "jack"]
}
```

Will become:
```json
{
  "card": ["ace", "jack", "king"]
}
```