﻿---
title: IP field type
description: An ip field can index/store either IPv4 or IPv6 addresses. The following parameters are accepted by ip fields: The most common way to query ip addresses...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/ip
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# IP field type
An `ip` field can index/store either [IPv4](https://en.wikipedia.org/wiki/IPv4) or [IPv6](https://en.wikipedia.org/wiki/IPv6) addresses.
```json

{
  "mappings": {
    "properties": {
      "ip_addr": {
        "type": "ip"
      }
    }
  }
}


{
  "ip_addr": "192.168.1.1"
}


{
  "query": {
    "term": {
      "ip_addr": "192.168.0.0/16"
    }
  }
}
```

<note>
  You can also store ip ranges in a single field using an [ip_range data type](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/range).
</note>


## Parameters for `ip` fields

The following parameters are accepted by `ip` fields:
<definitions>
  <definition term="doc_values">
    Should the field be stored on disk in a column-stride fashion, so that it can later be used for sorting, aggregations, or scripting? Accepts `true` (default) or `false`.
  </definition>
  <definition term="ignore_malformed">
    If `true`, malformed IP addresses are ignored. If `false` (default), malformed IP addresses throw an exception and reject the whole document. Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="index">
    Should the field be quickly searchable? Accepts `true` (default) and `false`. Fields that only have [`doc_values`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/doc-values) enabled can still be queried using term or range-based queries, albeit slower.
  </definition>
  <definition term="null_value">
    Accepts an IPv4 or IPv6 value which is substituted for any explicit `null` values. Defaults to `null`, which means the field is treated as missing. Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="on_script_error">
    Defines what to do if the script defined by the `script` parameter throws an error at indexing time. Accepts `reject` (default), which will cause the entire document to be rejected, and `ignore`, which will register the field in the document’s [`_ignored`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-ignored-field) metadata field and continue indexing. This parameter can only be set if the `script` field is also set.
  </definition>
  <definition term="script">
    If this parameter is set, then the field will index values generated by this script, rather than reading the values directly from the source. If a value is set for this field on the input document, then the document will be rejected with an error. Scripts are in the same format as their [runtime equivalent](https://www.elastic.co/docs/manage-data/data-store/mapping/map-runtime-field), and should emit strings containing IPv4 or IPv6 formatted addresses.
  </definition>
  <definition term="store">
    Whether the field value should be stored and retrievable separately from the [`_source`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-source-field) field. Accepts `true` or `false` (default).
  </definition>
  <definition term="time_series_dimension">
    (Optional, Boolean)
    Marks the field as a [time series dimension](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds#time-series-dimension). Defaults to `false`.
    The `index.mapping.dimension_fields.limit` [index setting](https://www.elastic.co/docs/reference/elasticsearch/index-settings/time-series) limits the number of dimensions in an index.
    Dimension fields have the following constraints:
    - The `doc_values` and `index` mapping parameters must be `true`.
  </definition>
</definitions>


## Querying `ip` fields

The most common way to query ip addresses is to use the [CIDR](https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation) notation: `[ip_address]/[prefix_length]`. For instance:
```json

{
  "query": {
    "term": {
      "ip_addr": "192.168.0.0/16"
    }
  }
}
```

or
```json

{
  "query": {
    "term": {
      "ip_addr": "2001:db8::/48"
    }
  }
}
```

Also beware that colons are special characters to the [`query_string`](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-query-string-query) query, so ipv6 addresses will need to be escaped. The easiest way to do so is to put quotes around the searched value:
```json

{
  "query": {
    "query_string" : {
      "query": "ip_addr:\"2001:db8::/48\""
    }
  }
}
```


## Synthetic `_source`

Synthetic source may sort `ip` field values and remove duplicates. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "ip": { "type": "ip" }
    }
  }
}

{
  "ip": ["192.168.0.1", "192.168.0.1", "10.10.12.123",
         "2001:db8::1:0:0:1", "::afff:4567:890a"]
}
```

Will become:
```json
{
  "ip": ["::afff:4567:890a", "10.10.12.123", "192.168.0.1", "2001:db8::1:0:0:1"]
}
```

<note>
  IPv4 addresses are sorted as though they were IPv6 addresses prefixed by `::ffff:0:0:0/96` as specified by [rfc6144](https://datatracker.ietf.org/doc/html/rfc6144).
</note>