﻿---
title: Geopoint field type
description: Fields of type geo_point accept latitude-longitude pairs, which can be used: to find geopoints within a bounding box, within a certain distance of a central...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/geo-point
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Geopoint field type
Fields of type `geo_point` accept latitude-longitude pairs, which can be used:
- to find geopoints within a [bounding box](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-geo-bounding-box-query),
  within a certain [distance](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-geo-distance-query) of a central point,
  or within a [`geo_shape` query](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-geo-shape-query) (for example, points in a polygon).
- to aggregate documents by [distance](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-geodistance-aggregation) from a central point.
- to aggregate documents by geographic grids: either
  [`geo_hash`](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-geohashgrid-aggregation),
  [`geo_tile`](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-geotilegrid-aggregation) or
  [`geo_hex`](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-geohexgrid-aggregation).
- to aggregate geopoints into a track using the metrics aggregation
  [`geo_line`](https://www.elastic.co/docs/reference/aggregations/search-aggregations-metrics-geo-line).
- to integrate distance into a document’s [relevance score](https://www.elastic.co/docs/reference/query-languages/query-dsl/query-dsl-function-score-query).
- to [sort](/docs/reference/elasticsearch/rest-apis/sort-search-results#geo-sorting) documents by distance.

As with [geo_shape](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/geo-shape) and [point](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/point), `geo_point` can be specified in [GeoJSON](http://geojson.org)
and [Well-Known Text](https://docs.opengeospatial.org/is/12-063r5/12-063r5.html) formats.
However, there are a number of additional formats that are supported for convenience and historical reasons.
In total there are six ways that a geopoint may be specified, as demonstrated below:
```json

{
  "mappings": {
    "properties": {
      "location": {
        "type": "geo_point"
      }
    }
  }
}


{
  "text": "Geopoint as an object using GeoJSON format",
  "location": { <1>
    "type": "Point",
    "coordinates": [-71.34, 41.12]
  }
}


{
  "text": "Geopoint as a WKT POINT primitive",
  "location" : "POINT (-71.34 41.12)" <2>
}


{
  "text": "Geopoint as an object with 'lat' and 'lon' keys",
  "location": { <3>
    "lat": 41.12,
    "lon": -71.34
  }
}


{
  "text": "Geopoint as an array",
  "location": [ -71.34, 41.12 ] <4>
}


{
  "text": "Geopoint as a string",
  "location": "41.12,-71.34" <5>
}


{
  "text": "Geopoint as a geohash",
  "location": "drm3btev3e86" <6>
}


{
  "query": {
    "geo_bounding_box": { <7>
      "location": {
        "top_left": {
          "lat": 42,
          "lon": -72
        },
        "bottom_right": {
          "lat": 40,
          "lon": -74
        }
      }
    }
  }
}
```

<admonition title="Geopoints expressed as an array or string">
  Please note that string geopoints are ordered as `lat,lon`, while array
  geopoints, GeoJSON and WKT are ordered as the reverse: `lon,lat`.The reasons for this are historical. Geographers traditionally write `latitude`
  before `longitude`, while recent formats specified for geographic data like
  [GeoJSON](https://geojson.org/) and [Well-Known Text](https://docs.opengeospatial.org/is/12-063r5/12-063r5.html)
  order `longitude` before `latitude` (easting before northing) in order to match
  the mathematical convention of ordering `x` before `y`.
</admonition>

<note>
  A point can be expressed as a [geohash](https://en.wikipedia.org/wiki/Geohash).
  Geohashes are [base32](https://en.wikipedia.org/wiki/Base32) encoded strings of
  the bits of the latitude and longitude interleaved. Each character in a geohash
  adds additional 5 bits to the precision. So the longer the hash, the more
  precise it is. For the indexing purposed geohashs are translated into
  latitude-longitude pairs. During this process only first 12 characters are
  used, so specifying more than 12 characters in a geohash doesn’t increase the
  precision. The 12 characters provide 60 bits, which should reduce a possible
  error to less than 2cm.
</note>


## Parameters for `geo_point` fields

The following parameters are accepted by `geo_point` fields:
<definitions>
  <definition term="ignore_malformed">
    If `true`, malformed geopoints are ignored.
    If `false` (default), malformed geopoints throw an exception and reject the whole document.
    A geopoint is considered malformed if its latitude is outside the range -90 ⇐ latitude ⇐ 90,
    or if its longitude is outside the range -180 ⇐ longitude ⇐ 180.
    When set to `true`, if the format is valid, but the values are out of range,
    the values will be normalized into the valid range, and the document will be indexed.
    This is a special case, and a [different behaviour](/docs/reference/elasticsearch/mapping-reference/ignore-malformed#_ignore_malformed_geo_point) from the normal for `ignore_malformed`.
    Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="ignore_z_value">
    If `true` (default) three dimension points will be accepted (stored in source)
    but only latitude and longitude values will be indexed; the third dimension is
    ignored. If `false`, geopoints containing any more than latitude and longitude
    (two dimensions) values throw an exception and reject the whole document. Note
    that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="index">
    Should the field be quickly searchable? Accepts `true` (default) and
    `false`. Fields that only have [`doc_values`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/doc-values)
    enabled can still be queried, albeit slower.
  </definition>
  <definition term="null_value">
    Accepts a geopoint value which is substituted for any explicit `null` values.
    Defaults to `null`, which means the field is treated as missing. Note that this
    cannot be set if the `script` parameter is used.
  </definition>
  <definition term="on_script_error">
    Defines what to do if the script defined by the `script` parameter
    throws an error at indexing time. Accepts `fail` (default), which
    will cause the entire document to be rejected, and `continue`, which
    will register the field in the document’s [`_ignored`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-ignored-field) metadata field and continue
    indexing. This parameter can only be set if the `script` field is
    also set.
  </definition>
  <definition term="script">
    If this parameter is set, then the field will index values generated
    by this script, rather than reading the values directly from the
    source. If a value is set for this field on the input document, then
    the document will be rejected with an error.
    Scripts are in the same format as their [runtime equivalent](https://www.elastic.co/docs/manage-data/data-store/mapping/map-runtime-field), and should emit points
    as a pair of (lat, lon) double values.
  </definition>
</definitions>


## Using geopoints in scripts

When accessing the value of a geopoint in a script, the value is returned as
a `GeoPoint` object, which allows access to the `.lat` and `.lon` values
respectively:
```java
def geopoint = doc['location'].value;
def lat      = geopoint.lat;
def lon      = geopoint.lon;
```

For performance reasons, it is better to access the lat/lon values directly:
```java
def lat      = doc['location'].lat;
def lon      = doc['location'].lon;
```


## Synthetic source

Synthetic source may sort `geo_point` fields (first by latitude and then
longitude) and reduces them to their stored precision. Additionally, unlike most
types, arrays of `geo_point` fields will not preserve order if
`synthetic_source_keep` is set to `arrays`. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "point": {
        "type": "geo_point",
        "synthetic_source_keep": "arrays"
      }
    }
  }
}

{
  "point": [
    {"lat":10, "lon":30},
    {"lat":-90, "lon":-80}
  ]
}
```

Will become:
```json
{
  "point": [
    {"lat":-90.0, "lon":-80.00000000931323},
    {"lat":9.999999990686774, "lon":29.999999972060323}
   ]
}
```