﻿---
title: Date field type
description: JSON doesn’t have a date data type, so dates in Elasticsearch can either be: strings containing formatted dates, e.g. "2015-01-01" or "2015/01/01 12:10:30".a...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Date field type
JSON doesn’t have a date data type, so dates in Elasticsearch can either be:
- strings containing formatted dates, e.g. `"2015-01-01"` or `"2015/01/01 12:10:30"`.
- a number representing *milliseconds-since-the-epoch*.
- a number representing *seconds-since-the-epoch* ([configuration](#date-epoch-seconds)).

Internally, dates are converted to UTC (if the time-zone is specified) and stored as a long number representing milliseconds-since-the-epoch.
<note>
  Use the [date_nanos](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/date_nanos) field type if a nanosecond resolution is expected.
</note>

Queries on dates are internally converted to range queries on this long representation, and the result of aggregations and stored fields is converted back to a string depending on the date format that is associated with the field.
<note>
  Dates will always be rendered as strings, even if they were initially supplied as a long in the JSON document.
</note>

Date formats can be customised, but if no `format` is specified then it uses the default:
```js
    "strict_date_optional_time||epoch_millis"
```

This means that it will accept dates with optional timestamps, which conform to the formats supported by [`strict_date_optional_time`](/docs/reference/elasticsearch/mapping-reference/mapping-date-format#strict-date-time) or milliseconds-since-the-epoch.
For instance:

```json

{
  "mappings": {
    "properties": {
      "date": {
        "type": "date" <1>
      }
    }
  }
}


{ "date": "2015-01-01" } <2>


{ "date": "2015-01-01T12:10:30Z" } <3>


{ "date": 1420070400001 } <4>


{
  "sort": { "date": "asc"} <5>
}
```

<warning>
  Dates will accept numbers with a decimal point like `{"date": 1618249875.123456}` but there are some cases ([#70085](https://github.com/elastic/elasticsearch/issues/70085)) where we’ll lose precision on those dates so they should be avoided.
</warning>


## Multiple date formats

Multiple formats can be specified by separating them with `||` as a separator. Each format will be tried in turn until a matching format is found. The first format will be used to convert the *milliseconds-since-the-epoch* value back into a string.

```json

{
  "mappings": {
    "properties": {
      "date": {
        "type":   "date",
        "format": "yyyy-MM-dd HH:mm:ss||yyyy-MM-dd||epoch_millis"
      }
    }
  }
}
```


## Parameters for `date` fields

The following parameters are accepted by `date` fields:
<definitions>
  <definition term="doc_values">
    Should the field be stored on disk in a column-stride fashion, so that it can later be used for sorting, aggregations, or scripting? Accepts `true` (default) or `false`.
  </definition>
  <definition term="format">
    The date format(s) that can be parsed. Defaults to `strict_date_optional_time||epoch_millis`.
  </definition>
  <definition term="locale">
    The locale to use when parsing dates since months do not have the same names and/or abbreviations in all languages. The default is ENGLISH.
  </definition>
  <definition term="ignore_malformed">
    If `true`, malformed numbers are ignored. If `false` (default), malformed numbers throw an exception and reject the whole document. Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="index">
    Should the field be quickly searchable? Accepts `true` (default) and `false`. Date fields that only have [`doc_values`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/doc-values) enabled can also be queried, albeit slower.
  </definition>
  <definition term="null_value">
    Accepts a date value in one of the configured `format’s as the field which is substituted for any explicit `null`values. Defaults to`null`, which means the field is treated as missing.  Note that this cannot be set of the `script` parameter is used.
  </definition>
  <definition term="on_script_error">
    Defines what to do if the script defined by the `script` parameter throws an error at indexing time. Accepts `fail` (default), which will cause the entire document to be rejected, and `continue`, which will register the field in the document’s [`_ignored`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-ignored-field) metadata field and continue indexing. This parameter can only be set if the `script` field is also set.
  </definition>
  <definition term="script">
    If this parameter is set, then the field will index values generated by this script, rather than reading the values directly from the source. If a value is set for this field on the input document, then the document will be rejected with an error. Scripts are in the same format as their [runtime equivalent](https://www.elastic.co/docs/manage-data/data-store/mapping/map-runtime-field), and should emit long-valued timestamps.
  </definition>
  <definition term="store">
    Whether the field value should be stored and retrievable separately from the [`_source`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-source-field) field. Accepts `true` or `false` (default).
  </definition>
  <definition term="meta">
    Metadata about the field.
  </definition>
</definitions>


## Epoch seconds

If you need to send dates as *seconds-since-the-epoch* then make sure the `format` lists `epoch_second`:

```json

{
  "mappings": {
    "properties": {
      "date": {
        "type":   "date",
        "format": "strict_date_optional_time||epoch_second"
      }
    }
  }
}


{ "date": 1618321898 }


{
  "fields": [ {"field": "date"}],
  "_source": false
}
```

Which will reply with a date like:
```json
{
  "hits": {
    "hits": [
      {
        "_id": "example",
        "_index": "my-index-000001",
        "_score": 1.0,
        "fields": {
          "date": ["2021-04-13T13:51:38.000Z"]
        }
      }
    ]
  }
}
```


## Synthetic `_source`

Synthetic source may sort `date` field values. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "date": { "type": "date" }
    }
  }
}

{
  "date": ["2015-01-01T12:10:30Z", "2014-01-01T12:10:30Z"]
}
```

Will become:
```json
{
  "date": ["2014-01-01T12:10:30.000Z", "2015-01-01T12:10:30.000Z"]
}
```