﻿---
title: Boolean field type
description: Boolean fields accept JSON true and false values, but can also accept strings which are interpreted as either true or false: For example: Aggregations...
url: https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/boolean
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Boolean field type
Boolean fields accept JSON `true` and `false` values, but can also accept strings which are interpreted as either true or false:
<definitions>
  <definition term="False values">
    `false`, `"false"`, `""` (empty string)
  </definition>
  <definition term="True values">
    `true`, `"true"`
  </definition>
</definitions>

For example:
```json

{
  "mappings": {
    "properties": {
      "is_published": {
        "type": "boolean"
      }
    }
  }
}


{
  "is_published": "true" <1>
}


{
  "query": {
    "term": {
      "is_published": true <2>
    }
  }
}
```

Aggregations like the [`terms` aggregation](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-terms-aggregation)  use `1` and `0` for the `key`, and the strings `"true"` and `"false"` for the `key_as_string`. Boolean fields when used in scripts, return `true` and `false`:
```json

{
  "is_published": true
}


{
  "is_published": false
}


{
  "aggs": {
    "publish_state": {
      "terms": {
        "field": "is_published"
      }
    }
  },
  "sort": [ "is_published" ],
  "fields": [
    {"field": "weight"}
  ],
  "runtime_mappings": {
    "weight": {
      "type": "long",
      "script": "emit(doc['is_published'].value ? 10 : 0)"
    }
  }
}
```


## Parameters for `boolean` fields

The following parameters are accepted by `boolean` fields:
<definitions>
  <definition term="doc_values">
    Should the field be stored on disk in a column-stride fashion, so that it can later be used for sorting, aggregations, or scripting? Accepts `true` (default) or `false`.
  </definition>
  <definition term="index">
    Should the field be quickly searchable? Accepts `true` (default) and `false`. Fields that only have [`doc_values`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/doc-values) enabled can still be queried using term or range-based queries, albeit slower.
  </definition>
  <definition term="ignore_malformed">
    Trying to index the wrong data type into a field throws an exception by default, and rejects the whole document. If this parameter is set to true, it allows the exception to be ignored. The malformed field is not indexed, but other fields in the document are processed normally. Accepts `true` or `false`. Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="null_value">
    Accepts any of the true or false values listed above. The value is substituted for any explicit `null` values. Defaults to `null`, which means the field is treated as missing. Note that this cannot be set if the `script` parameter is used.
  </definition>
  <definition term="on_script_error">
    Defines what to do if the script defined by the `script` parameter throws an error at indexing time. Accepts `fail` (default), which will cause the entire document to be rejected, and `continue`, which will register the field in the document’s [`_ignored`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-ignored-field) metadata field and continue indexing. This parameter can only be set if the `script` field is also set.
  </definition>
  <definition term="script">
    If this parameter is set, then the field will index values generated by this script, rather than reading the values directly from the source. If a value is set for this field on the input document, then the document will be rejected with an error. Scripts are in the same format as their [runtime equivalent](https://www.elastic.co/docs/manage-data/data-store/mapping/map-runtime-field).
  </definition>
  <definition term="store">
    Whether the field value should be stored and retrievable separately from the [`_source`](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/mapping-source-field) field. Accepts `true` or `false` (default).
  </definition>
  <definition term="meta">
    Metadata about the field.
  </definition>
  <definition term="time_series_dimension">
    (Optional, Boolean)
    Marks the field as a [time series dimension](https://www.elastic.co/docs/manage-data/data-store/data-streams/time-series-data-stream-tsds#time-series-dimension). Defaults to `false`.
    The `index.mapping.dimension_fields.limit` [index setting](https://www.elastic.co/docs/reference/elasticsearch/index-settings/time-series) index setting limits the number of dimensions in an index.
    Dimension fields have the following constraints:
    - The `doc_values` and `index` mapping parameters must be `true`.
  </definition>
</definitions>


## Synthetic `_source`

`boolean` fields support [synthetic `_source`](/docs/reference/elasticsearch/mapping-reference/mapping-source-field#synthetic-source) in their default configuration.
Synthetic source may sort `boolean` field values. For example:

```json

{
  "settings": {
    "index": {
      "mapping": {
        "source": {
          "mode": "synthetic"
        }
      }
    }
  },
  "mappings": {
    "properties": {
      "bool": { "type": "boolean" }
    }
  }
}

{
  "bool": [true, false, true, false]
}
```

Will become:
```json
{
  "bool": [false, false, true, true]
}
```