﻿---
title: Security settings in Elasticsearch
description: You configure xpack.security settings to enable anonymous access and perform message authentication, set up document and field level security, configure...
url: https://www.elastic.co/docs/reference/elasticsearch/configuration-reference/security-settings
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Hosted: Generally available
  - Self-managed Elastic deployments: Generally available
---

# Security settings in Elasticsearch
You configure `xpack.security` settings to [enable anonymous access](#anonymous-access-settings) and perform message authentication, [set up document and field level security](#field-document-security-settings), [configure realms](#realm-settings), [encrypt communications with SSL](#ssl-tls-settings),and [audit security events](https://www.elastic.co/docs/reference/elasticsearch/configuration-reference/auding-settings).
All of these settings can be added to the `elasticsearch.yml` configuration file, with the exception of the secure settings, which you add to the Elasticsearch keystore. For more information about creating and updating the Elasticsearch keystore, see [Secure settings](https://www.elastic.co/docs/deploy-manage/security/secure-settings).

### General security settings

<definitions>
  <definition term="xpack.security.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Defaults to `true`, which enables Elasticsearch security features on the node. This setting must be enabled to use Elasticsearch’s authentication, authorization and audit features.
    If set to `false`, security features are disabled, which is not recommended. It also affects all Kibana instances that connect to this Elasticsearch instance; you do not need to disable security features in those `kibana.yml` files. For more information about disabling security features in specific Kibana instances, see [Kibana security settings](https://www.elastic.co/docs/reference/kibana/configuration-reference/security-settings).
  </definition>
  <definition term="xpack.security.autoconfiguration.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Defaults to `true`, which enables [security auto configuration](https://www.elastic.co/docs/deploy-manage/deploy/self-managed/installing-elasticsearch).
    If set to `false`, security auto configuration is disabled, which is not recommended. When disabled, security is not configured automatically when starting Elasticsearch for the first time, which means that you must [manually configure security](https://www.elastic.co/docs/deploy-manage/security/self-setup).
  </definition>
  <definition term="xpack.security.enrollment.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Defaults to `false`. Controls enrollment (of nodes and Kibana) to a local node that’s been [autoconfigured for security](https://www.elastic.co/docs/deploy-manage/deploy/self-managed/installing-elasticsearch). When set to `true`, the local node can generate new enrollment tokens. Existing tokens can be used for enrollment if they are still valid.
    The security autoconfiguration process will set this to `true` unless an administrator sets it to `false` before starting Elasticsearch.
  </definition>
  <definition term="xpack.security.hide_settings">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A comma-separated list of settings that are omitted from the results of the [cluster nodes info API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-nodes-info). You can use wildcards to include multiple settings in the list. For example, the following value hides all the settings for the ad1 active_directory realm: `xpack.security.authc.realms.active_directory.ad1.*`. The API already omits all `ssl` settings, `bind_dn`, and `bind_password` due to the sensitive nature of the information.
  </definition>
  <definition term="xpack.security.fips_mode.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Enables fips mode of operation. Set this to `true` if you run this Elasticsearch instance in a FIPS 140-2 enabled JVM. For more information, see [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips). Defaults to `false`.
  </definition>
  <definition term="xpack.security.fips_mode.required_providers">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Optionally enforce specific Java JCE/JSSE security providers. For example, set this to `["BCFIPS", "BCJSSE"]` (case-insensitive) to require the Bouncy Castle FIPS JCE and JSSE security providers. Only applicable when `xpack.security.fips_mode.enabled` is set to `true`.
  </definition>
</definitions>


### Password hashing settings

<definitions>
  <definition term="xpack.security.authc.password_hashing.algorithm">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the hashing algorithm that is used for secure user credential storage. See [Table 2, Password hashing algorithms](#password-hashing-algorithms). If `xpack.security.fips_mode.enabled` is true (see [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips)), defaults to `pbkdf2_stretch`. In all other cases, defaults to `bcrypt`.
  </definition>
</definitions>


### Anonymous access settings

You can configure the following anonymous access settings in `elasticsearch.yml`. For more information, see [Enabling anonymous access](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/anonymous-access).
<definitions>
  <definition term="xpack.security.authc.anonymous.username logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The username (principal) of the anonymous user. Defaults to `_es_anonymous_user`.
  </definition>
  <definition term="xpack.security.authc.anonymous.roles logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The roles to associate with the anonymous user. Required.
  </definition>
  <definition term="xpack.security.authc.anonymous.authz_exception logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) When `true`, an HTTP 403 response is returned if the anonymous user does not have the appropriate permissions for the requested action. The user is not prompted to provide credentials to access the requested resource. When set to `false`, an HTTP 401 response is returned and the user can provide credentials with the appropriate permissions to gain access. Defaults to `true`.
  </definition>
</definitions>


### Automata Settings

In places where the security features accept wildcard patterns (e.g. index patterns in roles, group matches in the role mapping API), each pattern is compiled into an Automaton. The follow settings are available to control this behaviour.
<definitions>
  <definition term="xpack.security.automata.max_determinized_states">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The upper limit on how many automaton states may be created by a single pattern. This protects against too-difficult (e.g. exponentially hard) patterns. Defaults to `100,000`.
  </definition>
  <definition term="xpack.security.automata.cache.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Whether to cache the compiled automata. Compiling automata can be CPU intensive and may slowdown some operations. The cache reduces the frequency with which automata need to be compiled. Defaults to `true`.
  </definition>
  <definition term="xpack.security.automata.cache.size">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum number of items to retain in the automata cache. Defaults to `10,000`.
  </definition>
  <definition term="xpack.security.automata.cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The length of time to retain in an item in the automata cache (based on most recent usage). Defaults to `48h` (48 hours).
  </definition>
</definitions>


### Document and field level security settings

You can set the following document and field level security settings in `elasticsearch.yml`. For more information, see [Setting up field and document level security](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/controlling-access-at-document-field-level).
<definitions>
  <definition term="xpack.security.dls_fls.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set to `false` to prevent document and field level security from being configured. Defaults to `true`.
  </definition>
  <definition term="xpack.security.dls.bitset.cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The time-to-live for cached `BitSet` entries for document level security. Document level security queries may depend on Lucene BitSet objects, and these are automatically cached to improve performance. Defaults to expire entries that are unused for `2h` (2 hours).
  </definition>
  <definition term="xpack.security.dls.bitset.cache.size">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum memory usage of cached `BitSet` entries for document level security. Document level security queries may depend on Lucene BitSet objects, and these are automatically cached to improve performance. Can be configured as a raw number of bytes (such as `200mb` or `1g`) or a percentage of the node’s JVM heap memory (such as `5%`). When the default value is exceeded, the least recently used entries are evicted. Defaults to `10%` of the heap assigned to the node.
  </definition>
</definitions>


### Token service settings

You can set the following token service settings in `elasticsearch.yml`.
<definitions>
  <definition term="xpack.security.authc.token.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set to `false` to disable the built-in token service. Defaults to `true` unless `xpack.security.http.ssl.enabled` is `false`. This prevents sniffing the token from a connection over plain http.
  </definition>
  <definition term="xpack.security.authc.token.timeout">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The length of time that a token is valid for. By default this value is `20m` or 20 minutes. The maximum value is 1 hour.
  </definition>
</definitions>


### API key service settings

You can set the following API key service settings in `elasticsearch.yml`.
<definitions>
  <definition term="xpack.security.authc.api_key.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set to `false` to disable the built-in API key service. Defaults to `true`.
  </definition>
  <definition term="xpack.security.authc.api_key.cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The time-to-live for cached API key entries. A API key id and a hash of its API key are cached for this period of time. Specify the time period using the standard Elasticsearch [time units](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units). Defaults to `1d`.
  </definition>
  <definition term="xpack.security.authc.api_key.cache.max_keys">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum number of API key entries that can live in the cache at any given time. Defaults to 10,000.
  </definition>
  <definition term="xpack.security.authc.api_key.cache.hash_algo">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting), Expert) The hashing algorithm that is used for the in-memory cached API key credentials. For possible values, see [Table 1, Cache hash algorithms](#cache-hash-algo). Defaults to `ssha256`.
  </definition>
</definitions>


<definitions>
  <definition term="xpack.security.authc.api_key.delete.retention_period">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) Invalidated or expired API keys older than the retention period are eligible for deletion. Defaults to `7d`.
  </definition>
</definitions>

<note>
  Large real-time clock inconsistency across cluster nodes can cause problems with evaluating the API key retention period. That is, if the clock on the node invalidating the API key is significantly different than the one performing the deletion, the key may be retained for longer or shorter than the configured retention period.
</note>

<definitions>
  <definition term="xpack.security.authc.api_key.delete.interval">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting), Expert) Cluster nodes schedule the automatic deletion of invalidated or expired API keys that are older than the retention period. This setting controls the minimum time interval between two such deletion jobs. Defaults to `24h`.
    <note>
      This is a low-level setting that currently controls the interval between deletion jobs triggered per-node, not across the cluster.
    </note>
  </definition>
  <definition term="xpack.security.authc.api_key.delete.timeout">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting), Expert) Sets the timeout of the internal search and delete call.
  </definition>
  <definition term="xpack.security.authc.api_key.hashing.algorithm">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the hashing algorithm that is used for securing API key credentials. See [Table 3, Secure token hashing algorithms](#secure-token-hashing-algorithms). Defaults to `ssha256`.
  </definition>
</definitions>


### Security domain settings

You configure security domain settings in the `xpack.security.authc.domains` namespace in `elasticsearch.yml`.
For example:
```yaml
xpack:
  security:
    authc:
      domains:
        my_domain: 
          realms: [ 'default_native', 'saml1' ] 
```


### Realm settings

<definitions>
  <definition term="xpack.security.authc.realms.saml.* logo cloud">
    To learn more on how to enable SAML and related user settings, check [secure your clusters with SAML](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/saml).
  </definition>
  <definition term="xpack.security.authc.realms.oidc.* logo cloud">
    To learn more on how to enable OpenID Connect and related user settings, check [secure your clusters with OpenID Connect](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/openid-connect).
  </definition>
  <definition term="xpack.security.authc.realms.kerberos.* logo cloud">
    To learn more on how to enable Kerberos and relate user settings, check [secure your clusters with Kerberos](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/kerberos).
  </definition>
  <definition term="xpack.security.authc.realms.jwt.* logo cloud">
    To learn more on how to enable JWT and related user settings, check [secure your clusters with JWT](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/jwt).
  </definition>
</definitions>

<note>
  All SAML, OpenID Connect, Kerberos, and JWT settings are allowlisted.
</note>

You configure realm settings in the `xpack.security.authc.realms` namespace in `elasticsearch.yml`.
For example:
```yaml
xpack.security.authc.realms:

    native.realm1: 
        order: 0 
        ...

    ldap.realm2:
        order: 1
        ...

    active_directory.realm3:
        order: 2
        ...
    ...
```

The valid settings vary depending on the realm type. For more information, see [*User authentication*](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/user-authentication).

#### Settings valid for all realms

<definitions>
  <definition term="order">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The priority of the realm within the realm chain. Realms with a lower order are consulted first. The value must be unique for each realm. This setting is required.
  </definition>
  <definition term="enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Indicates whether a realm is enabled. You can use this setting to disable a realm without removing its configuration information. Defaults to `true`.
  </definition>
</definitions>


#### Native realm settings

In addition to the [settings that are valid for all realms](#ref-realm-settings), you can specify the following optional settings:
<definitions>
  <definition term="cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The time-to-live for cached user entries. A user and a hash of its credentials are cached for this period of time. Specify the time period using the standard Elasticsearch [time units](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units). Defaults to `20m`.
  </definition>
  <definition term="cache.max_users">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum number of user entries that can live in the cache at any given time. Defaults to 100,000.
  </definition>
  <definition term="cache.hash_algo">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting), Expert) The hashing algorithm that is used for the in-memory cached user credentials. For possible values, see [Table 1, Cache hash algorithms](#cache-hash-algo). Defaults to `ssha256`.
  </definition>
  <definition term="authentication.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) If set to `false`, disables authentication support in this realm, so that it only supports user lookups. (See the [run as](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/submitting-requests-on-behalf-of-other-users) and [authorization realms](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms) features). Defaults to `true`.
  </definition>
</definitions>


#### File realm settings

In addition to the [settings that are valid for all realms](#ref-realm-settings), you can specify the following settings:
<definitions>
  <definition term="cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The time-to-live for cached user entries. A user and a hash of its credentials are cached for this configured period of time. Defaults to `20m`. Specify values using the standard Elasticsearch [time units](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units). Defaults to `20m`.
  </definition>
  <definition term="cache.max_users">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum number of user entries that can live in the cache at a given time. Defaults to 100,000.
  </definition>
  <definition term="cache.hash_algo">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting), Expert) The hashing algorithm that is used for the in-memory cached user credentials. See [Table 1, Cache hash algorithms](#cache-hash-algo). Defaults to `ssha256`.
  </definition>
  <definition term="authentication.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) If set to `false`, disables authentication support in this realm, so that it only supports user lookups. (See the [run as](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/submitting-requests-on-behalf-of-other-users) and [authorization realms](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms) features). Defaults to `true`.
  </definition>
</definitions>


#### LDAP realm settings

In addition to the [Settings valid for all realms](#ref-realm-settings), you can specify the following settings:
<definitions>
  <definition term="url">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) One or more LDAP URLs in the `ldap[s]://<server>:<port>` format. Required.
    To provide multiple URLs, use a YAML array (`["ldap://server1:636", "ldap://server2:636"]`) or comma-separated string (`"ldap://server1:636, ldap://server2:636"`).
    While both are supported, you can’t mix the `ldap` and `ldaps` protocols.
  </definition>
  <definition term="load_balance.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The behavior to use when there are multiple LDAP URLs defined. For supported values see [load balancing and failover types](#load-balancing). Defaults to `failover`.
  </definition>
  <definition term="load_balance.cache_ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) When using `dns_failover` or `dns_round_robin` as the load balancing type, this setting controls the amount of time to cache DNS lookups. Defaults to `1h`.
  </definition>
  <definition term="bind_dn">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The DN of the user that is used to bind to the LDAP and perform searches. Only applicable in user search mode. If not specified, an anonymous bind is attempted. Defaults to Empty. Due to its potential security impact, `bind_dn` is not exposed via the [nodes info API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-nodes-info).
  </definition>
  <definition term="bind_password">
    <admonition title="Deprecated in 6.3">
      Use `secure_bind_password` instead.
    </admonition>
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the user that is used to bind to the LDAP directory. Defaults to Empty. Due to its potential security impact, `bind_password` is not exposed via the [nodes info API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-nodes-info).
  </definition>
  <definition term="secure_bind_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings), [Reloadable](https://www.elastic.co/docs/deploy-manage/security/secure-settings#reloadable-secure-settings)) The password for the user that is used to bind to the LDAP directory. Defaults to Empty.
  </definition>
  <definition term="user_dn_templates">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The DN template that replaces the user name with the string `{{0}}`. This setting is multivalued; you can specify multiple user contexts. Required to operate in user template mode. If `user_search.base_dn` is specified, this setting is not valid. For more information on the different modes, see [LDAP user authentication](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/ldap).
  </definition>
  <definition term="authorization_realms">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The names of the realms that should be consulted for delegated authorization. If this setting is used, then the LDAP realm does not perform role mapping and instead loads the user from the listed realms. The referenced realms are consulted in the order that they are defined in this list. See [Delegating authorization to another realm](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms).
    <note>
      If any settings starting with `user_search` are specified, the `user_dn_templates` settings are ignored.
    </note>
  </definition>
  <definition term="user_group_attribute">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the attribute to examine on the user for group membership. If any `group_search` settings are specified, this setting is ignored. Defaults to `memberOf`.
  </definition>
  <definition term="user_full_name_attribute">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the attribute to examine on the user for the full name of the user. Defaults to `cn`.
  </definition>
  <definition term="user_email_attribute">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the attribute to examine on the user for the email address of the user. Defaults to `mail`.
  </definition>
  <definition term="user_search.base_dn">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies a container DN to search for users. Required to operated in user search mode. If `user_dn_templates` is specified, this setting is not valid. For more information on the different modes, see [LDAP user authentication](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/ldap).
  </definition>
  <definition term="user_search.scope">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The scope of the user search. Valid values are `sub_tree`, `one_level` or `base`. `one_level` only searches objects directly contained within the `base_dn`. `sub_tree` searches all objects contained under `base_dn`. `base` specifies that the `base_dn` is the user object, and that it is the only user considered. Defaults to `sub_tree`.
  </definition>
  <definition term="user_search.filter">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the filter used to search the directory in attempts to match an entry with the username provided by the user. Defaults to `(uid={{0}})`. `{{0}}` is substituted with the username provided when searching.
  </definition>
  <definition term="user_search.attribute">
    <admonition title="Deprecated in 5.6">
      Use `user_search.filter` instead.
    </admonition>
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The attribute to match with the username sent with the request. Defaults to `uid`.
  </definition>
  <definition term="user_search.pool.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Enables or disables connection pooling for user search. If set to `false`, a new connection is created for every search. The default is `true` when `bind_dn` is set.
  </definition>
  <definition term="user_search.pool.size">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum number of connections to the LDAP server to allow in the connection pool. Defaults to `20`.
  </definition>
  <definition term="user_search.pool.initial_size">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The initial number of connections to create to the LDAP server on startup. Defaults to `0`. If the LDAP server is down, values greater than `0` could cause startup failures.
  </definition>
  <definition term="user_search.pool.health_check.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Enables or disables a health check on LDAP connections in the connection pool. Connections are checked in the background at the specified interval. Defaults to `true`.
  </definition>
  <definition term="user_search.pool.health_check.dn">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The distinguished name that is retrieved as part of the health check. Defaults to the value of `bind_dn` if present; if not, falls back to `user_search.base_dn`.
  </definition>
  <definition term="user_search.pool.health_check.interval">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The interval to perform background checks of connections in the pool. Defaults to `60s`.
  </definition>
  <definition term="group_search.base_dn">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The container DN to search for groups in which the user has membership. When this element is absent, Elasticsearch searches for the attribute specified by `user_group_attribute` set on the user in order to determine group membership.
  </definition>
  <definition term="group_search.scope">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether the group search should be `sub_tree`, `one_level` or `base`.  `one_level` only searches objects directly contained within the `base_dn`. `sub_tree` searches all objects contained under `base_dn`. `base` specifies that the `base_dn` is a group object, and that it is the only group considered. Defaults to  `sub_tree`.
  </definition>
  <definition term="group_search.filter">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies a filter to use to look up a group. When not set, the realm searches for `group`, `groupOfNames`, `groupOfUniqueNames`, or `posixGroup` with the attributes `member`, `memberOf`, or `memberUid`. Any instance of `{{0}}` in the filter is replaced by the user attribute defined in `group_search.user_attribute`.
  </definition>
  <definition term="group_search.user_attribute">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the user attribute that is fetched and provided as a parameter to the filter. If not set, the user DN is passed into the filter. Defaults to Empty.
  </definition>
  <definition term="unmapped_groups_as_roles">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) If set to `true`, the names of any unmapped LDAP groups are used as role names and assigned to the user. A group is considered to be *unmapped* if it is not referenced in a [role-mapping file](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/mapping-users-groups-to-roles#mapping-roles-file). API-based role mappings are not considered. Defaults to `false`.
  </definition>
  <definition term="files.role_mapping">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The [location](https://www.elastic.co/docs/deploy-manage/security) for the [YAML role mapping configuration file](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/mapping-users-groups-to-roles). Defaults to `ES_PATH_CONF/role_mapping.yml`.
  </definition>
  <definition term="follow_referrals">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether Elasticsearch should follow referrals returned by the LDAP server. Referrals are URLs returned by the server that are to be used to continue the LDAP operation (for example, search). Defaults to `true`.
  </definition>
  <definition term="metadata">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A list of additional LDAP attributes that should be loaded from the LDAP server and stored in the authenticated user’s metadata field.
  </definition>
  <definition term="timeout.tcp_connect">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The TCP connect timeout period for establishing an LDAP connection. An `s` at the end indicates seconds, or `ms` indicates milliseconds. Defaults to `5s` (5 seconds ).
  </definition>
  <definition term="timeout.tcp_read">
    <admonition title="Deprecated in 7.7">
      This setting was deprecated in 7.7.
    </admonition>
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The TCP read timeout period after establishing an LDAP connection. This is equivalent to and is deprecated in favor of `timeout.response` and they cannot be used simultaneously. An `s` at the end indicates seconds, or `ms` indicates milliseconds.
  </definition>
  <definition term="timeout.response">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The time interval to wait for the response from the LDAP server. An `s` at the end indicates seconds, or `ms` indicates milliseconds. Defaults to the value of `timeout.ldap_search`.
  </definition>
  <definition term="timeout.ldap_search">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The timeout period for an LDAP search. The value is specified in the request and is enforced by the receiving LDAP Server. An `s` at the end indicates seconds, or `ms` indicates milliseconds. Defaults to `5s` (5 seconds ).
  </definition>
  <definition term="ssl.key">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
    If the LDAP server requires client authentication, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="ssl.key_passphrase">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.secure_key_passphrase` instead.
    </admonition>
    You cannot use this setting and `ssl.secure_key_passphrase` at the same time.
  </definition>
  <definition term="ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
  </definition>
  <definition term="ssl.certificate">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
    This certificate is presented to clients when they connect.
  </definition>
  <definition term="ssl.certificate_authorities">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
    You cannot use this setting and `ssl.truststore.path` at the same time.
  </definition>
  <definition term="ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
    You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="ssl.keystore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.keystore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
  </definition>
  <definition term="ssl.keystore.secure_key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="ssl.truststore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
  <definition term="ssl.truststore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the truststore file. It must be either `jks` or `PKCS12`. If the file name ends in ".p12", ".pfx" or "pkcs12", the default is `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.verification_mode">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Indicates the type of verification when using `ldaps` to protect against man in the middle attacks and certificate forgery.
    Defaults to `full`.
    **Valid values**:
    - `full`: Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`: Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`: Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="ssl.supported_protocols">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. The default is `TLSv1.3,TLSv1.2`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="ssl.cipher_suites">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the cipher suites that should be supported when communicating with the LDAP server. Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
  <definition term="cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the time-to-live for cached user entries. A user and a hash of its credentials are cached for this period of time. Use the standard Elasticsearch [time units](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units). Defaults to  `20m`.
  </definition>
  <definition term="cache.max_users">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the maximum number of user entries that the cache can contain. Defaults to `100000`.
  </definition>
  <definition term="cache.hash_algo">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting), Expert) Specifies the hashing algorithm that is used for the in-memory cached user credentials. See [Table 1, Cache hash algorithms](#cache-hash-algo). Defaults to `ssha256`.
  </definition>
  <definition term="authentication.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) If set to `false`, disables authentication support in this realm, so that it only supports user lookups. (See the [run as](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/submitting-requests-on-behalf-of-other-users) and [authorization realms](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms) features). Defaults to `true`.
  </definition>
</definitions>


#### Active Directory realm settings

In addition to the [settings that are valid for all realms](#ref-realm-settings), you can specify the following settings:
<definitions>
  <definition term="url">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) One or more LDAP URLs in the `ldap[s]://<server>:<port>` format. Defaults to `ldap://<domain_name>:389`. This setting is required when connecting using SSL/TLS or when using a custom port.
    To provide multiple URLs, use a YAML array (`["ldap://server1:636", "ldap://server2:636"]`) or comma-separated string (`"ldap://server1:636, ldap://server2:636"`).
    While both are supported, you can’t mix the `ldap` and `ldaps` protocols.
    If no URL is provided, Elasticsearch uses a default of `ldap://<domain_name>:389`. This default uses the `domain_name` setting value and assumes an unencrypted connection to port 389.
  </definition>
  <definition term="load_balance.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The behavior to use when there are multiple LDAP URLs defined. For supported values see [load balancing and failover types](#load-balancing). Defaults to `failover`.
  </definition>
  <definition term="load_balance.cache_ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) When using `dns_failover` or `dns_round_robin` as the load balancing type, this setting controls the amount of time to cache DNS lookups. Defaults to `1h`.
  </definition>
  <definition term="domain_name">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The domain name of Active Directory. If the `url` and the `user_search.base_dn` settings are not specified, the cluster can derive those values from this setting. Required.
  </definition>
  <definition term="bind_dn">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The DN of the user that is used to bind to Active Directory and perform searches. Defaults to Empty. Due to its potential security impact, `bind_dn` is not exposed via the [nodes info API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-nodes-info).
  </definition>
  <definition term="bind_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the user that is used to bind to Active Directory. Defaults to Empty. Due to its potential security impact, `bind_password` is not exposed via the [nodes info API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-nodes-info).
    <admonition title="Deprecated in 6.3">
      Use `secure_bind_password` instead.
    </admonition>
  </definition>
  <definition term="secure_bind_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings), [Reloadable](https://www.elastic.co/docs/deploy-manage/security/secure-settings#reloadable-secure-settings)) The password for the user that is used to bind to Active Directory. Defaults to Empty.
  </definition>
  <definition term="unmapped_groups_as_roles">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) If set to `true`, the names of any unmapped Active Directory groups are used as role names and assigned to the user. A group is considered *unmapped* when it is not referenced in any role-mapping files. API-based role mappings are not considered. Defaults to `false`.
  </definition>
  <definition term="files.role_mapping">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The [location](https://www.elastic.co/docs/deploy-manage/security) for the YAML role mapping configuration file. Defaults to `ES_PATH_CONF/role_mapping.yml`.
  </definition>
  <definition term="user_search.base_dn">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The context to search for a user. Defaults to the root of the Active Directory domain.
  </definition>
  <definition term="user_search.scope">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether the user search should be `sub_tree`, `one_level` or `base`. `one_level` only searches users directly contained within the `base_dn`. `sub_tree` searches all objects contained under `base_dn`. `base` specifies that the `base_dn` is a user object, and that it is the only user considered. Defaults to `sub_tree`.
  </definition>
  <definition term="user_search.filter">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies a filter to use to lookup a user given a username. The default filter looks up `user` objects with either `sAMAccountName` or `userPrincipalName`. If specified, this must be a valid LDAP user search filter. For example `(&(objectClass=user)(sAMAccountName={{0}}))`. For more information, see [Search Filter Syntax](https://msdn.microsoft.com/en-us/library/aa746475(v=vs.85).aspx).
  </definition>
  <definition term="user_search.upn_filter">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies a filter to use to lookup a user given a user principal name. The default filter looks up `user` objects with a matching `userPrincipalName`. If specified, this must be a valid LDAP user search filter. For example, `(&(objectClass=user)(userPrincipalName={{1}}))`. `{{1}}` is the full user principal name provided by the user. For more information, see [Search Filter Syntax](https://msdn.microsoft.com/en-us/library/aa746475(v=vs.85).aspx).
  </definition>
  <definition term="user_search.down_level_filter">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies a filter to use to lookup a user given a down level logon name (DOMAIN\user). The default filter looks up `user` objects with a matching `sAMAccountName` in the domain provided. If specified, this must be a valid LDAP user search filter. For example, `(&(objectClass=user)(sAMAccountName={{0}}))`. For more information, see [Search Filter Syntax](https://msdn.microsoft.com/en-us/library/aa746475(v=vs.85).aspx).
  </definition>
  <definition term="user_search.pool.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Enables or disables connection pooling for user search. When disabled a new connection is created for every search. The default is `true` when `bind_dn` is provided.
  </definition>
  <definition term="user_search.pool.size">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum number of connections to the Active Directory server to allow in the connection pool. Defaults to `20`.
  </definition>
  <definition term="user_search.pool.initial_size">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The initial number of connections to create to the Active Directory server on startup. Defaults to `0`. If the LDAP server is down, values greater than 0 could cause startup failures.
  </definition>
  <definition term="user_search.pool.health_check.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Enables or disables a health check on Active Directory connections in the connection pool. Connections are checked in the background at the specified interval. Defaults to `true`.
  </definition>
  <definition term="user_search.pool.health_check.dn">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The distinguished name to be retrieved as part of the health check. Defaults to the value of `bind_dn` if that setting is present. Otherwise, it defaults to the value of the `user_search.base_dn` setting.
  </definition>
  <definition term="user_search.pool.health_check.interval">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The interval to perform background checks of connections in the pool. Defaults to `60s`.
  </definition>
  <definition term="group_search.base_dn">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The context to search for groups in which the user has membership. Defaults to the root of the Active Directory domain.
  </definition>
  <definition term="group_search.scope">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether the group search should be `sub_tree`, `one_level` or `base`.  `one_level` searches for groups directly contained within the `base_dn`. `sub_tree` searches all objects contained under `base_dn`. `base` specifies that the `base_dn` is a group object, and that it is the only group considered. Defaults to `sub_tree`.
  </definition>
  <definition term="metadata">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A list of additional LDAP attributes that should be loaded from the LDAP server and stored in the authenticated user’s metadata field.
  </definition>
  <definition term="timeout.tcp_connect">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The TCP connect timeout period for establishing an LDAP connection. An `s` at the end indicates seconds, or `ms` indicates milliseconds. Defaults to `5s` (5 seconds ).
  </definition>
  <definition term="timeout.tcp_read">
    <admonition title="Deprecated in 7.7">
      This is equivalent to and is deprecated in favor of `timeout.response` and they cannot be used simultaneously.
    </admonition>
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The TCP read timeout period after establishing an LDAP connection. An `s` at the end indicates seconds, or `ms` indicates milliseconds. Defaults to the value of `timeout.ldap_search`.
  </definition>
  <definition term="timeout.response">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The time interval to wait for the response from the AD server. An `s` at the end indicates seconds, or `ms` indicates milliseconds. Defaults to the value of `timeout.ldap_search`.
  </definition>
  <definition term="timeout.ldap_search">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The timeout period for an LDAP search. The value is specified in the request and is enforced by the receiving LDAP Server. An `s` at the end indicates seconds, or `ms` indicates milliseconds. Defaults to `5s` (5 seconds ).
  </definition>
  <definition term="ssl.certificate">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
    This certificate is presented to clients when they connect.
  </definition>
  <definition term="ssl.certificate_authorities">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
    You cannot use this setting and `ssl.truststore.path` at the same time.
  </definition>
  <definition term="ssl.key">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
    If the Active Directory server requires client authentication, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="ssl.key_passphrase">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.secure_key_passphrase` instead.
    </admonition>
    You cannot use this setting and `ssl.secure_key_passphrase` at the same time.
  </definition>
  <definition term="ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
  </definition>
  <definition term="ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
  </definition>
  <definition term="ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="ssl.keystore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="ssl.secure_keystore.password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
    You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="ssl.keystore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.truststore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
  <definition term="ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="ssl.truststore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the truststore file. It must be either `jks` or `PKCS12`. If the file name ends in ".p12", ".pfx" or "pkcs12", the default is `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.verification_mode">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Indicates the type of verification when using `ldaps` to protect against man in the middle attacks and certificate forgery.
    Defaults to `full`.
    **Valid values**:
    - `full`:   Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`:   Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`:   Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="ssl.supported_protocols">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. If the JVM’s SSL provider supports TLSv1.3, the default is `TLSv1.3,TLSv1.2,TLSv1.1`. Otherwise, the default is `TLSv1.2,TLSv1.1`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="ssl.cipher_suites">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the cipher suites that should be supported when communicating with the Active Directory server. Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
  <definition term="cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the time-to-live for cached user entries. A user and a hash of its credentials are cached for this configured period of time. Use the standard Elasticsearch [time units](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units)). Defaults to `20m`.
  </definition>
  <definition term="cache.max_users">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the maximum number of user entries that the cache can contain. Defaults to `100000`.
  </definition>
  <definition term="cache.hash_algo">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting), Expert) Specifies the hashing algorithm that is used for the in-memory cached user credentials. See [Table 1, Cache hash algorithms](#cache-hash-algo). Defaults to `ssha256`.
  </definition>
  <definition term="authentication.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) If set to `false`, disables authentication support in this realm, so that it only supports user lookups. (See the [run as](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/submitting-requests-on-behalf-of-other-users) and [authorization realms](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms) features). Defaults to `true`.
  </definition>
  <definition term="follow_referrals">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) If set to `true`, Elasticsearch follows referrals returned by the LDAP server. Referrals are URLs returned by the server that are to be used to continue the LDAP operation (such as `search`). Defaults to `true`.
  </definition>
</definitions>


#### PKI realm settings

In addition to the [settings that are valid for all realms](#ref-realm-settings), you can specify the following settings:
<definitions>
  <definition term="username_pattern">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The regular expression pattern used to extract the username from the certificate DN. The username is used for auditing and logging. The username can also be used with the [role mapping API](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/mapping-users-groups-to-roles) and [authorization delegation](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/authorization-delegation). The first match group is the used as the username. Defaults to `CN=(.*?)(?:,|$)`.
    This setting is ignored if either `username_rdn_oid` or `username_rdn_name` is set.
  </definition>
  <definition term="username_rdn_oid">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The relative distinguished name (RDN) attribute OID used to extract the username from the certificate DN. The username is used for auditing and logging. The username can also be used with the [role mapping API](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/mapping-users-groups-to-roles) and [authorization delegation](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/authorization-delegation). The value of the most specific RDN matching this attribute OID is used as the username.
    This setting takes precedent over `username_pattern`. You cannot use this setting and `username_rdn_name` at the same time.
  </definition>
  <definition term="username_rdn_name">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The relative distinguished name (RDN) attribute name used to extract the username from the certificate DN. The username is used for auditing and logging. The username can also be used with the [role mapping API](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/mapping-users-groups-to-roles) and [authorization delegation](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/authorization-delegation). The value of the most specific RDN matching this attribute name is used as the username.
    This setting takes precedent over `username_pattern`. You cannot use this setting and `username_rdn_oid` at the same time.
  </definition>
  <definition term="certificate_authorities">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to the PEM certificate files that should be used to authenticate a user’s certificate as trusted. Defaults to the trusted certificates configured for SSL. This setting cannot be used with `truststore.path`.
  </definition>
  <definition term="truststore.algorithm">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Algorithm for the truststore. Defaults to `SunX509`.
  </definition>
  <definition term="truststore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
</definitions>

If `truststore.path` is set, this setting is required.
<definitions>
  <definition term="truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
  <definition term="truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path of a truststore to use. Defaults to the trusted certificates configured for SSL. This setting cannot be used with `certificate_authorities`.
  </definition>
  <definition term="files.role_mapping">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the [location](https://www.elastic.co/docs/deploy-manage/security) of the [YAML role mapping configuration file](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/mapping-users-groups-to-roles). Defaults to `ES_PATH_CONF/role_mapping.yml`.
  </definition>
  <definition term="authorization_realms">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The names of the realms that should be consulted for delegated authorization. If this setting is used, then the PKI realm does not perform role mapping and instead loads the user from the listed realms. See [Delegating authorization to another realm](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms).
  </definition>
  <definition term="cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the time-to-live for cached user entries. A user and a hash of its credentials are cached for this period of time. Use the standard Elasticsearch [time units](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units)). Defaults to `20m`.
  </definition>
  <definition term="cache.max_users">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the maximum number of user entries that the cache can contain. Defaults to `100000`.
  </definition>
  <definition term="delegation.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Generally, in order for the clients to be authenticated by the PKI realm they must connect directly to Elasticsearch. That is, they must not pass through proxies which terminate the TLS connection. In order to allow for a **trusted** and **smart** proxy, such as Kibana, to sit before Elasticsearch and terminate TLS connections, but still allow clients to be authenticated on Elasticsearch by this realm, you need to toggle this to `true`. Defaults to `false`. If delegation is enabled, then either `truststore.path` or `certificate_authorities` setting must be defined. For more details, see [Configuring authentication delegation for PKI realms](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/pki#pki-realm-for-proxied-clients).
  </definition>
</definitions>


#### SAML realm settings

In addition to the [settings that are valid for all realms](#ref-realm-settings), you can specify the following settings.
<definitions>
  <definition term="idp.entity_id logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Entity ID of the SAML Identity Provider. An Entity ID is a URI with a maximum length of 1024 characters. It can be a URL ([https://idp.example.com/](https://idp.example.com/)) or a URN (`urn:example.com:idp`) and can be found in the configuration or the SAML metadata of the Identity Provider.
  </definition>
  <definition term="idp.metadata.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path *(recommended)* or URL to a SAML 2.0 metadata file describing the capabilities and configuration of the Identity Provider. If a path is provided, then it is resolved relative to the Elasticsearch config directory. If a URL is provided, then it must be either a `file` URL or a `https` URL.
    Elasticsearch automatically polls this metadata resource and reloads the IdP configuration when changes are detected. File based resources are polled at a frequency determined by the global Elasticsearch `resource.reload.interval.high` setting, which defaults to 5 seconds. HTTPS resources are polled at a frequency determined by the realm’s `idp.metadata.http.refresh` and `idp.metadata.http.minimum_refresh` settings.
    If the metadata resource is loaded from a file then the file must exist at node startup, if it does not exist then the node will fail to start. If the resource is loaded over HTTPS then (by default) the node will be tolerant of a failure to load the resource - the node will start and will continue to poll the URL for updates. The affected SAML realm will fail to authenticate users until the problem is resolved. To force the node to fail if the metadata is unavailable set `idp.metadata.http.fail_on_error` to `true`.
  </definition>
  <definition term="idp.metadata.http.fail_on_error">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) If set to `true`, the realm will fail on startup (and prevent the node from starting) if it attempts to load metadata over HTTPS and that metadata is not available. If set to `false` (the default), the node will start but the affected SAML realm will not support user authentication until the metadata can be successfully loaded. This setting is ignored if metadata is loaded from a file.
  </definition>
  <definition term="idp.metadata.http.connect_timeout logo cloud Elastic Stack: Generally available since 9.1">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the behavior of the HTTP client that is used for fetching the SAML 2.0 metadata file from a remote URL. Specifies the maximum time the client waits for a connection to be established. A value of zero means the timeout is not used. Defaults to `5s`.
  </definition>
  <definition term="idp.metadata.http.read_timeout logo cloud Elastic Stack: Generally available since 9.1">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the behavior of the HTTP client that is used for fetching the SAML 2.0 metadata file from a remote URL. Specifies the maximum time the client waits when requesting a connection from the connection manager. Defaults to `10s`.
  </definition>
  <definition term="idp.metadata.http.refresh logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the frequency with which `https` metadata is checked for changes. Defaults to `1h` (1 hour).
    Under some circumstances Elasticsearch may determine that the metadata needs to be checked more frequently. This may occur if previous attempts to load the metadata encountered an error, or if the metadata indicates that it is due to expire in less than the configured refresh interval. In these cases Elasticsearch will poll more often, but never more frequently than `idp.metadata.http.minimum_refresh`. If there is an attempt to authenticate against a realm that has not yet loaded metadata successfully, that realm may attempt to load metadata outside of the configured polling frequency.
  </definition>
  <definition term="idp.metadata.http.minimum_refresh">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the minimum frequency with which `https` metadata is checked for changes. In regular operation Elasticsearch will use the value of `idp.metadata.http.refresh` as the polling interval. However, under some circumstances Elasticsearch may determine that it needs to poll more frequently. In these cases, the `minimum_refresh` will set the minimum frequency at which the metdata will be checked. Defaults to `5m` (5 minutes) and must not be set to a value greater than `idp.metadata.http.refresh`
  </definition>
  <definition term="idp.use_single_logout logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Indicates whether to utilise the Identity Provider’s Single Logout service (if one exists in the IdP metadata file). Defaults to `true`.
  </definition>
  <definition term="sp.entity_id logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Entity ID to use for this SAML Service Provider. This should be entered as a URI. We recommend that you use the base URL of your Kibana instance. For example, `https://kibana.example.com/`.
  </definition>
  <definition term="sp.acs logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The URL of the Assertion Consumer Service within Kibana. Typically this is the "api/security/saml/callback" endpoint of your Kibana server. For example, `https://kibana.example.com/api/security/saml/callback`.
  </definition>
  <definition term="sp.logout logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The URL of the Single Logout service within Kibana. Typically this is the "logout" endpoint of your Kibana server. For example, `https://kibana.example.com/logout`.
  </definition>
  <definition term="attributes.principal logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Name of the SAML attribute that contains the user’s principal (username).
  </definition>
  <definition term="attributes.groups logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Name of the SAML attribute that contains the user’s groups.
  </definition>
  <definition term="attributes.name logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Name of the SAML attribute that contains the user’s full name.
  </definition>
  <definition term="attributes.mail logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Name of the SAML attribute that contains the user’s email address.
  </definition>
  <definition term="attributes.dn logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Name of the SAML attribute that contains the user’s X.50 *Distinguished Name*.
  </definition>
  <definition term="attribute_patterns.principal logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A Java regular expression that is matched against the SAML attribute specified by `attributes.principal` before it is applied to the user’s *principal* property. The attribute value must match the pattern and the value of the first *capturing group* is used as the principal. For example, `^([^@]+)@example\\.com$` matches email addresses from the "example.com" domain and uses the local-part as the principal.
  </definition>
  <definition term="attribute_patterns.groups logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) As per `attribute_patterns.principal`, but for the *group* property.
  </definition>
  <definition term="attribute_patterns.name logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) As per `attribute_patterns.principal`, but for the *name* property.
  </definition>
  <definition term="attribute_patterns.mail logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) As per `attribute_patterns.principal`, but for the *mail* property.
  </definition>
  <definition term="attribute_patterns.dn logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) As per `attribute_patterns.principal`, but for the *dn* property.
  </definition>
  <definition term="attribute_delimiters.groups logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A plain string that is used as a delimiter to split a single-valued SAML attribute specified by `attributes.groups` before it is applied to the user’s *groups* property. For example, splitting the SAML attribute value `engineering,elasticsearch-admins,employees` on a delimiter value of `,` will result in `engineering`, `elasticsearch-admins`, and `employees` as the list of groups for the user. The delimiter will always be split on, regardless of escaping in the input string. This setting does not support multi-valued SAML attributes. It cannot be used together with the `attribute_patterns` setting. You can only configure this setting for the groups attribute.
  </definition>
  <definition term="nameid_format logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The NameID format that should be requested when asking the IdP to authenticate the current user. The default is to not include the `nameid_format` attribute.
  </definition>
  <definition term="nameid.allow_create logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The value of the `AllowCreate` attribute of the `NameIdPolicy` element in an authentication request. The default value is false.
  </definition>
  <definition term="nameid.sp_qualifier logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The value of the `SPNameQualifier` attribute of the `NameIdPolicy` element in an authentication request. The default is to not include the `SPNameQualifier` attribute.
  </definition>
  <definition term="force_authn logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether to set the `ForceAuthn` attribute when requesting that the IdP authenticate the current user. If set to `true`, the IdP is required to verify the user’s identity, irrespective of any existing sessions they might have. Defaults to `false`.
  </definition>
  <definition term="populate_user_metadata logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether to populate the Elasticsearch user’s metadata with the values that are provided by the SAML attributes. Defaults to `true`.
  </definition>
  <definition term="authorization_realms">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The names of the realms that should be consulted for delegated authorization. If this setting is used, then the SAML realm does not perform role mapping and instead loads the user from the listed realms. See [Delegating authorization to another realm](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms).
  </definition>
  <definition term="allowed_clock_skew logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum amount of skew that can be tolerated between the IdP’s clock and the Elasticsearch node’s clock. Defaults to `3m` (3 minutes).
  </definition>
  <definition term="req_authn_context_class_ref logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A comma separated list of Authentication Context Class Reference values to be included in the Requested Authentication Context when requesting the IdP to authenticate the current user. The Authentication Context of the corresponding authentication response should contain at least one of the requested values.
    For more information, see [Requesting specific authentication methods](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/saml#req-authn-context).
  </definition>
</definitions>


#### SAML realm signing settings

If a signing key is configured (that is, either `signing.key` or `signing.keystore.path` is set), then Elasticsearch signs outgoing SAML messages. Signing can be configured using the following settings:
<definitions>
  <definition term="signing.saml_messages logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A list of SAML message types that should be signed or `*` to sign all messages. Each element in the list should be the local name of a SAML XML Element. Supported element types are `AuthnRequest`, `LogoutRequest` and `LogoutResponse`. Only valid if `signing.key` or `signing.keystore.path` is also specified. Defaults to `*`.
  </definition>
  <definition term="signing.key logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path to the PEM encoded private key to use for SAML message signing. `signing.key` and `signing.keystore.path` cannot be used at the same time.
  </definition>
  <definition term="signing.secure_key_passphrase logo cloud">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Specifies the passphrase to decrypt the PEM encoded private key (`signing.key`) if it is encrypted.
  </definition>
  <definition term="signing.certificate logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path to the PEM encoded certificate (or certificate chain) that corresponds to the `signing.key`. This certificate must also be included in the Service Provider metadata or manually configured within the IdP to allow for signature validation. This setting can only be used if `signing.key` is set.
  </definition>
  <definition term="signing.keystore.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path to the keystore that contains a private key and certificate. It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `signing.key` at the same time.
  </definition>
  <definition term="signing.keystore.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The type of the keystore in `signing.keystore.path`. Must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or "pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="signing.keystore.alias logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the alias of the key within the keystore that should be used for SAML message signing. If the keystore contains more than one private key, this setting must be specified.
  </definition>
  <definition term="signing.keystore.secure_password logo cloud">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password to the keystore in `signing.keystore.path`.
  </definition>
  <definition term="signing.keystore.secure_key_password logo cloud">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore (`signing.keystore.path`). Defaults to the keystore password.
  </definition>
</definitions>


#### SAML realm encryption settings

If an encryption key is configured (that is, either `encryption.key` or `encryption.keystore.path` is set), then Elasticsearch publishes an encryption certificate when generating metadata and attempts to decrypt incoming SAML content. Encryption can be configured using the following settings:
<definitions>
  <definition term="encryption.key logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path to the PEM encoded private key to use for SAML message decryption. `encryption.key` and `encryption.keystore.path` cannot be used at the same time.
  </definition>
  <definition term="encryption.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Specifies the passphrase to decrypt the PEM encoded private key (`encryption.key`) if it is encrypted.
  </definition>
  <definition term="encryption.certificate logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path to the PEM encoded certificate (or certificate chain) that is associated with the `encryption.key`. This certificate must also be included in the Service Provider metadata or manually configured within the IdP to enable message encryption. This setting can be used only if `encryption.key` is set.
  </definition>
  <definition term="encryption.keystore.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path to the keystore that contains a private key and certificate. It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `encryption.key` at the same time.
  </definition>
  <definition term="encryption.keystore.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The type of the keystore (`encryption.keystore.path`). Must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or "pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="encryption.keystore.alias logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the alias of the key within the keystore (`encryption.keystore.path`) that should be used for SAML message decryption. If not specified, all compatible key pairs from the keystore are considered as candidate keys for decryption.
  </definition>
  <definition term="encryption.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password to the keystore (`encryption.keystore.path`).
  </definition>
  <definition term="encryption.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore (`encryption.keystore.path`). Only a single password is supported. If you are using multiple decryption keys, they cannot have individual passwords.
  </definition>
</definitions>


#### SAML realm SSL settings

If you are loading the IdP metadata over SSL/TLS (that is, `idp.metadata.path` is a URL using the `https` protocol), the following settings can be used to configure SSL.
<note>
  These settings are not used for any purpose other than loading metadata over https.
</note>

<definitions>
  <definition term="ssl.key logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="ssl.key_passphrase logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.secure_key_passphrase` instead.
    </admonition>
    You cannot use this setting and `ssl.secure_key_passphrase` at the same time.
  </definition>
  <definition term="ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    You cannot use this setting and `ssl.key_passphrase` at the same time.
  </definition>
  <definition term="ssl.certificate logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
  </definition>
  <definition term="ssl.certificate_authorities logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
  </definition>
  <definition term="ssl.keystore.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="ssl.keystore.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.keystore.password logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
    You cannot use this setting and `ssl.keystore.password` at the same time.
  </definition>
  <definition term="ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
    You cannot use this setting and `ssl.keystore.secure_key_password` at the same time.
  </definition>
  <definition term="ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
</definitions>

You cannot use this setting and `ssl.keystore.key_password` at the same time.
<definitions>
  <definition term="ssl.truststore.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="ssl.truststore.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the truststore file. It must be either `jks` or `PKCS12`. If the file name ends in ".p12", ".pfx" or "pkcs12", the default is `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.truststore.password logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
    This setting cannot be used with `ssl.truststore.password`.
  </definition>
  <definition term="ssl.verification_mode logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the verification of certificates.
    Defaults to `full`.
    **Valid values**:
    - `full`:   Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`:   Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`:   Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="ssl.supported_protocols logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. If the JVM’s SSL provider supports TLSv1.3, the default is `TLSv1.3,TLSv1.2,TLSv1.1`. Otherwise, the default is `TLSv1.2,TLSv1.1`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="ssl.cipher_suites logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
</definitions>


#### Kerberos realm settings

In addition to the [settings that are valid for all realms](#ref-realm-settings), you can specify the following settings:
<definitions>
  <definition term="keytab.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path to the Kerberos keytab file that contains the service principal used by this Elasticsearch node. This must be a location within the Elasticsearch configuration directory and the file must have read permissions. Required.
  </definition>
  <definition term="remove_realm_name logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set to `true` to remove the realm part of principal names. Principal names in Kerberos have the form `user/instance@REALM`. If this option is `true`, the realm part (`@REALM`) will not be included in the username. Defaults to `false`.
  </definition>
  <definition term="krb.debug">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set to `true` to enable debug logs for the Java login module that provides support for Kerberos authentication. Defaults to `false`.
  </definition>
  <definition term="cache.ttl logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The time-to-live for cached user entries. A user is cached for this period of time. Specify the time period using the standard Elasticsearch [time units](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units). Defaults to `20m`.
  </definition>
  <definition term="cache.max_users logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum number of user entries that can live in the cache at any given time. Defaults to 100,000.
  </definition>
  <definition term="authorization_realms logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The names of the realms that should be consulted for delegated authorization. If this setting is used, then the Kerberos realm does not perform role mapping and instead loads the user from the listed realms. See [Delegating authorization to another realm](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms).
  </definition>
</definitions>


#### OpenID Connect realm settings

In addition to the [settings that are valid for all realms](#ref-realm-settings), you can specify the following settings.
<definitions>
  <definition term="op.issuer logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A verifiable Identifier for your OpenID Connect Provider. An Issuer Identifier is usually a case sensitive URL using the https scheme that contains scheme, host, and optionally, port number and path components and no query or fragment components. The value for this setting should be provided by your OpenID Connect Provider.
  </definition>
  <definition term="op.authorization_endpoint logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The URL for the Authorization Endpoint at the OpenID Connect Provider. The value for this setting should be provided by your OpenID Connect Provider.
  </definition>
  <definition term="op.token_endpoint logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The URL for the Token Endpoint at the OpenID Connect Provider. The value for this setting should be provided by your OpenID Connect Provider.
  </definition>
  <definition term="op.userinfo_endpoint logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The URL for the User Info Endpoint at the OpenID Connect Provider. The value for this setting should be provided by your OpenID Connect Provider.
  </definition>
  <definition term="op.endsession_endpoint logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The URL for the End Session Endpoint at the OpenID Connect Provider. The value for this setting should be provided by your OpenID Connect Provider.
  </definition>
  <definition term="op.jwkset_path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting))
  </definition>
</definitions>

The file name or URL to a JSON Web Key Set (JWKS) with the public key material used to verify tokens and claims responses signed by the OpenID Connect Provider. A value is considered a file name if it does not begin with `https` or `http`. The file name is resolved relative to the Elasticsearch configuration directory.  Changes to the file are polled at a frequency determined by the global Elasticsearch `resource.reload.interval.high` setting, which defaults to 5 seconds.
- If a URL is provided, then it must begin with `https://` or `http://`. Elasticsearch automatically caches the retrieved JWK and will attempt to refresh the JWK upon signature verification failure, as this might indicate that the OpenID Connect Provider has rotated the signing keys.

<definitions>
  <definition term="authorization_realms">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The names of the realms that should be consulted for delegated authorization. If this setting is used, then the OpenID Connect realm does not perform role mapping and instead loads the user from the listed realms. See [Delegating authorization to another realm](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms).
  </definition>
  <definition term="rp.client_id logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The OAuth 2.0 Client Identifier that was assigned to Elasticsearch during registration at the OpenID Connect Provider.
  </definition>
  <definition term="rp.client_secret">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The OAuth 2.0 Client Secret that was assigned to Elasticsearch during registration at the OpenID Connect Provider.
  </definition>
  <definition term="rp.client_auth_method logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The client authentication method used by Elasticsearch to authenticate to the OpenID Connect Provider. Can be `client_secret_basic`, `client_secret_post`, or `client_secret_jwt`. Defaults to `client_secret_basic`.
  </definition>
  <definition term="rp.client_auth_jwt_signature_algorithm logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The signature algorithm that Elasticsearch uses to sign the JWT with which it authenticates as a client to the OpenID Connect Provider when `client_secret_jwt` is selected for `rp.client_auth_method`. Can be either `HS256`, `HS384`, or `HS512`. Defaults to `HS384`.
  </definition>
  <definition term="rp.redirect_uri logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Redirect URI within Kibana. If you want to use the authorization code flow, this is the `api/security/oidc/callback` endpoint of your Kibana server. If you want to use the implicit flow,  it is the `api/security/oidc/implicit` endpoint. For example, `https://kibana.example.com/api/security/oidc/callback`.
  </definition>
  <definition term="rp.response_type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) OAuth 2.0 Response Type value that determines the authorization processing flow to be used. Can be `code` for authorization code grant flow, or one of `id_token`, `id_token token` for the implicit flow.
  </definition>
  <definition term="rp.signature_algorithm logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The signature algorithm that will be used by Elasticsearch in order to verify the signature of the id tokens it will receive from the OpenID Connect Provider. Allowed values are `HS256`, `HS384`, `HS512`, `ES256`, `ES384`, `ES512`, `RS256`, `RS384`, `RS512`, `PS256`, `PS384`, `PS512`. Defaults to `RS256`.
  </definition>
  <definition term="rp.requested_scopes logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The scope values that will be requested by the OpenID Connect Provider as part of the Authentication Request. Optional, defaults to `openid`
  </definition>
  <definition term="rp.post_logout_redirect_uri logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The Redirect URI (usually within Kibana) that the OpenID Connect Provider should redirect the browser to after a successful Single Logout.
  </definition>
  <definition term="claims.principal">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the OpenID Connect claim that contains the user’s principal (username).
  </definition>
  <definition term="claims.groups logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the OpenID Connect claim that contains the user’s groups.
  </definition>
  <definition term="claims.name logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the OpenID Connect claim that contains the user’s full name.
  </definition>
  <definition term="claims.mail logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the OpenID Connect claim that contains the user’s email address.
  </definition>
  <definition term="claims.dn logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the OpenID Connect claim that contains the user’s X.509 *Distinguished Name*.
  </definition>
  <definition term="claim_patterns.principal logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A Java regular expression that is matched against the OpenID Connect claim specified by `claims.principal` before it is applied to the user’s *principal* property. The attribute value must match the pattern and the value of the first *capturing group* is used as the principal. For example, `^([^@]+)@example\\.com$` matches email addresses from the "example.com" domain and uses the local-part as the principal.
  </definition>
  <definition term="claim_patterns.groups logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) As per `claim_patterns.principal`, but for the *group* property.
  </definition>
  <definition term="claim_patterns.name logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) As per `claim_patterns.principal`, but for the *name* property.
  </definition>
  <definition term="claim_patterns.mail logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) As per `claim_patterns.principal`, but for the *mail* property.
  </definition>
  <definition term="claim_patterns.dn logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) As per `claim_patterns.principal`, but for the *dn* property.
  </definition>
  <definition term="allowed_clock_skew logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum allowed clock skew to be taken into consideration when validating id tokens with regards to their creation and expiration times. Defaults to `60s`.
  </definition>
  <definition term="populate_user_metadata logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether to populate the Elasticsearch user’s metadata with the values that are provided by the OpenID Connect claims. Defaults to `true`.
  </definition>
  <definition term="http.proxy.host">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the address of the proxy server that will be used by the internal http client for all back-channel communication to the OpenID Connect Provider endpoints. This includes requests to the Token Endpoint, the Userinfo Endpoint and requests to fetch the JSON Web Key Set from the OP if `op.jwkset_path` is set as a URL.
  </definition>
  <definition term="http.proxy.scheme">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the protocol to use to connect to the proxy server that will be used by the http client for all back-channel communication to the OpenID Connect Provider endpoints. Defaults to `http`. Allowed values are `http` or `https`.
  </definition>
  <definition term="http.proxy.port">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the port of the proxy server that will be used by the http client for all backchannel communication to the OpenID Connect Provider endpoints. Defaults to `80`.
  </definition>
  <definition term="http.connect_timeout logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the behavior of the http client used for back-channel communication to the OpenID Connect Provider endpoints. Specifies the timeout until a connection is established. A value of zero means the timeout is not used. Defaults to `5s`.
  </definition>
  <definition term="http.connection_read_timeout logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the behavior of the http client used for back-channel communication to the OpenID Connect Provider endpoints. Specifies the timeout used when requesting a connection from the connection manager. Defaults to `5s`
  </definition>
  <definition term="http.socket_timeout logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the behavior of the http client used for back-channel communication to the OpenID Connect Provider endpoints. Specifies the socket timeout (SO_TIMEOUT) in milliseconds, which is the timeout for waiting for data or, put differently, a maximum period inactivity between two consecutive data packets). Defaults to `5s`.
  </definition>
  <definition term="http.max_connections logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the behavior of the http client used for back-channel communication to the OpenID Connect Provider endpoints. Specifies the maximum number of connections allowed across all endpoints. Defaults to `200`.
  </definition>
  <definition term="http.max_endpoint_connections logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the behavior of the http client used for back-channel communication to the OpenID Connect Provider endpoints. Specifies the maximum number of connections allowed per endpoint. Defaults to `200`.
  </definition>
  <definition term="http.tcp.keep_alive logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Whether to enable TCP keepalives on HTTP connections used for back-channel communication to the OpenID Connect Provider endpoints. Defaults to `true`.
  </definition>
  <definition term="http.connection_pool_ttl logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the behavior of the http client used for back-channel communication to the OpenID Connect Provider endpoints. Specifies the time-to-live of connections in the connection pool (default to 3 minutes). A connection is closed if it is idle for more than the specified timeout.
  </definition>
</definitions>

The server can also set the `Keep-Alive` HTTP response header. The effective time-to-live value is the smaller value between this setting and the `Keep-Alive` response header. Configure this setting to `-1` to let the server dictate the value. If the header is not set by the server and the setting has value of `-1`, the time-to-live is infinite and connections never expire.

#### OpenID Connect realm SSL settings

The following settings can be used to configure SSL for all outgoing http connections to the OpenID Connect Provider endpoints.
<note>
  These settings are *only* used for the back-channel communication between Elasticsearch and the OpenID Connect Provider
</note>

<definitions>
  <definition term="ssl.key logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="ssl.key_passphrase logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.secure_key_passphrase` instead.
    </admonition>
    You cannot use this setting and `ssl.secure_key_passphrase` at the same time.
  </definition>
  <definition term="ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    You cannot use this setting and `ssl.key_passphrase` at the same time.
  </definition>
  <definition term="ssl.certificate logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
  </definition>
  <definition term="ssl.certificate_authorities logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
  </definition>
  <definition term="ssl.keystore.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="ssl.keystore.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.keystore.password logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
    You cannot use this setting and `ssl.keystore.password` at the same time.
  </definition>
  <definition term="ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
    You cannot use this setting and `ssl.keystore.secure_key_password` at the same time.
  </definition>
  <definition term="ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
    You cannot use this setting and `ssl.keystore.key_password` at the same time.
  </definition>
  <definition term="ssl.truststore.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="ssl.truststore.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the truststore file. It must be either `jks` or `PKCS12`. If the file name ends in ".p12", ".pfx" or "pkcs12", the default is `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.truststore.password logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
    You cannot use this setting and `ssl.truststore.password` at the same time.
  </definition>
  <definition term="ssl.verification_mode logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the verification of certificates.
    Defaults to `full`.
    **Valid values**:
    - `full`:   Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`:   Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`:   Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="ssl.supported_protocols logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. If the JVM’s SSL provider supports TLSv1.3, the default is `TLSv1.3,TLSv1.2,TLSv1.1`. Otherwise, the default is `TLSv1.2,TLSv1.1`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="ssl.cipher_suites logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
</definitions>


#### JWT realm settings

In addition to the [settings that are valid for all realms](#ref-realm-settings), you can specify the following settings.
<definitions>
  <definition term="token_type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The token type, `id_token` or `access_token`, that the JWT realm uses to verify incoming JWTs. Defaults to `id_token`.
  </definition>
  <definition term="allowed_audiences logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A list of allowed JWT audiences that Elasticsearch should verify. Elasticsearch will only consume JWTs that were intended for any of these audiences, as denoted by the `aud` claim in the JWT). The audiences are compared with exact string matches and do not support wildcards or regex. Examples of `aud` claim are `<example-url>/client1` and `other_service,elasticsearch`. When `token_type` is `access_token`, the audiences can be optionally denoted by a different claim in the JWT if `aud` does not exist. See also [`fallback_claims.aud`](#security-settings-jwt-fallback-claims-aud).
  </definition>
  <definition term="allowed_clock_skew logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The maximum allowed clock skew to be taken into consideration when validating JWTs with regards to their creation, not before, and expiration times.
  </definition>
  <definition term="allowed_issuer logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A verifiable Identifier for your JWT Issuer. An Issuer Identifier is usually a case sensitive URL using the https scheme that contains scheme, host, and optionally, port number and path components and no query or fragment components. However, it can be any string. The value for this setting should be provided by your JWT Issuer. The issuer is compared with exact string matches and do not support wildcards or regex. Examples of `iss` claim are `<example-url>:8443/jwt` and `issuer123`.
  </definition>
  <definition term="allowed_subjects logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A list of allowed JWT subjects that Elasticsearch should verify. Elasticsearch will only consume JWTs that were issued for any of these subjects, as denoted by the `sub` claim in the JWT. The subjects are compared with exact string matches and do not support wildcards or regex. Examples of `sub` claim are `<example-url>/user1` and `user_1,user2`. When `token_type` is `access_token`, this setting is mandatory and the subject can be optionally denoted by a different claim in the JWT if `sub` does not exist. See also [`fallback_claims.sub`](#security-settings-jwt-fallback-claims-sub).
  </definition>
</definitions>


<definitions>
  <definition term="fallback_claims.sub logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The alternative claim to look for the subject information if the `sub` claim does not exist. It is configurable only when the `token_type` is `access_token`. The fallback is applied everywhere the `sub` claim is used.
  </definition>
</definitions>


<definitions>
  <definition term="fallback_claims.aud logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The alternative claim to look for the audiences information if the `aud` claim does not exist. It is configurable only when the `token_type` is `access_token`. The fallback is applied everywhere the `aud` claim is used.
  </definition>
  <definition term="required_claims logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Additional claims and associated values that Elasticsearch should verify. This is a group setting that takes key/value pairs, where the key is a string and the value must be either a string or an array of strings. The values are compared with exact string matches and do not support wildcards or regex.
  </definition>
</definitions>

For example:
```yaml
xpack.security.authc.realms.jwt.jwt1:
  required_claims:
    token_use: "id"
    versions: ["1.0", "2.0"]
```

<definitions>
  <definition term="allowed_signature_algorithms logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) A list of signature algorithms that will be used by Elasticsearch in order to verify the signature of the JWT it will receive from the JWT Issuer. Defaults to `RS256`. Examples are `HS512,RS512,ES512` and `ES384`. Allowed values are `HS256`, `HS384`, `HS512`, `ES256`, `ES384`, `ES512`, `RS256`, `RS384`, `RS512`, `PS256`, `PS384`, `PS512`.
  </definition>
  <definition term="authorization_realms">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The names of the realms that should be consulted for delegated authorization. If this setting is used, then the JWT realm does not perform role mapping and instead loads the user from the listed realms. See [Delegating authorization to another realm](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/realm-chains#authorization_realms).
  </definition>
  <definition term="claims.dn logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the JWT claim that contains the user’s Distinguished Name (DN), which uniquely identifies a user or group.
  </definition>
  <definition term="claim_patterns.dn logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Accepts the same Java regular expression as [`claim_patterns.principal`](#jwt-claim-pattern-principal), but for the `dn` property.
  </definition>
  <definition term="claims.groups logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the JWT claim that contains the user’s groups, such as `groups` and `roles`.
  </definition>
  <definition term="claim_patterns.group logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Accepts the same Java regular expression as [`claim_patterns.principal`](#jwt-claim-pattern-principal), but for the `group` property.
  </definition>
  <definition term="claims.mail logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the JWT claim that contains the user’s e-mail address.
  </definition>
  <definition term="claim_patterns.mail logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Accepts the same Java regular expression as [`claim_patterns.principal`](#jwt-claim-pattern-principal), but for the `mail` property.
  </definition>
  <definition term="claims.name logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the JWT claim that contains the user’s username.
  </definition>
  <definition term="claim_patterns.name logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Accepts the same Java regular expression as [`claim_patterns.principal`](#jwt-claim-pattern-principal), but for the `name` property.
  </definition>
  <definition term="claims.principal">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The name of the JWT claim that contains the user’s principal (username), such as `sub`, `name`, `email`, and `dn`.
  </definition>
</definitions>


<definitions>
  <definition term="claim_patterns.principal logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) An optional Java regular expression that is matched against the JWT claim specified by `claims.principal` before it is applied to the user’s `principal` property. The value must match the pattern and the value of the first *capturing group* is used as the principal. For example, `^([^@]+)@example\\.com$` matches email addresses from the `example.com` domain and uses the local-part as the principal. Another example is `sub` which may not need a pattern setting.
  </definition>
  <definition term="client_authentication.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether to use `shared_secret` or `none` to authenticate incoming client requests. If this value is `shared_secret`, the client is authenticated using an HTTP request header that must match a pre-configured secret value. The client must provide this shared secret with every request in the `ES-Client-Authentication` header. If this value is `none`, then the request header `ES-Client-Authentication` is ignored. Defaults to `shared_secret`.
    Enabling client authentication is recommended. If JWT bearer tokens are shared with other clients or services, client authentication restricts which ones are allowed to submit those JWTs to Elasticsearch.
  </definition>
  <definition term="client_authentication.shared_secret logo cloud">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings), [reloadable](https://www.elastic.co/docs/deploy-manage/security/secure-settings#reloadable-secure-settings)) Secret value string for client authentication. Required if `client_authentication.type` is `shared_secret`.
  </definition>
  <definition term="client_authentication.rotation_grace_period">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Sets the grace period for how long after rotating the `client_authentication.shared_secret` is valid. `client_authentication.shared_secret` can be rotated by updating the keystore then calling the [reload API](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-nodes-reload-secure-settings). Defaults to `1m`.
  </definition>
  <definition term="http.proxy.host">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the address of the proxy server for the HTTP client that is used for fetching the JSON Web Key Set from a remote URL.
  </definition>
  <definition term="http.proxy.scheme">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the protocol to use to connect to the proxy server for the HTTP client that is used for fetching the JSON Web Key Set from a remote URL. Must be `http`.
  </definition>
  <definition term="http.proxy.port">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the port of the proxy server for the HTTP client that is used for fetching the JSON Web Key Set from a remote URL. Defaults to `80`.
  </definition>
  <definition term="http.connect_timeout logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Sets the timeout for the HTTP client that is used for fetching the JSON Web Key Set from a remote URL. A value of zero means the timeout is not used. Defaults to `5s`.
  </definition>
  <definition term="http.connection_read_timeout logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the HTTP timeout used when requesting a connection from the connection manager. Defaults to `5s`.
  </definition>
  <definition term="http.socket_timeout logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the maximum socket timeout (SO_TIMEOUT) for the HTTP client to wait for inactivity between two consecutive data packets. Defaults to `5s`.
  </definition>
  <definition term="http.max_connections logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the maximum number of connections allowed across all endpoints.
  </definition>
  <definition term="http.max_endpoint_connections logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the maximum number of connections allowed per endpoint.
  </definition>
  <definition term="jwt.cache.size">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the maximum number of JWT cache entries. If clients use a different JWT for every request, set to `0` to disable the JWT cache. Defaults to `100000`.
  </definition>
  <definition term="jwt.cache.ttl">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the time-to-live for the period of time to cache JWT entries. JWTs can only be cached if client authentication is successful (or disabled). Uses the standard Elasticsearch [time units](/docs/reference/elasticsearch/rest-apis/api-conventions#time-units). If clients use a different JWT for every request, set to `0` to disable the JWT cache. Defaults to `20m`.
  </definition>
  <definition term="pkc_jwkset_path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The file name or URL to a JSON Web Key Set (JWKS) with the public key material that the JWT Realm uses for verifying token signatures. A value is considered a file name if it does not begin with `https`. The file name is resolved relative to the Elasticsearch configuration directory. If a URL is provided, then it must begin with `https://` (`http://` is not supported). Elasticsearch automatically caches the JWK set and will attempt to refresh the JWK set upon signature verification failure, as this might indicate that the JWT Provider has rotated the signing keys. Background JWKS reloading can also be configured with the setting `pkc_jwkset_reload.enabled`. This ensures that rotated keys are automatically discovered and used to verify JWT signatures.
  </definition>
  <definition term="pkc_jwkset_reload.enabled Elastic Stack: Generally available since 9.3 logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Indicates whether JWKS background reloading is enabled. Defaults to `false`.
  </definition>
  <definition term="pkc_jwkset_reload.file_interval Elastic Stack: Generally available since 9.3 logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the reload interval for file-based JWKS. Defaults to `5m`.
  </definition>
  <definition term="pkc_jwkset_reload.url_interval_min Elastic Stack: Generally available since 9.3 logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the minimum reload interval for URL-based JWKS. The `Expires` and `Cache-Control` HTTP response headers inform the reload interval. This configuration setting is the lower bound of what is considered, and it is also the default interval in the absence of useful response headers. Defaults to `1h`.
  </definition>
  <definition term="pkc_jwkset_reload.url_interval_max Elastic Stack: Generally available since 9.3 logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the maximum reload interval for URL-based JWKS. This configuration setting is the upper bound of what is considered from header responses (`5d`).
  </definition>
  <definition term="hmac_jwkset logo cloud">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Contents of a JSON Web Key Set (JWKS), including the secret key that the JWT realm uses to verify token signatures. This format supports multiple keys and optional attributes, and is preferred over the `hmac_key` setting. Cannot be used in conjunction with the `hmac_key` setting. Refer to [Configure Elasticsearch to use a JWT realm](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/jwt).
  </definition>
  <definition term="hmac_key logo cloud">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Contents of a single JSON Web Key (JWK), including the secret key that the JWT realm uses to verify token signatures. This format only supports a single key without attributes, and cannot be used with the `hmac_jwkset` setting. This format is compatible with OIDC. The HMAC key must be a UNICODE string, where the key bytes are the UTF-8 encoding of the UNICODE string. The `hmac_jwkset` setting is preferred. Refer to [Configure Elasticsearch to use a JWT realm](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/jwt).
  </definition>
  <definition term="populate_user_metadata logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies whether to populate the Elasticsearch user’s metadata with the values that are provided by the JWT claims. Defaults to `true`.
  </definition>
</definitions>


#### JWT realm SSL settings

The following settings can be used to configure SSL for fetching the JSON Web Key Set from a remote URL.
<note>
  These settings are *only* used for the back-channel communication between Elasticsearch and the JWT Issuer.
</note>

<definitions>
  <definition term="ssl.key logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="ssl.key_passphrase logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.secure_key_passphrase` instead.
    </admonition>
    You cannot use this setting and `ssl.secure_key_passphrase` at the same time.
  </definition>
  <definition term="ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    You cannot use this setting and `ssl.key_passphrase` at the same time.
  </definition>
  <definition term="ssl.certificate logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
  </definition>
  <definition term="ssl.certificate_authorities logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
  </definition>
  <definition term="ssl.keystore.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="ssl.keystore.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.keystore.password logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
    You cannot use this setting and `ssl.keystore.password` at the same time.
  </definition>
  <definition term="ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
    You cannot use this setting and `ssl.keystore.secure_key_password` at the same time.
  </definition>
  <definition term="ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
    You cannot use this setting and `ssl.keystore.key_password` at the same time.
  </definition>
  <definition term="ssl.truststore.path logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="ssl.truststore.type logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the truststore file. It must be either `jks` or `PKCS12`. If the file name ends in ".p12", ".pfx" or "pkcs12", the default is `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="ssl.truststore.password logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
    You cannot use this setting and `ssl.truststore.password` at the same time.
  </definition>
  <definition term="ssl.verification_mode logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the verification of certificates.
    Defaults to `full`.
    **Valid values**:
    - `full`:   Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`:   Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`:   Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="ssl.supported_protocols logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. If the JVM’s SSL provider supports TLSv1.3, the default is `TLSv1.3,TLSv1.2,TLSv1.1`. Otherwise, the default is `TLSv1.2,TLSv1.1`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="ssl.cipher_suites logo cloud">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
</definitions>


#### Load balancing and failover

The [Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting) `load_balance.type` setting can have the following values:
- `failover`: The URLs specified are used in the order that they are specified. The first server that can be connected to will be used for all subsequent connections. If a connection to that server fails then the next server that a connection can be established to will be used for subsequent connections.
- `dns_failover`: In this mode of operation, only a single URL may be specified. This URL must contain a DNS name. The system will be queried for all IP addresses that correspond to this DNS name. Connections to the Active Directory or LDAP server will always be tried in the order in which they were retrieved. This differs from `failover` in that there is no reordering of the list and if a server has failed at the beginning of the list, it will still be tried for each subsequent connection.
- `round_robin`: Connections will continuously iterate through the list of provided URLs. If a server is unavailable, iterating through the list of URLs will continue until a successful connection is made.
- `dns_round_robin`: In this mode of operation, only a single URL may be specified. This URL must contain a DNS name. The system will be queried for all IP addresses that correspond to this DNS name. Connections will continuously iterate through the list of addresses. If a server is unavailable, iterating through the list of URLs will continue until a successful connection is made.


### General TLS settings

<definitions>
  <definition term="xpack.security.ssl.diagnose.trust">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls whether to output diagnostic messages for SSL/TLS trust failures. If this is `true` (the default), a message will be printed to the Elasticsearch log whenever an SSL connection (incoming or outgoing) is rejected due to a failure to establish trust. This diagnostic message contains information that can be used to determine the cause of the failure and assist with resolving the problem. Set to `false` to disable these messages.
  </definition>
</definitions>


#### TLS/SSL key and trusted certificate settings

The following settings are used to specify a private key, certificate, and the trusted certificates that should be used when communicating over an SSL/TLS connection. If no trusted certificates are configured, the default certificates that are trusted by the JVM will be trusted along with the certificate(s) associated with a key in the same context. The key and certificate must be in place for connections that require client authentication or when acting as a SSL enabled server.
<note>
  Storing trusted certificates in a PKCS#12 file, although supported, is uncommon in practice. The [`elasticsearch-certutil` tool](https://www.elastic.co/docs/reference/elasticsearch/command-line-tools/certutil), as well as Java’s `keytool`, are designed to generate PKCS#12 files that can be used both as a keystore and as a truststore, but this may not be the case for container files that are created using other tools. Usually, PKCS#12 files only contain secret and private entries. To confirm that a PKCS#12 container includes trusted certificate ("anchor") entries look for `2.16.840.1.113894.746875.1.1: <Unsupported tag 6>` in the `openssl pkcs12 -info` output, or `trustedCertEntry` in the `keytool -list` output.
</note>


## HTTP TLS/SSL settings

You can configure the following TLS/SSL settings.
<definitions>
  <definition term="xpack.security.http.ssl.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Used to enable or disable TLS/SSL on the HTTP networking layer, which Elasticsearch uses to communicate with other clients. The default is `false`.
  </definition>
  <definition term="xpack.security.http.ssl.supported_protocols">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. If the JVM’s SSL provider supports TLSv1.3, the default is `TLSv1.3,TLSv1.2,TLSv1.1`. Otherwise, the default is `TLSv1.2,TLSv1.1`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="xpack.security.http.ssl.client_authentication">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the server’s behavior in regard to requesting a certificate from client connections. Valid values are `required`, `optional`, and `none`. `required` forces a client to present a certificate, while `optional` requests a client certificate but the client is not required to present one. Defaults to `none`.
  </definition>
  <definition term="xpack.security.http.ssl.verification_mode">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The SSL settings in `xpack.security.http.ssl` control a *server context* for TLS, which defines the settings for the TLS connection. The use of `verification_mode` in a TLS *server* is discouraged. Defines how to verify the certificates presented by another party in the TLS connection:
    Defaults to `full`.
    **Valid values**:
    - `full`:   Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`:   Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`:   Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="xpack.security.http.ssl.cipher_suites">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
</definitions>


### HTTP TLS/SSL key and trusted certificate settings

The following settings are used to specify a private key, certificate, and the trusted certificates that should be used when communicating over an SSL/TLS connection. A private key and certificate must be configured.

### PEM encoded files

When using PEM encoded files, use the following settings:
<definitions>
  <definition term="xpack.security.http.ssl.key">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.key_passphrase">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.secure_key_passphrase` instead.
    </admonition>
    You cannot use this setting and `ssl.secure_key_passphrase` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
  </definition>
  <definition term="xpack.security.http.ssl.certificate">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
  </definition>
  <definition term="xpack.security.http.ssl.certificate_authorities">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
  </definition>
</definitions>


### Java keystore files

When using Java keystore files (JKS), which contain the private key, certificate and certificates that should be trusted, use the following settings:
<definitions>
  <definition term="xpack.security.http.ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.keystore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="xpack.security.http.ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="xpack.security.http.ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="xpack.security.http.ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.truststore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
</definitions>


### PKCS#12 files

Elasticsearch can be configured to use PKCS#12 container files (`.p12` or `.pfx` files) that contain the private key, certificate and certificates that should be trusted.
PKCS#12 files are configured in the same way as Java keystore files:
<definitions>
  <definition term="xpack.security.http.ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.keystore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="xpack.security.http.ssl.keystore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="xpack.security.http.ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="xpack.security.http.ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="xpack.security.http.ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.truststore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set this to `PKCS12` to indicate that the truststore is a PKCS#12 file.
  </definition>
  <definition term="xpack.security.http.ssl.truststore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="xpack.security.http.ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
</definitions>


## Transport TLS/SSL settings

The settings in this section relate to node-to-node transport connections.
By default, Elasticsearch uses mutual TLS (mTLS) to secure node-to-node transport connections within a cluster. With mTLS, data is encrypted in transit and both nodes must present valid certificates when connecting. Each node requires that certificates be issued by a trusted certificate authority, ensuring that only authorized nodes can connect. [Learn about configuring node-to-node mTLS](https://www.elastic.co/docs/deploy-manage/security/secure-cluster-communications#encrypt-internode-communication).
<warning>
  Transport connections between Elasticsearch nodes are security-critical and you must protect them carefully. Malicious actors who can observe or interfere with unencrypted node-to-node transport traffic can read or modify cluster data. A malicious actor who can establish a transport connection might be able to invoke system-internal APIs, including APIs that read or modify cluster data.
</warning>

<definitions>
  <definition term="xpack.security.transport.ssl.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Used to enable or disable TLS/SSL on the transport networking layer, which nodes use to communicate with each other. The default is `false`.
  </definition>
  <definition term="xpack.security.transport.ssl.supported_protocols">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. If the JVM’s SSL provider supports TLSv1.3, the default is `TLSv1.3,TLSv1.2,TLSv1.1`. Otherwise, the default is `TLSv1.2,TLSv1.1`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="xpack.security.transport.ssl.client_authentication">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the node's behavior in regard to requesting a certificate when accepting an inbound transport connections from another Elasticsearch node. Valid values are `required`, `optional`, and `none`. The default is `required` which means that the connecting node must present a valid client certificate during the connection process. May also be set to `optional` which means that a client certificate is requested but the connecting node may choose not to present one, or `none` which means that no client certificate is even requested during the connection process.
    <warning>
      Turning off mTLS by setting `xpack.security.transport.ssl.client_authentication` to `optional` or `none` allows anyone with network access to establish transport connections. Malicious actors can use these connections to invoke system-internal APIs that might read or modify cluster data. Use mTLS to protect your node-to-node transport connections unless you are absolutely certain that unauthorized network access to these nodes cannot occur.
    </warning>
  </definition>
  <definition term="xpack.security.transport.ssl.verification_mode">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Defines how to verify the certificates presented by another party in the TLS connection:
    Defaults to `full`.
    **Valid values**:
    - `full`:   Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`:   Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`:   Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="xpack.security.transport.ssl.cipher_suites">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
  <definition term="xpack.security.transport.ssl.trust_restrictions.x509_fields logo cloud">
    Specifies which field(s) from the TLS certificate is used to match for the restricted trust management that is used for remote clusters connections. This should only be set when a self managed cluster can not create certificates that follow the Elastic Cloud pattern. The default value is ["subjectAltName.otherName.commonName"], the Elastic Cloud pattern. "subjectAltName.dnsName" is also supported and can be configured in addition to or in replacement of the default.
  </definition>
  <definition term="xpack.security.transport.ssl.handshake_timeout Elastic Stack: Generally available since 9.2">
    Specifies the timeout for a TLS handshake when opening a transport connection. Defaults to `10s`.
  </definition>
</definitions>


### Transport TLS/SSL key and trusted certificate settings

The following settings are used to specify a private key, certificate, and the trusted certificates that should be used when communicating over an SSL/TLS connection. A private key and certificate must be configured.

### PEM encoded files

When using PEM encoded files, use the following settings:
<definitions>
  <definition term="xpack.security.transport.ssl.key">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.key_passphrase">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.secure_key_passphrase` instead.
    </admonition>
    You cannot use this setting and `ssl.secure_key_passphrase` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
  </definition>
  <definition term="xpack.security.transport.ssl.certificate">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
  </definition>
  <definition term="xpack.security.transport.ssl.certificate_authorities">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
  </definition>
  <definition term="xpack.security.loginAssistanceMessage logo cloud">
    Adds a message to the login screen. Useful for displaying corporate messages.
  </definition>
</definitions>


### Java keystore files

When using Java keystore files (JKS), which contain the private key, certificate and certificates that should be trusted, use the following settings:
<definitions>
  <definition term="xpack.security.transport.ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="xpack.security.transport.ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.truststore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
</definitions>


### PKCS#12 files

Elasticsearch can be configured to use PKCS#12 container files (`.p12` or `.pfx` files) that contain the private key, certificate and certificates that should be trusted.
PKCS#12 files are configured in the same way as Java keystore files:
<definitions>
  <definition term="xpack.security.transport.ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the keystore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_password` instead.
    </admonition>
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.key_password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the key in the keystore. The default is the keystore password.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.keystore.secure_key_password` instead.
    </admonition>
    You cannot use this setting and `ssl.keystore.secure_password` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="xpack.security.transport.ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.truststore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set this to `PKCS12` to indicate that the truststore is a PKCS#12 file.
  </definition>
  <definition term="xpack.security.transport.ssl.truststore.password">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The password for the truststore.
    <admonition title="Deprecated in 7.17.0">
      Prefer `ssl.truststore.secure_password` instead.
    </admonition>
    You cannot use this setting and `ssl.truststore.secure_password` at the same time.
  </definition>
  <definition term="xpack.security.transport.ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
</definitions>


## Remote cluster server (API key based model) TLS/SSL settings

You can configure the following TLS/SSL settings.
<definitions>
  <definition term="xpack.security.remote_cluster_server.ssl.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Used to enable or disable TLS/SSL on the remote cluster server networking layer, which Elasticsearch uses to communicate with remote cluster clients. The default is `true`.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.supported_protocols">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. If the JVM’s SSL provider supports TLSv1.3, the default is `TLSv1.3,TLSv1.2,TLSv1.1`. Otherwise, the default is `TLSv1.2,TLSv1.1`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.client_authentication">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Controls the server’s behavior in regard to requesting a certificate from client connections. Valid values are `required`, `optional`, and `none`. `required` forces a client to present a certificate, while `optional` requests a client certificate but the client is not required to present one. Defaults to `none`.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.verification_mode">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The SSL settings in `xpack.security.remote_cluster_server.ssl` control a *server context* for TLS, which defines the settings for the TLS connection. The use of `verification_mode` in a TLS *server* is discouraged. Defines how to verify the certificates presented by another party in the TLS connection:
    Defaults to `full`.
    **Valid values**:
    - `full`:   Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`:   Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`:   Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.cipher_suites">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.handshake_timeout Elastic Stack: Generally available since 9.2">
    Specifies the timeout for a TLS handshake when handling an inbound remote-cluster connection. Defaults to `10s`.
  </definition>
</definitions>


### Remote cluster server (API key based model) TLS/SSL key and trusted certificate settings

The following settings are used to specify a private key, certificate, and the trusted certificates that should be used when communicating over an SSL/TLS connection. A private key and certificate must be configured.

### PEM encoded files

When using PEM encoded files, use the following settings:
<definitions>
  <definition term="xpack.security.remote_cluster_server.ssl.key">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.certificate">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.certificate_authorities">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
  </definition>
</definitions>


### Java keystore files

When using Java keystore files (JKS), which contain the private key, certificate and certificates that should be trusted, use the following settings:
<definitions>
  <definition term="xpack.security.remote_cluster_server.ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
</definitions>


### PKCS#12 files

Elasticsearch can be configured to use PKCS#12 container files (`.p12` or `.pfx` files) that contain the private key, certificate and certificates that should be trusted.
PKCS#12 files are configured in the same way as Java keystore files:
<definitions>
  <definition term="xpack.security.remote_cluster_server.ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.keystore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.truststore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set this to `PKCS12` to indicate that the truststore is a PKCS#12 file.
  </definition>
  <definition term="xpack.security.remote_cluster_server.ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
</definitions>


## Remote cluster client (API key based model) TLS/SSL settings

You can configure the following TLS/SSL settings.
<definitions>
  <definition term="xpack.security.remote_cluster_client.ssl.enabled">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Used to enable or disable TLS/SSL on the remote cluster client networking layer, which Elasticsearch uses to communicate with remote cluster servers. The default is `true`.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.supported_protocols">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported protocols with versions. Valid protocols: `SSLv2Hello`, `SSLv3`, `TLSv1`, `TLSv1.1`, `TLSv1.2`, `TLSv1.3`. If the JVM’s SSL provider supports TLSv1.3, the default is `TLSv1.3,TLSv1.2,TLSv1.1`. Otherwise, the default is `TLSv1.2,TLSv1.1`.
    Elasticsearch relies on your JDK’s implementation of SSL and TLS. View [Supported SSL/TLS versions by JDK version](https://www.elastic.co/docs/deploy-manage/security/supported-ssltls-versions-by-jdk-version) for more information.
    <note>
      If `xpack.security.fips_mode.enabled` is `true`, you cannot use `SSLv2Hello` or `SSLv3`. See [FIPS 140-2](https://www.elastic.co/docs/deploy-manage/security/fips).
    </note>
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.verification_mode">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Defines how to verify the certificates presented by another party in the TLS connection:
    Defaults to `full`.
    **Valid values**:
    - `full`:   Validates that the provided certificate: has an issue date that’s within the `not_before` and `not_after` dates; chains to a trusted Certificate Authority (CA); has a `hostname` or IP address that matches the names within the certificate.
    - `certificate`:   Validates the provided certificate and verifies that it’s signed by a trusted authority (CA), but doesn’t check the certificate `hostname`.
    - `none`:   Performs no certificate validation.
    <important>
      Setting certificate validation to `none` disables many security benefits of SSL/TLS, which is very dangerous. Only set this value if instructed by Elastic Support as a temporary diagnostic mechanism when attempting to resolve TLS errors.
    </important>
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.cipher_suites">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Supported cipher suites vary depending on which version of Java you use. For example, for version 12 the default value is `TLS_AES_256_GCM_SHA384`, `TLS_AES_128_GCM_SHA256`, `TLS_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256`, `TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256`, `TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA`, `TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA`, `TLS_RSA_WITH_AES_256_GCM_SHA384`, `TLS_RSA_WITH_AES_128_GCM_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA256`, `TLS_RSA_WITH_AES_128_CBC_SHA256`, `TLS_RSA_WITH_AES_256_CBC_SHA`, `TLS_RSA_WITH_AES_128_CBC_SHA`.
    For more information, see Oracle’s [Java Cryptography Architecture documentation](https://docs.oracle.com/en/java/javase/11/security/java-cryptography-architecture-jca-reference-guide.html).
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.handshake_timeout Elastic Stack: Generally available since 9.2">
    Specifies the timeout for a TLS handshake when opening a remote-cluster connection. Defaults to `10s`.
  </definition>
</definitions>


### Remote cluster client (API key based model) TLS/SSL key and trusted certificate settings

The following settings are used to specify a private key, certificate, and the trusted certificates that should be used when communicating over an SSL/TLS connection. A private key and certificate are optional and would be used if the server requires client authentication for PKI authentication.

### PEM encoded files

When using PEM encoded files, use the following settings:
<definitions>
  <definition term="xpack.security.remote_cluster_client.ssl.key">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Path to a PEM encoded file containing the private key.
    If HTTP client authentication is required, it uses this file. You cannot use this setting and `ssl.keystore.path` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.secure_key_passphrase">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The passphrase that is used to decrypt the private key. Since the key might not be encrypted, this value is optional.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.certificate">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Specifies the path for the PEM encoded certificate (or certificate chain) that is associated with the key.
    This setting can be used only if `ssl.key` is set.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.certificate_authorities">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) List of paths to PEM encoded certificate files that should be trusted.
    This setting and `ssl.truststore.path` cannot be used at the same time.
  </definition>
</definitions>


### Java keystore files

When using Java keystore files (JKS), which contain the private key, certificate and certificates that should be trusted, use the following settings:
<definitions>
  <definition term="xpack.security.remote_cluster_client.ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
</definitions>


### PKCS#12 files

Elasticsearch can be configured to use PKCS#12 container files (`.p12` or `.pfx` files) that contain the private key, certificate and certificates that should be trusted.
PKCS#12 files are configured in the same way as Java keystore files:
<definitions>
  <definition term="xpack.security.remote_cluster_client.ssl.keystore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore file that contains a private key and certificate.
    It must be either a Java keystore (jks) or a PKCS#12 file. You cannot use this setting and `ssl.key` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.keystore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The format of the keystore file. It must be either `jks` or `PKCS12`. If the keystore path ends in ".p12", ".pfx", or ".pkcs12", this setting defaults to `PKCS12`. Otherwise, it defaults to `jks`.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.keystore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the keystore.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.keystore.secure_key_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) The password for the key in the keystore. The default is the keystore password.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.truststore.path">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) The path for the keystore that contains the certificates to trust. It must be either a Java keystore (jks) or a PKCS#12 file.
    You cannot use this setting and `ssl.certificate_authorities` at the same time.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.truststore.type">
    ([Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting)) Set this to `PKCS12` to indicate that the truststore is a PKCS#12 file.
  </definition>
  <definition term="xpack.security.remote_cluster_client.ssl.truststore.secure_password">
    ([Secure](https://www.elastic.co/docs/deploy-manage/security/secure-settings)) Password for the truststore.
  </definition>
</definitions>


#### Transport profile TLS/SSL settings

The same settings that are available for the [default transport](#transport-tls-ssl-settings) are also available for each transport profile. By default, the settings for a transport profile will be the same as the default transport unless they are specified.
As an example, lets look at the key setting. For the default transport this is `xpack.security.transport.ssl.key`. In order to use this setting in a transport profile, use the prefix `transport.profiles.$PROFILE.xpack.security.` and append the portion of the setting after `xpack.security.transport.`. For the key setting, this would be `transport.profiles.$PROFILE.xpack.security.ssl.key`.

## IP filtering settings

You can configure the following settings for [IP filtering](https://www.elastic.co/docs/deploy-manage/security/ip-filtering).
<definitions>
  <definition term="xpack.security.transport.filter.allow">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) List of IP addresses to allow.
  </definition>
  <definition term="xpack.security.transport.filter.deny">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) List of IP addresses to deny.
  </definition>
  <definition term="xpack.security.http.filter.allow">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) List of IP addresses to allow just for HTTP.
  </definition>
  <definition term="xpack.security.http.filter.deny">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) List of IP addresses to deny just for HTTP.
  </definition>
  <definition term="transport.profiles.$PROFILE.xpack.security.filter.allow">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) List of IP addresses to allow for this profile.
  </definition>
  <definition term="transport.profiles.$PROFILE.xpack.security.filter.deny">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) List of IP addresses to deny for this profile.
  </definition>
  <definition term="xpack.security.remote_cluster.filter.allow">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) List of IP addresses to allow just for the [remote cluster server configured with the API key based model](https://www.elastic.co/docs/deploy-manage/remote-clusters/remote-clusters-api-key).
  </definition>
  <definition term="xpack.security.remote_cluster.filter.deny">
    ([Dynamic](https://www.elastic.co/docs/deploy-manage/stack-settings#dynamic-cluster-setting)) List of IP addresses to deny just for the remote cluster server configured with the [API key based model](https://www.elastic.co/docs/deploy-manage/remote-clusters/remote-clusters-api-key).
  </definition>
</definitions>


## User cache and password hash algorithms

Certain realms store user credentials in memory. To limit exposure to credential theft and mitigate credential compromise, the cache only stores a hashed version of the user credentials in memory. By default, the user cache is hashed with a salted `sha-256` hash algorithm. You can use a different hashing algorithm by setting the [Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting) `cache.hash_algo` realm settings to any of the following values:


| Algorithm                | Description                                                                                                                                                                                                                                                                                          |
|--------------------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `ssha256`                | Uses a salted `sha-256` algorithm (default).                                                                                                                                                                                                                                                         |
| `md5`                    | Uses `MD5` algorithm.                                                                                                                                                                                                                                                                                |
| `sha1`                   | Uses `SHA1` algorithm.                                                                                                                                                                                                                                                                               |
| `bcrypt`                 | Uses `bcrypt` algorithm with salt generated in 1024 rounds.                                                                                                                                                                                                                                          |
| `bcrypt4`                | Uses `bcrypt` algorithm with salt generated in 16 rounds.                                                                                                                                                                                                                                            |
| `bcrypt5`                | Uses `bcrypt` algorithm with salt generated in 32 rounds.                                                                                                                                                                                                                                            |
| `bcrypt6`                | Uses `bcrypt` algorithm with salt generated in 64 rounds.                                                                                                                                                                                                                                            |
| `bcrypt7`                | Uses `bcrypt` algorithm with salt generated in 128 rounds.                                                                                                                                                                                                                                           |
| `bcrypt8`                | Uses `bcrypt` algorithm with salt generated in 256 rounds.                                                                                                                                                                                                                                           |
| `bcrypt9`                | Uses `bcrypt` algorithm with salt generated in 512 rounds.                                                                                                                                                                                                                                           |
| `pbkdf2`                 | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations.                                                                                                                                                              |
| `pbkdf2_1000`            | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000 iterations.                                                                                                                                                               |
| `pbkdf2_10000`           | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations.                                                                                                                                                              |
| `pbkdf2_50000`           | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 50000 iterations.                                                                                                                                                              |
| `pbkdf2_100000`          | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 100000 iterations.                                                                                                                                                             |
| `pbkdf2_500000`          | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                              pseudorandom function using 500000 iterations.                                                                                                                                                            |
| `pbkdf2_1000000`         | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000000 iterations.                                                                                                                                                            |
| `pbkdf2_stretch`         | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations, after hashing the                             initial input with SHA512 first.                                                                               |
| `pbkdf2_stretch_1000`    | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000 iterations, after hashing the                             initial input with SHA512 first.                                                                                |
| `pbkdf2_stretch_10000`   | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations, after hashing the                             initial input with SHA512 first.                                                                               |
| `pbkdf2_stretch_50000`   | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 50000 iterations, after hashing the                             initial input with SHA512 first.                                                                               |
| `pbkdf2_stretch_100000`  | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 100000 iterations, after hashing the                             initial input with SHA512 first.                                                                              |
| `pbkdf2_stretch_500000`  | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 500000 iterations, after hashing the                             initial input with SHA512 first.                                                                              |
| `pbkdf2_stretch_1000000` | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000000 iterations, after hashing the                             initial input with SHA512 first.                                                                             |
| `noop`,`clear_text`      | Doesn’t hash the credentials and keeps it in clear text in                            memory. CAUTION: keeping clear text is considered insecure                            and can be compromised at the OS level (for example through                            memory dumps and using `ptrace`). |

Likewise, realms that store passwords hash them using cryptographically strong and password-specific salt values. You can configure the algorithm for password hashing by setting the [Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting) `xpack.security.authc.password_hashing.algorithm` setting to one of the following:


| Algorithm                | Description                                                                                                                                                                                                              |
|--------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `bcrypt`                 | Uses `bcrypt` algorithm with salt generated in 1024 rounds. (default)                                                                                                                                                    |
| `bcrypt4`                | Uses `bcrypt` algorithm with salt generated in 16 rounds.                                                                                                                                                                |
| `bcrypt5`                | Uses `bcrypt` algorithm with salt generated in 32 rounds.                                                                                                                                                                |
| `bcrypt6`                | Uses `bcrypt` algorithm with salt generated in 64 rounds.                                                                                                                                                                |
| `bcrypt7`                | Uses `bcrypt` algorithm with salt generated in 128 rounds.                                                                                                                                                               |
| `bcrypt8`                | Uses `bcrypt` algorithm with salt generated in 256 rounds.                                                                                                                                                               |
| `bcrypt9`                | Uses `bcrypt` algorithm with salt generated in 512 rounds.                                                                                                                                                               |
| `bcrypt10`               | Uses `bcrypt` algorithm with salt generated in 1024 rounds.                                                                                                                                                              |
| `bcrypt11`               | Uses `bcrypt` algorithm with salt generated in 2048 rounds.                                                                                                                                                              |
| `bcrypt12`               | Uses `bcrypt` algorithm with salt generated in 4096 rounds.                                                                                                                                                              |
| `bcrypt13`               | Uses `bcrypt` algorithm with salt generated in 8192 rounds.                                                                                                                                                              |
| `bcrypt14`               | Uses `bcrypt` algorithm with salt generated in 16384 rounds.                                                                                                                                                             |
| `pbkdf2`                 | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations.                                                                                  |
| `pbkdf2_1000`            | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000 iterations.                                                                                   |
| `pbkdf2_10000`           | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations.                                                                                  |
| `pbkdf2_50000`           | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 50000 iterations.                                                                                  |
| `pbkdf2_100000`          | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 100000 iterations.                                                                                 |
| `pbkdf2_500000`          | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                              pseudorandom function using 500000 iterations.                                                                                |
| `pbkdf2_1000000`         | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000000 iterations.                                                                                |
| `pbkdf2_stretch`         | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations, after hashing the                             initial input with SHA512 first.   |
| `pbkdf2_stretch_1000`    | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000 iterations, after hashing the                             initial input with SHA512 first.    |
| `pbkdf2_stretch_10000`   | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations, after hashing the                             initial input with SHA512 first.   |
| `pbkdf2_stretch_50000`   | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 50000 iterations, after hashing the                             initial input with SHA512 first.   |
| `pbkdf2_stretch_100000`  | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 100000 iterations, after hashing the                             initial input with SHA512 first.  |
| `pbkdf2_stretch_500000`  | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 500000 iterations, after hashing the                             initial input with SHA512 first.  |
| `pbkdf2_stretch_1000000` | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000000 iterations, after hashing the                             initial input with SHA512 first. |

Furthermore, Elasticsearch supports authentication via securely-generated high entropy tokens, for instance [API keys](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-security-create-api-key). Analogous to passwords, only the tokens' hashes are stored. Since the tokens are guaranteed to have sufficiently high entropy to resist offline attacks, secure salted hash functions are supported in addition to the password-hashing algorithms mentioned above.
You can configure the algorithm for API key stored credential hashing by setting the [Static](https://www.elastic.co/docs/deploy-manage/stack-settings#static-cluster-setting) `xpack.security.authc.api_key.hashing.algorithm` setting to one of the following


| Algorithm                | Description                                                                                                                                                                                                              |
|--------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `ssha256`                | Uses a salted `sha-256` algorithm. (default)                                                                                                                                                                             |
| `bcrypt`                 | Uses `bcrypt` algorithm with salt generated in 1024 rounds.                                                                                                                                                              |
| `bcrypt4`                | Uses `bcrypt` algorithm with salt generated in 16 rounds.                                                                                                                                                                |
| `bcrypt5`                | Uses `bcrypt` algorithm with salt generated in 32 rounds.                                                                                                                                                                |
| `bcrypt6`                | Uses `bcrypt` algorithm with salt generated in 64 rounds.                                                                                                                                                                |
| `bcrypt7`                | Uses `bcrypt` algorithm with salt generated in 128 rounds.                                                                                                                                                               |
| `bcrypt8`                | Uses `bcrypt` algorithm with salt generated in 256 rounds.                                                                                                                                                               |
| `bcrypt9`                | Uses `bcrypt` algorithm with salt generated in 512 rounds.                                                                                                                                                               |
| `bcrypt10`               | Uses `bcrypt` algorithm with salt generated in 1024 rounds.                                                                                                                                                              |
| `bcrypt11`               | Uses `bcrypt` algorithm with salt generated in 2048 rounds.                                                                                                                                                              |
| `bcrypt12`               | Uses `bcrypt` algorithm with salt generated in 4096 rounds.                                                                                                                                                              |
| `bcrypt13`               | Uses `bcrypt` algorithm with salt generated in 8192 rounds.                                                                                                                                                              |
| `bcrypt14`               | Uses `bcrypt` algorithm with salt generated in 16384 rounds.                                                                                                                                                             |
| `pbkdf2`                 | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations.                                                                                  |
| `pbkdf2_1000`            | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000 iterations.                                                                                   |
| `pbkdf2_10000`           | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations.                                                                                  |
| `pbkdf2_50000`           | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 50000 iterations.                                                                                  |
| `pbkdf2_100000`          | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 100000 iterations.                                                                                 |
| `pbkdf2_500000`          | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                              pseudorandom function using 500000 iterations.                                                                                |
| `pbkdf2_1000000`         | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000000 iterations.                                                                                |
| `pbkdf2_stretch`         | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations, after hashing the                             initial input with SHA512 first.   |
| `pbkdf2_stretch_1000`    | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000 iterations, after hashing the                             initial input with SHA512 first.    |
| `pbkdf2_stretch_10000`   | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 10000 iterations, after hashing the                             initial input with SHA512 first.   |
| `pbkdf2_stretch_50000`   | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 50000 iterations, after hashing the                             initial input with SHA512 first.   |
| `pbkdf2_stretch_100000`  | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 100000 iterations, after hashing the                             initial input with SHA512 first.  |
| `pbkdf2_stretch_500000`  | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 500000 iterations, after hashing the                             initial input with SHA512 first.  |
| `pbkdf2_stretch_1000000` | Uses `PBKDF2` key derivation function with `HMAC-SHA512` as a                             pseudorandom function using 1000000 iterations, after hashing the                             initial input with SHA512 first. |