﻿---
title: Elasticsearch Python client
description: This documentation covers the official Python client for Elasticsearch. The Python client provides a comprehensive foundation for working with Elasticsearch...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/python
products:
  - Elasticsearch
  - Elasticsearch Client
  - Elasticsearch Python Client
---

# Elasticsearch Python client
This documentation covers the [official Python client for Elasticsearch](https://github.com/elastic/elasticsearch-py). The Python client provides a comprehensive foundation for working with Elasticsearch in Python. The client is designed to be unopinionated and extensible.

## Example

Here's an example of basic Python client usage:
<tab-set>
  <tab-item title="Standard Python">
    ```python
    import os
    from elasticsearch import Elasticsearch

    def main():
        client = Elasticsearch(
            hosts=[os.getenv("ELASTICSEARCH_URL")],
            api_key=os.getenv("ELASTIC_API_KEY"),
        )

        resp = client.search(
            index="my-index-000001",
            from_=40,
            size=20,
            query={
                "term": {
                    "user.id": "kimchy"
                }
            },
        )
        print(resp)
    ```
  </tab-item>

  <tab-item title="Async Python">
    ```python
    import os
    from elasticsearch import AsyncElasticsearch

    async def main():
        client = AsyncElasticsearch(
            hosts=[os.getenv("ELASTICSEARCH_URL")],
            api_key=os.getenv("ELASTIC_API_KEY"),
        )

        resp = await client.search(
            index="my-index-000001",
            from_=40,
            size=20,
            query={
                "term": {
                    "user.id": "kimchy"
                }
            },
        )
        print(resp)
    ```
  </tab-item>
</tab-set>


## Overview

The Elasticsearch Python client package consists of several modules: the core client, a set of bulk helper functions, an ESQL query builder, and a DSL module.

### The core client

This module, also known as the low-level client, enables sending requests to Elasticsearch servers. The client provides access to the entire surface of the Elasticsearch API.
- [Getting started](https://www.elastic.co/docs/reference/elasticsearch/clients/python/getting-started)
- [Walkthrough: Ingest data with Python](https://www.elastic.co/docs/manage-data/ingest/ingesting-data-from-applications/ingest-data-with-python-on-elasticsearch-service)
- [Reference documentation](https://elasticsearch-py.readthedocs.io/en/stable/es_api.html)


#### Features

The core client's features include:
- Translating basic Python data types to and from JSON
- Configurable automatic discovery of cluster nodes
- Persistent connections
- Load balancing (with pluggable selection strategy) across all available nodes
- Node timeouts on transient errors
- Thread safety
- Pluggable architecture


### Bulk helpers

The bulk helpers simplify ingesting large amounts of data, by providing a high-level interface based on Python iterables.
- [Client helpers > Bulk helpers](/docs/reference/elasticsearch/clients/python/client-helpers#bulk-helpers)
- [Reference documentation](https://elasticsearch-py.readthedocs.io/en/stable/api_helpers.html)


### ES|QL query builder

The ESQL query builder offers an idiomatic interface for constructing ESQL queries using Python expressions.
- [ES|QL query builder](https://www.elastic.co/docs/reference/elasticsearch/clients/python/esql-query-builder)
- [Reference documentation](https://elasticsearch-py.readthedocs.io/en/stable/esql.html)


### DSL module

The DSL module can be thought of as a high-level client for Elasticsearch. It allows applications to manipulate documents and queries using Python classes and objects, instead of primitive types such as dictionaries and lists.
- [Elasticsearch Python DSL](https://www.elastic.co/docs/reference/elasticsearch/clients/python/elasticsearch-dsl)
- [Reference documentation](https://elasticsearch-py.readthedocs.io/en/stable/dsl.html)


## Compatibility


| Client version | Elasticsearch `8.x`                   | Elasticsearch `9.x`               | Elasticsearch `10.x`                   |
|----------------|---------------------------------------|-----------------------------------|----------------------------------------|
| 9.x client     | Not compatible with Elasticsearch 8.x | Compatible with Elasticsearch 9.x | Compatible with Elasticsearch 10.x     |
| 8.x client     | Compatible with Elasticsearch 8.x     | Compatible with Elasticsearch 9.x | Not compatible with Elasticsearch 10.x |

Compatibility does not imply feature parity. New Elasticsearch features are supported only in equivalent client versions. For example, an 8.12 client fully supports Elasticsearch 8.12 features and works with 8.13 without breaking, but it does not support new Elasticsearch 8.13 features. An 8.13 client fully supports Elasticsearch 8.13 features.
Elasticsearch language clients are also **backward compatible** across minor versions, with default distributions and without guarantees.

### Major version upgrades

<important>
  To upgrade to a new major version, first [upgrade Elasticsearch](https://www.elastic.co/docs/deploy-manage/upgrade), then upgrade the Python client.
</important>

As of version 8.0, Elasticsearch offers a [compatibility mode](https://www.elastic.co/docs/reference/elasticsearch/rest-apis/compatibility) for smoother upgrades. In compatibility mode, you can upgrade your Elasticsearch cluster to the next major version while continuing to use your existing client during the transition.
For example, if you're upgrading Elasticsearch from 8.x to 9.x, you can continue to use the 8.x Python client during and after the Elasticsearch server upgrade, with the exception of [breaking changes](https://www.elastic.co/docs/release-notes/elasticsearch/clients/python/breaking-changes).
In the Python client, compatibility mode is always enabled.
<tip>
  To support working with multiple client versions, the Python client is also released under the package names `elasticsearch8` and `elasticsearch9` (to prevent name collisions).
</tip>