﻿---
title: Dealing with JSON arrays and objects in PHP
description: A common source of confusion with the client revolves around JSON arrays and objects, and how to specify them in PHP. In particular, problems are caused...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/php/php_json_objects
products:
  - Elasticsearch
  - Elasticsearch Client
  - Elasticsearch PHP Client
---

# Dealing with JSON arrays and objects in PHP
A common source of confusion with the client revolves around JSON arrays and objects, and how to specify them in PHP. In particular, problems are caused by empty objects and arrays of objects. This page shows you some common patterns used in Elasticsearch JSON API and how to convert that to a PHP representation.

## Empty Objects

The Elasticsearch API uses empty JSON objects in several locations which can cause problems for PHP. Unlike other languages, PHP does not have a "short" notation for empty objects and many developers are unaware how to specify an empty object.
Consider adding a highlight to a query:
```json
{
    "query" : {
        "match" : {
            "content" : "quick brown fox"
        }
    },
    "highlight" : {
        "fields" : {
            "content" : {} 
        }
    }
}
```

The problem is that PHP will automatically convert `"content" : {}` into `"content" : []`, which is no longer valid Elasticsearch DSL. We need to tell PHP that the empty object is explicitly an object, not an array. To define this query in PHP, you would do:
```json
$params['body'] = array(
    'query' => array(
        'match' => array(
            'content' => 'quick brown fox'
        )
    ),
    'highlight' => array(
        'fields' => array(
            'content' => new \stdClass() 
        )
    )
);
$results = $client->search($params);
```

By using an explicit stdClass object, we can force the `json_encode` parser to correctly output an empty object, instead of an empty array. This verbose solution is the only way to acomplish the goal in PHP… there is no "short" version of an empty object.

## Arrays of Objects

Another common pattern in Elasticsearch DSL is an array of objects. For example, consider adding a sort to your query:
```json
{
    "query" : {
        "match" : { "content" : "quick brown fox" }
    },
    "sort" : [  
        {"time" : {"order" : "desc"}},
        {"popularity" : {"order" : "desc"}}
    ]
}
```

This arrangement is very common, but the construction in PHP can be tricky since it requires nesting arrays. The verbosity of PHP tends to obscure what is actually going on. To construct an array of objects, you actually need an array of arrays:
```json
$params['body'] = array(
    'query' => array(
        'match' => array(
            'content' => 'quick brown fox'
        )
    ),
    'sort' => array(    
        array('time' => array('order' => 'desc')),  
        array('popularity' => array('order' => 'desc')) 
    )
);
$results = $client->search($params);
```

If you are on PHP 5.4+, we strongly encourage you to use the short array syntax. It makes these nested arrays much simpler to read:
```json
$params['body'] = [
    'query' => [
        'match' => [
            'content' => 'quick brown fox'
        ]
    ],
    'sort' => [
        ['time' => ['order' => 'desc']],
        ['popularity' => ['order' => 'desc']]
    ]
];
$results = $client->search($params);
```


## Arrays of empty objects

Occasionally, you’ll encounter DSL that requires both of the previous patterns. The function score query is a good example, it sometimes requires an array of objects, and some of those objects might be empty JSON objects.
Given this query:
```json
{
   "query":{
      "function_score":{
         "functions":[
            {
               "random_score":{}
            }
         ],
         "boost_mode":"replace"
      }
   }
}
```

We can build it using the following PHP code:
```json
$params['body'] = array(
    'query' => array(
        'function_score' => array(
            'functions' => array(  
                array(  
                    'random_score' => new \stdClass() 
                )
            )
        )
    )
);
$results = $client->search($params);
```